<?php
/**
 * Main Plugin class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Converzo\KiyohReviews\Admin\Settings;
use Converzo\KiyohReviews\Frontend\Reviews;
use Converzo\KiyohReviews\Service\CompanyReview;
use Converzo\KiyohReviews\Service\Invite;
use Converzo\KiyohReviews\Service\ProductSync;
use Converzo\KiyohReviews\Widget\CompanyWidget;

/**
 * Main plugin orchestrator class.
 */
class Plugin {

	/**
	 * Whether the plugin has been initiated.
	 *
	 * @var bool
	 */
	private static bool $initiated = false;

	/**
	 * Initialize the plugin.
	 *
	 * @return void
	 */
	public static function init(): void {
		if ( self::$initiated ) {
			return;
		}

		// Check if WooCommerce is active.
		if ( ! self::is_woocommerce_active() ) {
			add_action( 'admin_notices', array( static::class, 'woocommerce_missing_notice' ) );
			return;
		}

		self::$initiated = true;

		// Declare HPOS compatibility.
		add_action( 'before_woocommerce_init', array( static::class, 'declare_hpos_compatibility' ) );

		// Initialize components.
		Settings::init();
		Reviews::init();
		Invite::init();
		ProductSync::init();
		CompanyReview::init();

		// Initialize premium features if available.
		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Freemius SDK.
		if ( function_exists( 'converzo_kiyoh_fs' ) && converzo_kiyoh_fs()->can_use_premium_code__premium_only() ) {
			Premium__premium_only\Loader::init();
		}
	}

	/**
	 * Initialize widgets.
	 *
	 * @return void
	 */
	public static function init_widgets(): void {
		add_action( 'widgets_init', array( static::class, 'register_widgets' ) );
	}

	/**
	 * Register widgets.
	 *
	 * @return void
	 */
	public static function register_widgets(): void {
		register_widget( CompanyWidget::class );
	}

	/**
	 * Plugin activation handler.
	 *
	 * @return void
	 */
	public static function activate(): void {
		// Fetch initial company review data.
		CompanyReview::fetch_and_cache();

		// Schedule cron jobs.
		if ( ! wp_next_scheduled( 'converzo_kiyoh_fetch_company_reviews' ) ) {
			wp_schedule_event( time(), 'hourly', 'converzo_kiyoh_fetch_company_reviews' );
		}
		if ( ! wp_next_scheduled( 'converzo_kiyoh_sync_product_catalog' ) ) {
			wp_schedule_event( time(), 'hourly', 'converzo_kiyoh_sync_product_catalog' );
		}
	}

	/**
	 * Plugin deactivation handler.
	 *
	 * @return void
	 */
	public static function deactivate(): void {
		wp_clear_scheduled_hook( 'converzo_kiyoh_fetch_company_reviews' );
		wp_clear_scheduled_hook( 'converzo_kiyoh_sync_product_catalog' );
	}

	/**
	 * Declare compatibility with WooCommerce HPOS.
	 *
	 * @return void
	 */
	public static function declare_hpos_compatibility(): void {
		if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', CONVERZO_KIYOH_PLUGIN_FILE, true );
		}
	}

	/**
	 * Check if WooCommerce is active.
	 *
	 * @return bool
	 */
	private static function is_woocommerce_active(): bool {
		return class_exists( 'WooCommerce' );
	}

	/**
	 * Display admin notice when WooCommerce is not active.
	 *
	 * @return void
	 */
	public static function woocommerce_missing_notice(): void {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				printf(
					/* translators: %s: WooCommerce plugin name */
					esc_html__( 'Kiyoh Reviews requires %s to be installed and active.', 'kiyoh-reviews' ),
					'<strong>' . esc_html__( 'WooCommerce', 'kiyoh-reviews' ) . '</strong>'
				);
				?>
			</p>
		</div>
		<?php
	}
}
