<?php
/**
 * Logger class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews;

/**
 * Handles debug logging with rotation and security.
 */
class Logger {

	/**
	 * Log level constants.
	 */
	const DEBUG   = 'debug';
	const INFO    = 'info';
	const WARNING = 'warning';
	const ERROR   = 'error';

	/**
	 * Maximum log file size in bytes (5 MB).
	 *
	 * @var int
	 */
	private const MAX_SIZE = 5 * 1024 * 1024;

	/**
	 * Maximum number of archive files to keep.
	 *
	 * @var int
	 */
	private const MAX_ARCHIVES = 3;

	/**
	 * Get the log directory path.
	 *
	 * @return string The log directory path.
	 */
	public static function get_log_dir(): string {
		$wp_upload_dir = wp_upload_dir( null, false );
		return $wp_upload_dir['basedir'] . '/kiyoh-logs';
	}

	/**
	 * Get the current log file path.
	 *
	 * @return string The log file path.
	 */
	public static function get_log_file_path(): string {
		return self::get_log_dir() . '/kiyoh-debug.log';
	}

	/**
	 * Initialize the log directory with protection.
	 *
	 * @return bool True if directory is ready, false on failure.
	 */
	private static function init_log_dir(): bool {
		$log_dir = self::get_log_dir();

		if ( ! file_exists( $log_dir ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir -- Simple directory creation.
			if ( ! mkdir( $log_dir, 0750, true ) ) {
				return false;
			}
		}

		// Add .htaccess protection.
		$htaccess_file = $log_dir . '/.htaccess';
		if ( ! file_exists( $htaccess_file ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Simple file creation.
			file_put_contents( $htaccess_file, "Order deny,allow\nDeny from all" );
		}

		// Add index.php protection.
		$index_file = $log_dir . '/index.php';
		if ( ! file_exists( $index_file ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Simple file creation.
			file_put_contents( $index_file, '<?php // Silence is golden.' );
		}

		return true;
	}

	/**
	 * Rotate log file if it exceeds max size.
	 *
	 * @return void
	 */
	private static function maybe_rotate_log(): void {
		$log_file = self::get_log_file_path();

		if ( ! file_exists( $log_file ) ) {
			return;
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_filesize -- Simple file size check.
		if ( filesize( $log_file ) < self::MAX_SIZE ) {
			return;
		}

		// Rotate existing archives.
		for ( $i = self::MAX_ARCHIVES - 1; $i >= 1; $i-- ) {
			$old_archive = $log_file . '.' . $i;
			$new_archive = $log_file . '.' . ( $i + 1 );
			if ( file_exists( $old_archive ) ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Simple file rename.
				rename( $old_archive, $new_archive );
			}
		}

		// Archive current log.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Simple file rename.
		rename( $log_file, $log_file . '.1' );

		// Remove oldest archive if exceeds max.
		$oldest = $log_file . '.' . ( self::MAX_ARCHIVES + 1 );
		if ( file_exists( $oldest ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Simple file delete.
			unlink( $oldest );
		}
	}

	/**
	 * Log a message to the debug log file.
	 *
	 * @param string|array|object $message The message to log.
	 * @param string              $level   Log level: debug, info, warning, error.
	 * @return void
	 */
	public static function log( $message, string $level = self::INFO ): void {
		if ( ! self::init_log_dir() ) {
			return;
		}

		self::maybe_rotate_log();

		if ( is_array( $message ) || is_object( $message ) ) {
			$message = wp_json_encode( $message, JSON_PRETTY_PRINT );
		}

		$log_file    = self::get_log_file_path();
		$level_upper = strtoupper( $level );
		$log_entry   = sprintf(
			"[%s] [%s] %s\n",
			gmdate( 'Y-m-d H:i:s' ),
			$level_upper,
			$message
		);

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Simple debug logging.
		file_put_contents( $log_file, $log_entry, FILE_APPEND | LOCK_EX );
	}

	/**
	 * Log a debug message.
	 *
	 * @param string|array|object $message The message to log.
	 * @return void
	 */
	public static function debug( $message ): void {
		self::log( $message, self::DEBUG );
	}

	/**
	 * Log an info message.
	 *
	 * @param string|array|object $message The message to log.
	 * @return void
	 */
	public static function info( $message ): void {
		self::log( $message, self::INFO );
	}

	/**
	 * Log a warning message.
	 *
	 * @param string|array|object $message The message to log.
	 * @return void
	 */
	public static function warning( $message ): void {
		self::log( $message, self::WARNING );
	}

	/**
	 * Log an error message.
	 *
	 * @param string|array|object $message The message to log.
	 * @return void
	 */
	public static function error( $message ): void {
		self::log( $message, self::ERROR );
	}

	/**
	 * Get the debug log download URL (for admin).
	 *
	 * @return string The admin download URL.
	 */
	public static function get_download_url(): string {
		return admin_url( 'admin-ajax.php?action=converzo_kiyoh_download_log&nonce=' . wp_create_nonce( 'converzo_kiyoh_download_log' ) );
	}

	/**
	 * Handle log file download via admin AJAX.
	 *
	 * @return void
	 */
	public static function handle_download(): void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access.', 'kiyoh-reviews' ) );
		}

		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'converzo_kiyoh_download_log' ) ) {
			wp_die( esc_html__( 'Invalid nonce.', 'kiyoh-reviews' ) );
		}

		$log_file = self::get_log_file_path();

		if ( ! file_exists( $log_file ) ) {
			wp_die( esc_html__( 'Log file not found.', 'kiyoh-reviews' ) );
		}

		header( 'Content-Type: text/plain' );
		header( 'Content-Disposition: attachment; filename="kiyoh-debug.log"' );
		header( 'Content-Length: ' . filesize( $log_file ) );

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile -- Direct file output for download.
		readfile( $log_file );
		exit;
	}
}
