<?php
/**
 * Frontend Reviews class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Converzo\KiyohReviews\Admin\Settings;
use Converzo\KiyohReviews\Api\Client;

/**
 * Handles frontend display of product reviews.
 */
class Reviews {

	/**
	 * Store KiyOh ratings for comment meta interception.
	 *
	 * @var array
	 */
	private static array $kiyoh_ratings = array();

	/**
	 * Initialize frontend hooks.
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'woocommerce_before_single_product', array( static::class, 'before_product_load' ) );
		add_action( 'woocommerce_before_shop_loop_item', array( static::class, 'before_product_load' ) );
		add_filter( 'comments_array', array( static::class, 'merge_kiyoh_reviews' ), 10, 2 );
		add_action( 'woocommerce_review_before_comment_meta', array( static::class, 'display_kiyoh_rating' ), 10 );
		add_action( 'wp_enqueue_scripts', array( static::class, 'enqueue_styles' ) );

		if ( 'yes' === Settings::get_option( 'disable_wc_review_form' ) ) {
			add_filter( 'woocommerce_product_review_list_args', array( static::class, 'after_review_list_disable_comments' ) );
			add_filter( 'body_class', array( static::class, 'add_hide_review_form_class' ) );
		}
	}

	/**
	 * Enqueue frontend styles.
	 *
	 * @return void
	 */
	public static function enqueue_styles(): void {
		// Check if we need to load styles.
		$should_load = false;

		// Load on product/shop pages.
		if ( is_product() || is_shop() || is_product_category() || is_product_tag() ) {
			$should_load = true;
		}

		// Load if company widget is active.
		if ( is_active_widget( false, false, 'kiyoh_company_widget' ) ) {
			$should_load = true;
		}

		if ( ! $should_load ) {
			return;
		}

		wp_enqueue_style(
			'kiyoh-stylesheet',
			CONVERZO_KIYOH_PLUGIN_URL . 'assets/css/kiyoh-stylesheet.css',
			array(),
			CONVERZO_KIYOH_VERSION
		);
	}

	/**
	 * Display KiYOH rating for KiYOH reviews.
	 *
	 * @param \WP_Comment $comment The comment object.
	 * @return void
	 */
	public static function display_kiyoh_rating( $comment ): void {
		if ( ! isset( self::$kiyoh_ratings[ $comment->comment_ID ] ) ) {
			return;
		}

		$rating = self::$kiyoh_ratings[ $comment->comment_ID ];
		if ( $rating && wc_review_ratings_enabled() ) {
			echo '<div class="review-rating">' . wp_kses_post( wc_get_rating_html( $rating ) ) . '</div>';
		}
	}

	/**
	 * Disable comments after the review list renders, so the form won't show.
	 *
	 * @param array $args The review list args.
	 * @return array
	 */
	public static function after_review_list_disable_comments( array $args ): array {
		add_filter( 'comments_open', '__return_false', 999 );
		return $args;
	}

	/**
	 * Add body class to hide review form on product pages.
	 *
	 * @param array $classes The body classes.
	 * @return array The modified body classes.
	 */
	public static function add_hide_review_form_class( array $classes ): array {
		if ( is_product() ) {
			$classes[] = 'kiyoh-hide-review-form';
		}
		return $classes;
	}

	/**
	 * Merge KiyOh reviews into the WooCommerce comments array.
	 *
	 * @param array $comments The existing comments.
	 * @param int   $post_id  The post ID.
	 * @return array The merged comments array.
	 */
	public static function merge_kiyoh_reviews( array $comments, int $post_id ): array {
		$product = wc_get_product( $post_id );

		if ( ! $product instanceof \WC_Product ) {
			return $comments;
		}

		$converzo_kiyoh_reviews = self::get_product_review_data( $product->get_id() );

		if ( empty( $converzo_kiyoh_reviews ) ) {
			return $comments;
		}

		// Convert KiyOh reviews to WP_Comment objects.
		foreach ( $converzo_kiyoh_reviews as $kiyoh_review ) {
			$comments[] = self::convert_to_comment( $kiyoh_review, $post_id );
		}

		// Sort by date descending.
		usort(
			$comments,
			function ( $a, $b ) {
				return strtotime( $b->comment_date ) - strtotime( $a->comment_date );
			}
		);

		return $comments;
	}

	/**
	 * Convert a KiyOh review to a WP_Comment object.
	 *
	 * @param array $kiyoh_review The KiyOh review data.
	 * @param int   $post_id      The post ID.
	 * @return \WP_Comment The comment object.
	 */
	private static function convert_to_comment( array $kiyoh_review, int $post_id ): \WP_Comment {
		$comment_id = 'kiyoh_' . ( $kiyoh_review['productReviewId'] ?? $kiyoh_review['id'] ?? wp_generate_uuid4() );
		$author     = $kiyoh_review['reviewAuthor'] ?? __( 'Anonymous', 'kiyoh-reviews' );

		// Extract content from productReviewContent array.
		$oneliner = '';
		$content  = '';
		if ( ! empty( $kiyoh_review['productReviewContent'] ) && is_array( $kiyoh_review['productReviewContent'] ) ) {
			foreach ( $kiyoh_review['productReviewContent'] as $review_item ) {
				$question_group = $review_item['questionGroup'] ?? '';
				$rating_value   = $review_item['rating'] ?? '';

				if ( 'DEFAULT_ONELINER' === $question_group && ! empty( $rating_value ) ) {
					$oneliner = $rating_value;
				} elseif ( 'DEFAULT_OPINION' === $question_group && ! empty( $rating_value ) ) {
					$content = $rating_value;
				}
			}
		}

		// Convert Kiyoh 10-point rating to WooCommerce 5-point scale.
		$rating = self::convert_kiyoh_rating( $kiyoh_review['rating'] ?? null );

		// Validate date string, fallback to current time if invalid.
		$date_string = $kiyoh_review['dateSince'] ?? '';
		$timestamp   = strtotime( $date_string );
		if ( false === $timestamp ) {
			$timestamp = time();
		}

		// Combine oneliner and description if both exist.
		if ( ! empty( $oneliner ) && ! empty( $content ) ) {
			$content = '<strong>' . esc_html( $oneliner ) . '</strong><br>' . esc_html( $content );
		} elseif ( ! empty( $oneliner ) ) {
			$content = $oneliner;
		} elseif ( empty( $content ) ) {
			$content = '';
		}

		$comment_data = array(
			'comment_ID'           => $comment_id,
			'comment_post_ID'      => $post_id,
			'comment_author'       => $author,
			'comment_author_email' => '',
			'comment_author_url'   => '',
			'comment_author_IP'    => '',
			'comment_date'         => gmdate( 'Y-m-d H:i:s', $timestamp ),
			'comment_date_gmt'     => gmdate( 'Y-m-d H:i:s', $timestamp ),
			'comment_content'      => $content,
			'comment_karma'        => 0,
			'comment_approved'     => '1',
			'comment_agent'        => 'KiyOh',
			'comment_type'         => 'review',
			'comment_parent'       => 0,
			'user_id'              => 0,
		);

		$comment = new \WP_Comment( (object) $comment_data );

		// Store rating for meta interception.
		self::$kiyoh_ratings[ $comment_id ] = $rating;

		return $comment;
	}

	/**
	 * Convert Kiyoh rating (0-10 scale) to WooCommerce rating (0-5 scale).
	 *
	 * @param mixed $kiyoh_rating The Kiyoh rating value.
	 * @return float The converted rating for WooCommerce.
	 */
	private static function convert_kiyoh_rating( $kiyoh_rating ): float {
		if ( ! isset( $kiyoh_rating ) || ! is_numeric( $kiyoh_rating ) ) {
			return 0.0;
		}
		return (float) $kiyoh_rating / 2;
	}

	/**
	 * Load Kiyoh reviews before product display and combine ratings.
	 *
	 * @return void
	 */
	public static function before_product_load(): void {
		global $product;

		if ( ! $product instanceof \WC_Product ) {
			return;
		}

		$converzo_kiyoh_reviews = self::get_product_review_data( $product->get_id() );

		if ( ! $converzo_kiyoh_reviews ) {
			return;
		}

		$wc_review_count = $product->get_review_count();
		$wc_rating_count = $product->get_rating_count();
		$wc_avg_rating   = $product->get_average_rating();

		// Count Kiyoh reviews.
		$kiyoh_review_count = count( $converzo_kiyoh_reviews );

		// Collect Kiyoh ratings (on 0-10 scale).
		$kiyoh_ratings = array();
		foreach ( $converzo_kiyoh_reviews as $kiyoh_review ) {
			if ( isset( $kiyoh_review['rating'] ) && is_numeric( $kiyoh_review['rating'] ) ) {
				$kiyoh_ratings[] = (float) $kiyoh_review['rating'];
			}
		}

		/*
		 * Calculate combined average rating.
		 *
		 * WooCommerce stores ratings on a 0-5 scale, Kiyoh uses 0-10 scale.
		 * To combine: convert WC average back to sum, add Kiyoh sum, then average.
		 */
		$wc_rating_sum_on_10_scale = (float) ( $wc_avg_rating * 2 ) * $wc_rating_count;
		$kiyoh_rating_sum          = array_sum( $kiyoh_ratings );
		$total_rating_count        = count( $kiyoh_ratings ) + $wc_rating_count;

		if ( $total_rating_count > 0 ) {
			// Combined average on 10-point scale, then convert to 5-point.
			$combined_sum = $kiyoh_rating_sum + $wc_rating_sum_on_10_scale;
			$combined_avg = $combined_sum / $total_rating_count;
			$final_avg    = $combined_avg / 2; // Convert to 0-5 scale.
		} else {
			$final_avg = 0;
		}

		// Update product with combined counts and rating.
		$product->set_rating_counts( $total_rating_count );
		$product->set_average_rating( $final_avg );
		$product->set_review_count( $wc_review_count + $kiyoh_review_count );
	}

	/**
	 * Get product review data with caching.
	 *
	 * @param int $product_id The product ID.
	 * @return array|null The product reviews or null.
	 */
	public static function get_product_review_data( int $product_id ): ?array {
		$product = wc_get_product( $product_id );

		if ( ! $product instanceof \WC_Product ) {
			return null;
		}

		$cache_timestamp = $product->get_meta( 'converzo_kiyoh_reviews_timestamp', true );

		if ( empty( $cache_timestamp ) || strtotime( $cache_timestamp ) <= strtotime( '-1 hours' ) ) {
			$product_review_data = self::fetch_and_cache_reviews( $product );
		} else {
			$product_review_data = $product->get_meta( 'converzo_kiyoh_reviews', true );
		}

		return is_array( $product_review_data ) ? $product_review_data : null;
	}

	/**
	 * Fetch and cache product reviews.
	 *
	 * @param \WC_Product $product The product.
	 * @return array|null The product reviews or null.
	 */
	private static function fetch_and_cache_reviews( \WC_Product $product ): ?array {
		// Get existing cached reviews to serve as fallback.
		$cached_reviews = $product->get_meta( 'converzo_kiyoh_reviews', true );

		$reviews = Client::fetch_product_reviews( $product->get_id() );

		if ( null !== $reviews ) {
			// Success - cache fresh data.
			$product->update_meta_data( 'converzo_kiyoh_reviews', $reviews );
			$product->update_meta_data( 'converzo_kiyoh_reviews_timestamp', gmdate( 'Y-m-d\TH:i:s' ) );
			$product->save();
			return $reviews;
		}

		// API failed - update timestamp to prevent repeated calls, serve stale data.
		$product->update_meta_data( 'converzo_kiyoh_reviews_timestamp', gmdate( 'Y-m-d\TH:i:s' ) );
		$product->save();

		return is_array( $cached_reviews ) ? $cached_reviews : null;
	}
}
