<?php
/**
 * API Client class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews\Api;

use Converzo\KiyohReviews\Admin\Settings;
use Converzo\KiyohReviews\Logger;

/**
 * Handles all communication with the Kiyoh API.
 */
class Client {

	/**
	 * API endpoint paths.
	 */
	const INVITE_URL          = '/v1/invite/external';
	const COMPANY_REVIEW_URL  = '/v1/publication/review/external/location/statistics';
	const PRODUCT_REVIEWS_URL = '/v1/publication/product/review/external';
	const PRODUCT_UPDATE_URL  = '/v1/location/product/external/bulk';

	/**
	 * Default Kiyoh API server URL.
	 */
	const DEFAULT_SERVER = 'https://www.kiyoh.com';

	/**
	 * Circuit breaker flag - set to true when a request fails.
	 *
	 * @var bool
	 */
	private static bool $circuit_open = false;

	/**
	 * Get the full API URL for an action.
	 *
	 * @param string $action The API action path.
	 * @return string The full API URL.
	 */
	public static function get_api_url( string $action ): string {
		$server = Settings::get_option( 'server', self::DEFAULT_SERVER );

		if ( empty( $server ) || ! filter_var( $server, FILTER_VALIDATE_URL ) ) {
			$server = self::DEFAULT_SERVER;
		}

		return $server . $action;
	}

	/**
	 * Get the API key.
	 *
	 * @return string The API key.
	 */
	public static function get_api_key(): string {
		return Settings::get_option( 'api_key', '' );
	}

	/**
	 * Get the location ID.
	 *
	 * @return string The location ID.
	 */
	public static function get_location_id(): string {
		return Settings::get_option( 'location', '' );
	}

	/**
	 * Check if API credentials are configured.
	 *
	 * @return bool True if credentials are set.
	 */
	public static function has_credentials(): bool {
		return ! empty( self::get_api_key() ) && ! empty( self::get_location_id() );
	}

	/**
	 * Check if debug mode is enabled.
	 *
	 * @return bool Whether debug is enabled.
	 */
	public static function debug_enabled(): bool {
		return 'yes' === Settings::get_option( 'debug' );
	}

	/**
	 * Format error message from API response.
	 *
	 * @param mixed $error The error from API (can be string or array).
	 * @return string Formatted error message.
	 */
	private static function format_error( $error ): string {
		if ( is_array( $error ) ) {
			// Extract just the message field if available.
			if ( isset( $error[0] ) && is_array( $error[0] ) && isset( $error[0]['message'] ) ) {
				return $error[0]['message'];
			}
			// Handle associative array with message key.
			if ( isset( $error['message'] ) ) {
				return $error['message'];
			}
			return wp_json_encode( $error );
		}
		return (string) $error;
	}

	/**
	 * Get default HTTP arguments for API requests.
	 *
	 * @return array Default HTTP arguments.
	 */
	private static function get_default_http_args(): array {
		return array(
			'headers'     => array(
				'Content-Type'            => 'application/json',
				'X-Publication-Api-Token' => self::get_api_key(),
			),
			'httpversion' => '1.0',
			'timeout'     => 30,
		);
	}

	/**
	 * Send a review invitation.
	 *
	 * @param array $invite_data The invitation data.
	 * @return array|\WP_Error Response data or WP_Error on failure.
	 */
	public static function send_invite( array $invite_data ) {
		$http_args = array_merge(
			self::get_default_http_args(),
			array(
				'body'    => wp_json_encode( $invite_data ),
				'timeout' => 45,
			)
		);

		try {
			$response = wp_remote_post( self::get_api_url( self::INVITE_URL ), $http_args );

			if ( is_wp_error( $response ) ) {
				$error_message = $response->get_error_message();
				if ( self::debug_enabled() ) {
					Logger::error( sprintf( 'Invite failed: %s', $error_message ) );
				}
				return new \WP_Error( 'kiyoh_invite_failed', $error_message );
			}

			$response_code = wp_remote_retrieve_response_code( $response );
			$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( 200 === $response_code && is_array( $response_body ) ) {
				if ( self::debug_enabled() ) {
					Logger::info( sprintf( 'Invite sent for order %s', $invite_data['ref_code'] ?? 'unknown' ) );
					Logger::debug( $response_body );
				}
				return $response_body;
			}

			// Determine error message.
			$error_message = __( 'Unknown error', 'kiyoh-reviews' );
			if ( is_array( $response_body ) && isset( $response_body['detailedError'] ) ) {
				$error_message = self::format_error( $response_body['detailedError'] );
			}

			if ( self::debug_enabled() ) {
				Logger::error( sprintf( 'Invite failed: %s', $error_message ) );
			}

			return new \WP_Error( 'kiyoh_invite_failed', $error_message );
		} catch ( \Exception $e ) {
			if ( self::debug_enabled() ) {
				Logger::error( sprintf( 'Invite exception: %s', $e->getMessage() ) );
			}
			return new \WP_Error( 'kiyoh_invite_exception', $e->getMessage() );
		}
	}

	/**
	 * Fetch company review data from Kiyoh API.
	 *
	 * @return array|false The company review data or false on failure.
	 */
	public static function fetch_company_reviews() {
		if ( ! self::has_credentials() ) {
			return false;
		}

		$url = add_query_arg(
			array( 'locationId' => self::get_location_id() ),
			self::get_api_url( self::COMPANY_REVIEW_URL )
		);

		try {
			$response = wp_remote_get( $url, self::get_default_http_args() );

			if ( is_wp_error( $response ) ) {
				if ( self::debug_enabled() ) {
					Logger::error( sprintf( 'Company review fetch failed: %s', $response->get_error_message() ) );
				}
				return false;
			}

			$response_code = wp_remote_retrieve_response_code( $response );
			$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( 200 === $response_code && is_array( $response_body ) ) {
				if ( self::debug_enabled() ) {
					Logger::info( 'Company review data fetched' );
				}
				return $response_body;
			}

			if ( self::debug_enabled() ) {
				if ( is_array( $response_body ) && isset( $response_body['detailedError'] ) ) {
					Logger::error( sprintf( 'Company review fetch failed: %s', self::format_error( $response_body['detailedError'] ) ) );
				} else {
					Logger::error( 'Company review fetch failed' );
					Logger::debug( $response_body );
				}
			}
		} catch ( \Exception $e ) {
			if ( self::debug_enabled() ) {
				Logger::error( sprintf( 'Company review exception: %s', $e->getMessage() ) );
			}
		}

		return false;
	}

	/**
	 * Fetch product review data from Kiyoh API.
	 *
	 * @param int $product_id The product ID.
	 * @return array|null The product reviews or null on failure.
	 */
	public static function fetch_product_reviews( int $product_id ): ?array {
		// Circuit breaker - skip if a previous request failed this page load.
		if ( self::$circuit_open ) {
			return null;
		}

		if ( ! self::has_credentials() ) {
			return null;
		}

		$url = add_query_arg(
			array(
				'locationId'  => self::get_location_id(),
				'productCode' => $product_id,
				'limit'       => 200,
			),
			self::get_api_url( self::PRODUCT_REVIEWS_URL )
		);

		try {
			$http_args = array_merge( self::get_default_http_args(), array( 'timeout' => 3 ) );
			$response  = wp_remote_get( $url, $http_args );

			if ( is_wp_error( $response ) ) {
				self::$circuit_open = true;
				if ( self::debug_enabled() ) {
					Logger::error( sprintf( 'Product reviews fetch failed for product %d: %s', $product_id, $response->get_error_message() ) );
				}
				return null;
			}

			$response_code = wp_remote_retrieve_response_code( $response );
			$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( 200 === $response_code && is_array( $response_body ) ) {
				if ( self::debug_enabled() ) {
					Logger::info( sprintf( 'Product reviews fetched for product %d', $product_id ) );
				}
				return $response_body['reviews'] ?? null;
			}

			// Non-200 response - open circuit breaker.
			self::$circuit_open = true;
			if ( self::debug_enabled() ) {
				if ( is_array( $response_body ) && isset( $response_body['detailedError'] ) ) {
					Logger::error( sprintf( 'Product reviews fetch failed for product %d: %s', $product_id, self::format_error( $response_body['detailedError'] ) ) );
				} else {
					Logger::error( sprintf( 'Product reviews fetch failed for product %d', $product_id ) );
					Logger::debug( $response_body );
				}
			}
		} catch ( \Exception $e ) {
			self::$circuit_open = true;
			if ( self::debug_enabled() ) {
				Logger::error( sprintf( 'Product reviews exception for product %d: %s', $product_id, $e->getMessage() ) );
			}
		}

		return null;
	}

	/**
	 * Test the API connection with current credentials.
	 *
	 * @param string|null $api_key    Optional API key to test (uses saved option if null).
	 * @param string|null $server     Optional server URL to test (uses saved option if null).
	 * @param string|null $location_id Optional location ID to test (uses saved option if null).
	 * @return array{success: bool, message: string} Test result with success status and message.
	 */
	public static function test_connection( ?string $api_key = null, ?string $server = null, ?string $location_id = null ): array {
		$api_key     = $api_key ?? self::get_api_key();
		$server      = $server ?? Settings::get_option( 'server', '' );
		$location_id = $location_id ?? self::get_location_id();

		if ( empty( $api_key ) || empty( $server ) || empty( $location_id ) ) {
			return array(
				'success' => false,
				'message' => __( 'API Key, Server, and Location ID are required.', 'kiyoh-reviews' ),
			);
		}

		$http_args = array(
			'headers'     => array(
				'Content-Type'            => 'application/json',
				'X-Publication-Api-Token' => $api_key,
			),
			'httpversion' => '1.0',
			'timeout'     => 15,
		);

		$url      = add_query_arg( array( 'locationId' => $location_id ), $server . self::COMPANY_REVIEW_URL );
		$response = wp_remote_get( $url, $http_args );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => sprintf(
					/* translators: %s: error message */
					__( 'Connection failed: %s', 'kiyoh-reviews' ),
					$response->get_error_message()
				),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 === $response_code ) {
			return array(
				'success' => true,
				'message' => __( 'Connection successful! API credentials are valid.', 'kiyoh-reviews' ),
			);
		}

		if ( 401 === $response_code || 403 === $response_code ) {
			return array(
				'success' => false,
				'message' => __( 'Authentication failed. Please check your API key.', 'kiyoh-reviews' ),
			);
		}

		if ( isset( $response_body['detailedError'] ) ) {
			return array(
				'success' => false,
				'message' => sprintf(
					/* translators: %s: error message from API */
					__( 'API error: %s', 'kiyoh-reviews' ),
					self::format_error( $response_body['detailedError'] )
				),
			);
		}

		return array(
			'success' => false,
			'message' => sprintf(
				/* translators: %d: HTTP response code */
				__( 'Unexpected response (HTTP %d). Please verify your settings.', 'kiyoh-reviews' ),
				$response_code
			),
		);
	}

	/**
	 * Push products to Kiyoh API.
	 *
	 * @param array $products The products to push.
	 * @return bool True on success.
	 */
	public static function push_products( array $products ): bool {
		$http_args = array_merge(
			self::get_default_http_args(),
			array(
				'method'  => 'PUT',
				'body'    => wp_json_encode(
					array(
						'location_id' => self::get_location_id(),
						'products'    => $products,
					)
				),
				'timeout' => 45,
			)
		);

		try {
			$response = wp_remote_post( self::get_api_url( self::PRODUCT_UPDATE_URL ), $http_args );

			if ( is_wp_error( $response ) ) {
				Logger::error( sprintf( 'Product sync failed: %s', $response->get_error_message() ) );
				return false;
			}

			$response_code = wp_remote_retrieve_response_code( $response );

			if ( 200 === $response_code ) {
				if ( self::debug_enabled() ) {
					Logger::info( 'Product catalog synced successfully' );
				}
				return true;
			}

			$response_body = json_decode( wp_remote_retrieve_body( $response ), true );
			if ( is_array( $response_body ) && isset( $response_body['detailedError'] ) ) {
				Logger::error( sprintf( 'Product sync failed: %s', self::format_error( $response_body['detailedError'] ) ) );
			} else {
				Logger::error( sprintf( 'Product sync failed with HTTP %d', $response_code ) );
			}

			return false;
		} catch ( \Exception $e ) {
			Logger::error( sprintf( 'Product sync exception: %s', $e->getMessage() ) );
			return false;
		}
	}
}
