<?php
/**
 * Admin Settings class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Converzo\KiyohReviews\Api\Client;
use Converzo\KiyohReviews\Logger;
use Converzo\KiyohReviews\Service\ProductSync;

/**
 * Handles admin settings and menu pages.
 */
class Settings {

	/**
	 * Option prefix for all settings.
	 *
	 * @var string
	 */
	const OPTION_PREFIX = 'converzo_kiyoh_';

	/**
	 * Initialize admin hooks.
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'admin_menu', array( static::class, 'register_settings_page' ) );
		add_action( 'admin_enqueue_scripts', array( static::class, 'enqueue_admin_scripts' ) );
		add_action( 'wp_ajax_converzo_kiyoh_sync_products', array( static::class, 'ajax_sync_products' ) );
		add_action( 'wp_ajax_converzo_kiyoh_download_log', array( Logger::class, 'handle_download' ) );
		add_filter( 'plugin_action_links_' . plugin_basename( CONVERZO_KIYOH_PLUGIN_FILE ), array( static::class, 'add_settings_link' ) );

		// Register custom field types for section wrappers.
		add_action( 'woocommerce_admin_field_converzo_kiyoh_section_start', array( static::class, 'output_section_start' ) );
		add_action( 'woocommerce_admin_field_converzo_kiyoh_section_end', array( static::class, 'output_section_end' ) );
	}

	/**
	 * Output opening div for settings section.
	 *
	 * @param array $value Field value.
	 * @return void
	 */
	public static function output_section_start( array $value ): void {
		$class = 'kiyoh-settings-section';
		if ( ! empty( $value['class'] ) ) {
			$class .= ' ' . $value['class'];
		}
		echo '<div class="' . esc_attr( $class ) . '">';
	}

	/**
	 * Output closing div for settings section.
	 *
	 * @param array $value Field value (unused but required by WooCommerce hook signature).
	 * @return void
	 */
	public static function output_section_end( array $value ): void { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		echo '</div>';
	}

	/**
	 * Enqueue admin scripts on settings page.
	 *
	 * @param string $hook_suffix The current admin page.
	 * @return void
	 */
	public static function enqueue_admin_scripts( string $hook_suffix ): void {
		if ( 'settings_page_kiyoh' !== $hook_suffix ) {
			return;
		}

		// Always enqueue admin styles.
		wp_enqueue_style(
			'kiyoh-admin',
			CONVERZO_KIYOH_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			CONVERZO_KIYOH_VERSION
		);

		// Only enqueue sync script if connection is valid.
		if ( ! self::is_connection_valid() ) {
			return;
		}

		wp_enqueue_script(
			'kiyoh-admin-settings',
			CONVERZO_KIYOH_PLUGIN_URL . 'assets/js/admin-settings.js',
			array( 'jquery' ),
			CONVERZO_KIYOH_VERSION,
			true
		);

		wp_localize_script(
			'kiyoh-admin-settings',
			'kiyohSettings',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'converzo_kiyoh_sync_products' ),
				'i18n'    => array(
					'syncing'    => __( 'Syncing...', 'kiyoh-reviews' ),
					'syncFailed' => __( 'Sync failed. Please try again.', 'kiyoh-reviews' ),
				),
			)
		);
	}

	/**
	 * Add settings link to plugin actions.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 */
	public static function add_settings_link( array $links ): array {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'options-general.php?page=kiyoh' ),
			__( 'Settings', 'kiyoh-reviews' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Get option with automatic prefix.
	 *
	 * @param string $key           Option key (without prefix).
	 * @param mixed  $default_value Default value if option doesn't exist.
	 * @return mixed Option value.
	 */
	public static function get_option( string $key, $default_value = '' ) {
		return get_option( self::OPTION_PREFIX . $key, $default_value );
	}

	/**
	 * Update option with automatic prefix.
	 *
	 * @param string $key   Option key (without prefix).
	 * @param mixed  $value Option value.
	 * @return bool True if updated, false otherwise.
	 */
	public static function update_option( string $key, $value ): bool {
		return update_option( self::OPTION_PREFIX . $key, $value );
	}

	/**
	 * Prefix field IDs with the plugin prefix.
	 *
	 * @param array $fields Settings fields array.
	 * @return array Fields with prefixed IDs.
	 */
	private static function prefix_field_ids( array $fields ): array {
		foreach ( $fields as &$field ) {
			if ( isset( $field['id'] ) && strpos( $field['id'], self::OPTION_PREFIX ) !== 0 ) {
				$field['id'] = self::OPTION_PREFIX . $field['id'];
			}
		}
		return $fields;
	}

	/**
	 * Get settings fields configuration for WooCommerce admin fields.
	 *
	 * @return array Settings fields.
	 */
	public static function get_settings_fields(): array {
		return self::prefix_field_ids(
			array(
				// API Connection section.
				array(
					'type'  => 'converzo_kiyoh_section_start',
					'class' => 'kiyoh-settings-section--api',
				),
				array(
					'title' => __( 'API Connection', 'kiyoh-reviews' ),
					'desc'  => __( 'Find your API credentials in your Kiyoh dashboard under Invite > Extra Options. Copy the API key and the location_id.', 'kiyoh-reviews' ),
					'type'  => 'title',
					'id'    => 'api_section',
				),
				array(
					'title'   => __( 'Kiyoh server', 'kiyoh-reviews' ),
					'id'      => 'server',
					'type'    => 'select',
					'default' => 'https://www.kiyoh.com',
					'options' => array(
						'https://www.kiyoh.com'           => 'kiyoh.com',
						'https://www.klantenvertellen.nl' => 'klantenvertellen.nl',
					),
				),
				array(
					'title'       => __( 'API Key', 'kiyoh-reviews' ),
					'id'          => 'api_key',
					'type'        => 'text',
					'css'         => 'min-width: 300px;',
					'placeholder' => __( 'Enter your API key', 'kiyoh-reviews' ),
				),
				array(
					'title'       => __( 'Location ID', 'kiyoh-reviews' ),
					'id'          => 'location',
					'type'        => 'text',
					'css'         => 'min-width: 300px;',
					'placeholder' => __( 'Enter your location ID', 'kiyoh-reviews' ),
				),
				array(
					'type' => 'sectionend',
					'id'   => 'api_section_end',
				),
				array(
					'type' => 'converzo_kiyoh_section_end',
				),

				// Review Invitations section.
				array(
					'type'  => 'converzo_kiyoh_section_start',
					'class' => 'kiyoh-settings-section--invitations',
				),
				array(
					'title' => __( 'Review Invitations', 'kiyoh-reviews' ),
					'desc'  => __( 'Configure when and how review invitation emails are sent to customers.', 'kiyoh-reviews' ),
					'type'  => 'title',
					'id'    => 'invitations_section',
				),
				array(
					'title'   => __( 'Enable invitations', 'kiyoh-reviews' ),
					'desc'    => __( 'Send review invitation emails after order completion', 'kiyoh-reviews' ),
					'id'      => 'reviews_enabled',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'title'   => __( 'Include products', 'kiyoh-reviews' ),
					'desc'    => __( 'Include purchased products in the review invitation', 'kiyoh-reviews' ),
					'id'      => 'review_products',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'title'   => __( 'Order status trigger', 'kiyoh-reviews' ),
					'desc'    => __( 'Send invitation when order reaches this status.', 'kiyoh-reviews' ),
					'id'      => 'order_state_trigger',
					'type'    => 'select',
					'default' => 'completed',
					'options' => array(
						'completed'  => __( 'Completed', 'kiyoh-reviews' ),
						'processing' => __( 'Processing', 'kiyoh-reviews' ),
					),
				),
				array(
					'title'             => __( 'Delay', 'kiyoh-reviews' ),
					'desc'              => __( 'days after order status change. Enter 0 to send immediately.', 'kiyoh-reviews' ),
					'id'                => 'delay',
					'type'              => 'number',
					'default'           => '0',
					'css'               => 'width: 60px;',
					'custom_attributes' => array(
						'min'  => '0',
						'step' => '1',
					),
				),
				array(
					'type' => 'sectionend',
					'id'   => 'invitations_section_end',
				),
				array(
					'type' => 'converzo_kiyoh_section_end',
				),

				// Email Language section.
				array(
					'type'  => 'converzo_kiyoh_section_start',
					'class' => 'kiyoh-settings-section--language',
				),
				array(
					'title' => __( 'Email Language', 'kiyoh-reviews' ),
					'type'  => 'title',
					'id'    => 'language_section',
				),
				array(
					'title'   => __( 'Auto-detect language', 'kiyoh-reviews' ),
					'desc'    => __( 'Set email language based on customer billing country', 'kiyoh-reviews' ),
					'id'      => 'email_lang_billing_country',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'title'   => __( 'Default language', 'kiyoh-reviews' ),
					'desc'    => __( 'Used when auto-detect is disabled or country is not supported.', 'kiyoh-reviews' ),
					'id'      => 'email_lang',
					'type'    => 'select',
					'default' => 'en',
					'options' => self::get_email_lang_codes(),
				),
				array(
					'type' => 'sectionend',
					'id'   => 'language_section_end',
				),
				array(
					'type' => 'converzo_kiyoh_section_end',
				),

				// Product Reviews Display section.
				array(
					'type'  => 'converzo_kiyoh_section_start',
					'class' => 'kiyoh-settings-section--display',
				),
				array(
					'title' => __( 'Product Reviews Display', 'kiyoh-reviews' ),
					'desc'  => __( 'Control how reviews are displayed on product pages.', 'kiyoh-reviews' ),
					'type'  => 'title',
					'id'    => 'display_section',
				),
				array(
					'title'   => __( 'Disable WooCommerce reviews', 'kiyoh-reviews' ),
					'desc'    => __( 'Hide the WooCommerce review form and only show Kiyoh reviews', 'kiyoh-reviews' ),
					'id'      => 'disable_wc_review_form',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'type' => 'sectionend',
					'id'   => 'display_section_end',
				),
				array(
					'type' => 'converzo_kiyoh_section_end',
				),

				// Advanced section.
				array(
					'type'  => 'converzo_kiyoh_section_start',
					'class' => 'kiyoh-settings-section--advanced',
				),
				array(
					'title' => __( 'Advanced', 'kiyoh-reviews' ),
					'type'  => 'title',
					'id'    => 'advanced_section',
				),
				array(
					'title'   => __( 'Debug logging', 'kiyoh-reviews' ),
					'desc'    => sprintf(
						/* translators: %s: download link */
						__( 'Log API requests and errors for troubleshooting. %s', 'kiyoh-reviews' ),
						'<a href="' . esc_url( Logger::get_download_url() ) . '" target="_blank">' . __( 'Download log', 'kiyoh-reviews' ) . '</a>'
					),
					'id'      => 'debug',
					'default' => 'no',
					'type'    => 'checkbox',
				),
				array(
					'type' => 'sectionend',
					'id'   => 'advanced_section_end',
				),
				array(
					'type' => 'converzo_kiyoh_section_end',
				),
			)
		);
	}

	/**
	 * Save settings using WooCommerce helper.
	 *
	 * @return void
	 */
	public static function save_settings(): void {
		woocommerce_update_options( self::get_settings_fields() );
	}

	/**
	 * Register the settings page in admin menu.
	 *
	 * @return void
	 */
	public static function register_settings_page(): void {
		add_options_page(
			'Kiyoh reviews settings',
			'Kiyoh',
			'manage_options',
			'kiyoh',
			array( static::class, 'render_settings_page' )
		);
	}

	/**
	 * Render the settings page.
	 *
	 * @return void
	 */
	public static function render_settings_page(): void {
		include CONVERZO_KIYOH_PLUGIN_DIR . 'templates/admin/kiyoh-settings.php';
	}

	/**
	 * AJAX handler to sync products with Kiyoh.
	 *
	 * @return void
	 */
	public static function ajax_sync_products(): void {
		check_ajax_referer( 'converzo_kiyoh_sync_products', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'kiyoh-reviews' ) ) );
		}

		if ( ! self::is_connection_valid() ) {
			wp_send_json_error( array( 'message' => __( 'No valid API connection. Please save valid credentials first.', 'kiyoh-reviews' ) ) );
		}

		if ( ProductSync::sync_catalog() ) {
			wp_send_json_success( array( 'message' => __( 'Products successfully synced!', 'kiyoh-reviews' ) ) );
		}

		wp_send_json_error( array( 'message' => __( 'Failed to sync products. Check the debug log for details.', 'kiyoh-reviews' ) ) );
	}

	/**
	 * Get available email language codes.
	 *
	 * @return array<string, string> Language codes and labels.
	 */
	public static function get_email_lang_codes(): array {
		return array(
			'en'    => 'English',
			'nl'    => 'Nederlands',
			'fi-FI' => 'Suomalainen',
			'fr'    => 'Français',
			'be'    => 'Vlaams',
			'de'    => 'German',
			'hu'    => 'Hungarian',
			'bg'    => 'Bulgarian',
			'ro'    => 'Romanian',
			'hr'    => 'Croatian',
			'ja'    => 'Japanese',
			'es-ES' => 'Spanish',
			'it'    => 'Italian',
			'pt-PT' => 'Portuguese',
			'tr'    => 'Turkish',
			'nn-NO' => 'Norwegian',
			'sv-SE' => 'Swedish',
			'da'    => 'Danish',
			'pt-BR' => 'Brazilian Portuguese',
			'pl'    => 'Polish',
			'sl'    => 'Slovenian',
			'zh-CN' => 'Chinese',
			'ru'    => 'Russian',
			'el'    => 'Greek',
			'cs'    => 'Czech',
			'et'    => 'Estonian',
			'lt'    => 'Lithuanian',
			'lv'    => 'Latvian',
			'sk'    => 'Sloviak',
		);
	}

	/**
	 * Map country codes to language codes.
	 *
	 * @return array<string, string> Country to language mapping.
	 */
	public static function map_country_lang(): array {
		return array(
			'GB' => 'en',
			'NL' => 'nl',
			'FI' => 'fi-FI',
			'FR' => 'fr',
			'BE' => 'be',
			'DE' => 'de',
			'HU' => 'hu',
			'BG' => 'bg',
			'RO' => 'ro',
			'HR' => 'hr',
			'JP' => 'ja',
			'ES' => 'es-ES',
			'IT' => 'it',
			'PT' => 'pt-PT',
			'TR' => 'tr',
			'NO' => 'nn-NO',
			'SE' => 'sv-SE',
			'DK' => 'da',
			'BR' => 'pt-BR',
			'PL' => 'pl',
			'SI' => 'sl',
			'CN' => 'zh-CN',
			'RU' => 'ru',
			'GR' => 'el',
			'CZ' => 'cs',
			'EE' => 'et',
			'LT' => 'lt',
			'LV' => 'lv',
			'SK' => 'sk',
		);
	}

	/**
	 * Get available order state triggers.
	 *
	 * @return array<string> Order states that can trigger invites.
	 */
	public static function get_order_state_triggers(): array {
		return array( 'completed', 'processing' );
	}

	/**
	 * Get the debug log download URL.
	 *
	 * @return string The admin download URL.
	 */
	public static function get_debug_log_url(): string {
		return Logger::get_download_url();
	}

	/**
	 * Test API connection and store result.
	 *
	 * @return array{success: bool, message: string} Connection test result.
	 */
	public static function test_and_store_connection(): array {
		$result = Client::test_connection();

		// Store connection status persistently.
		self::update_option( 'connection_valid', $result['success'] ? '1' : '0' );

		return $result;
	}

	/**
	 * Check if the API connection is valid.
	 *
	 * @return bool True if connection is valid.
	 */
	public static function is_connection_valid(): bool {
		return '1' === self::get_option( 'connection_valid', '0' );
	}
}
