<?php
/**
 * Plugin Name:       Kiswa COD Fee for WooCommerce
 * Plugin URI:        https://kiswa.net/plugins/cod/
 * Description:       Add a simple Cash on Delivery (COD) fee to WooCommerce orders with fixed or percentage options.
 * Version:           1.0.0
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Requires Plugins:  woocommerce
 * Author:            Kiswa Solutions
 * Author URI:        https://www.kiswa.net
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       kiswa-cod-fee-for-woocommerce
 * Domain Path:       /languages
 */


if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! defined( 'WCCF_VERSION' ) ) {
	define( 'WCCF_VERSION', '1.0.0' );
}

if ( ! defined( 'WCCF_PLUGIN_FILE' ) ) {
	define( 'WCCF_PLUGIN_FILE', __FILE__ );
}



/**
 * Initialize plugin only if WooCommerce is active.
 */
function wccf_init_plugin() {

	// Check WooCommerce.
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'wccf_admin_notice_woocommerce_required' );
		return;
	}

	// Add settings tab.
	add_filter( 'woocommerce_settings_tabs_array', 'wccf_add_settings_tab', 50 );
	add_action( 'woocommerce_settings_tabs_wccf_cod_fee', 'wccf_render_settings_tab' );
	add_action( 'woocommerce_update_options_wccf_cod_fee', 'wccf_update_settings' );

	// Apply COD fee.
	add_action( 'woocommerce_cart_calculate_fees', 'wccf_apply_cod_fee' );
}
add_action( 'plugins_loaded', 'wccf_init_plugin', 20 );

/**
 * Enqueue frontend script for syncing payment method.
 */
function wccf_enqueue_frontend_scripts() {
	if ( function_exists( 'is_checkout' ) && is_checkout() && ! is_order_received_page() ) {
		wp_enqueue_script(
			'wccf-checkout',
			plugins_url( 'assets/js/wccf-checkout.js', WCCF_PLUGIN_FILE ),
			array( 'jquery', 'wc-checkout' ),
			WCCF_VERSION,
			true
		);
	}
}
add_action( 'wp_enqueue_scripts', 'wccf_enqueue_frontend_scripts' );



/**
 * Admin notice if WooCommerce is not active.
 */
function wccf_admin_notice_woocommerce_required() {
	if ( ! current_user_can( 'activate_plugins' ) ) {
		return;
	}

	$screen = get_current_screen();
	if ( isset( $screen->id ) && 'plugins' !== $screen->id ) {
		// Only show on plugins page.
		return;
	}

	echo '<div class="notice notice-error"><p>';
	echo esc_html__( 'Cash on Delivery Fee for WooCommerce requires WooCommerce to be installed and active.', 'kiswa-cod-fee-for-woocommerce' );
	echo '</p></div>';
}

/**
 * Add a new settings tab in WooCommerce > Settings.
 *
 * @param array $tabs Existing tabs.
 * @return array
 */
function wccf_add_settings_tab( $tabs ) {
	$tabs['wccf_cod_fee'] = __( 'COD Fee', 'kiswa-cod-fee-for-woocommerce' );
	return $tabs;
}

/**
 * Get plugin settings fields.
 *
 * @return array
 */
function wccf_get_settings() {

	$settings = array(

		array(
			'title' => __( 'Cash on Delivery Fee Settings', 'kiswa-cod-fee-for-woocommerce' ),
			'type'  => 'title',
			'id'    => 'wccf_cod_fee_options',
			'desc'  => __( 'Configure an extra fee when customers choose the Cash on Delivery payment method.', 'kiswa-cod-fee-for-woocommerce' ),
		),

		array(
			'title'   => __( 'Enable COD Fee', 'kiswa-cod-fee-for-woocommerce' ),
			'id'      => 'wccf_cod_fee_enabled',
			'type'    => 'checkbox',
			'default' => 'no',
			'desc'    => __( 'Enable an additional fee when Cash on Delivery is selected.', 'kiswa-cod-fee-for-woocommerce' ),
		),

		array(
			'title'   => __( 'Fee Label', 'kiswa-cod-fee-for-woocommerce' ),
			'id'      => 'wccf_cod_fee_label',
			'type'    => 'text',
			'default' => __( 'Cash on Delivery Fee', 'kiswa-cod-fee-for-woocommerce' ),
			'desc'    => __( 'This label appears in the cart and checkout totals.', 'kiswa-cod-fee-for-woocommerce' ),
		),

		array(
			'title'   => __( 'Fee Type', 'kiswa-cod-fee-for-woocommerce' ),
			'id'      => 'wccf_cod_fee_type',
			'type'    => 'select',
			'options' => array(
				'fixed'      => __( 'Fixed amount', 'kiswa-cod-fee-for-woocommerce' ),
				'percentage' => __( 'Percentage of cart subtotal', 'kiswa-cod-fee-for-woocommerce' ),
			),
			'default' => 'fixed',
			'desc'    => __( 'Choose whether the fee is a fixed amount or a percentage of the cart subtotal.', 'kiswa-cod-fee-for-woocommerce' ),
		),

		array(
			'title'    => __( 'Fee Amount', 'kiswa-cod-fee-for-woocommerce' ),
			'id'       => 'wccf_cod_fee_amount',
			'type'     => 'number',
			'desc'     => __( 'Enter the fee amount. For percentage, use numbers like 2 for 2%.', 'kiswa-cod-fee-for-woocommerce' ),
			'css'      => 'max-width: 100px;',
			'default'  => '0',
			'custom_attributes' => array(
				'step' => '0.01',
				'min'  => '0',
			),
		),

		array(
			'title'    => __( 'Minimum Order Total', 'kiswa-cod-fee-for-woocommerce' ),
			'id'       => 'wccf_cod_fee_min_total',
			'type'     => 'number',
			'desc'     => __( 'Optional: only apply the fee if the cart subtotal is at least this amount. Leave empty or 0 to always apply.', 'kiswa-cod-fee-for-woocommerce' ),
			'css'      => 'max-width: 100px;',
			'default'  => '0',
			'custom_attributes' => array(
				'step' => '0.01',
				'min'  => '0',
			),
		),

		array(
			'type' => 'sectionend',
			'id'   => 'wccf_cod_fee_options',
		),

	);

	/**
	 * Filter settings in case a developer wants to extend.
	 */
	return apply_filters( 'wccf_cod_fee_settings', $settings );
}

/**
 * Render settings tab.
 */
function wccf_render_settings_tab() {
	woocommerce_admin_fields( wccf_get_settings() );
}

/**
 * Save settings.
 */
function wccf_update_settings() {
	woocommerce_update_options( wccf_get_settings() );
}


/**
 * Apply the COD fee when appropriate.
 *
 * Hooked into woocommerce_cart_calculate_fees.
 *
 * @param WC_Cart $cart Cart object.
 */
function wccf_apply_cod_fee( $cart ) {

	// Don't run in admin (except during AJAX).
	if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
		return;
	}

	if ( ! $cart instanceof WC_Cart ) {
		return;
	}

	// Check if enabled.
	$enabled = get_option( 'wccf_cod_fee_enabled', 'no' );
	if ( 'yes' !== $enabled ) {
		return;
	}

	$chosen_payment_method = '';

	// Case 1: AJAX checkout update – Woo sends everything in post_data.
	if ( defined( 'DOING_AJAX' ) && DOING_AJAX && isset( $_POST['post_data'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$data           = array();
		$post_data_raw  = wp_unslash( $_POST['post_data'] ); // phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$post_data_raw  = is_string( $post_data_raw ) ? sanitize_textarea_field( $post_data_raw ) : '';
		parse_str( $post_data_raw, $data );

		if ( isset( $data['wccf_payment_method'] ) ) {
			$chosen_payment_method = sanitize_text_field( $data['wccf_payment_method'] );
		} elseif ( isset( $data['payment_method'] ) ) {
			$chosen_payment_method = sanitize_text_field( $data['payment_method'] );
		}
	}

		// Case 2: Non-AJAX post (edge cases).
	// PHPCS: WooCommerce checkout POST is already nonce-protected at the endpoint level.
	// We only read and sanitize the current payment method, no data is stored or modified.
	// phpcs:disable WordPress.Security.NonceVerification.Missing
	if ( empty( $chosen_payment_method ) && isset( $_POST['wccf_payment_method'] ) ) {
		$chosen_payment_method = sanitize_text_field( wp_unslash( $_POST['wccf_payment_method'] ) );
	} elseif ( empty( $chosen_payment_method ) && isset( $_POST['payment_method'] ) ) {
		$chosen_payment_method = sanitize_text_field( wp_unslash( $_POST['payment_method'] ) );
	}
	// phpcs:enable WordPress.Security.NonceVerification.Missing


	// Case 3: Fallback to Woo session.
	if ( empty( $chosen_payment_method ) && WC()->session ) {
		$chosen_payment_method = WC()->session->get( 'chosen_payment_method', '' );
	}

	// Only proceed for COD.
	if ( 'cod' !== $chosen_payment_method ) {
		return;
	}

	$label      = get_option( 'wccf_cod_fee_label', __( 'Cash on Delivery Fee', 'kiswa-cod-fee-for-woocommerce' ) );
	$fee_type   = get_option( 'wccf_cod_fee_type', 'fixed' );
	$fee_amount = wc_format_decimal( get_option( 'wccf_cod_fee_amount', '0' ) );
	$min_total  = wc_format_decimal( get_option( 'wccf_cod_fee_min_total', '0' ) );

	if ( $fee_amount <= 0 ) {
		return;
	}

	$cart_subtotal = (float) $cart->get_subtotal();

	// Minimum order total check.
	if ( $min_total > 0 && $cart_subtotal < $min_total ) {
		return;
	}

	// Calculate final fee.
	if ( 'percentage' === $fee_type ) {
		$fee_to_add = ( $cart_subtotal * ( $fee_amount / 100 ) );
	} else {
		$fee_to_add = $fee_amount;
	}

	if ( $fee_to_add <= 0 ) {
		return;
	}

	// Apply fee (non-taxable by default).
	$cart->add_fee(
		$label,
		$fee_to_add,
		false
	);
}
add_action( 'woocommerce_cart_calculate_fees', 'wccf_apply_cod_fee' );





/**
 * Output hidden field to store the selected payment method.
 */
function wccf_output_hidden_payment_field() {
	echo '<input type="hidden" name="wccf_payment_method" id="wccf_payment_method" value="" />';
}
add_action( 'woocommerce_review_order_before_payment', 'wccf_output_hidden_payment_field' );


