<?php

/**
 * Plugin Name: KissMyUpTime
 * Description: Publishes a simple endpoint (/kmut.txt) to expose your account identifier for integration with KissMyUpTime - the completely free uptime monitoring service.
 * Version: 1.0.0
 * Author: HAZOTA Europe Kft.
 * Author URI: https://www.hazota.eu/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires PHP: 7.0
 * Requires at least: 5.2
 */

// Exit on direct access

if (
  ! defined('ABSPATH')
) {
  exit;
}

// Register kmut_account_id setting

function kmut_register_settings () {
  register_setting('kmut_options_group', 'kmut_account_id', [
    'type' => 'string',
    'sanitize_callback' => 'sanitize_text_field',
    'default' => '',
  ]);
}

add_action('admin_init', 'kmut_register_settings');

// Add settings link

function kmut_plugin_action_links ( $links ) {
  $settings_link = '<a href="options-general.php?page=kmut_settings">Settings</a>';
  array_unshift($links, $settings_link);
  return $links;
}

add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'kmut_plugin_action_links');

// Add settings menu

function kmut_add_admin_menu () {
  add_options_page(
    'KissMyUpTime Settings',
    'KissMyUpTime',
    'manage_options',
    'kmut_settings',
    'kmut_options_page'
  );
}

add_action('admin_menu', 'kmut_add_admin_menu');

// Render settings page

function kmut_options_page () {

  if (
    ! current_user_can('manage_options')
  ) {
    wp_die('You do not have sufficient permissions to access this page.');
  }

  $account_id = get_option('kmut_account_id');
  $kmut_txt_url = site_url('/kmut.txt');
  $site_host = wp_parse_url(site_url(), PHP_URL_HOST);

  echo '<div class="wrap">';
  echo '<h1>KissMyUpTime Settings</h1>';
  echo '<form method="post" action="options.php">';

  settings_fields('kmut_options_group');
  do_settings_sections('kmut_options_group');

  echo '<table class="form-table">';
  echo '<tr valign="top">';
  echo '<th scope="row">Account Identifier</th>';
  echo '<td><input type="text" name="kmut_account_id" value="' . esc_attr($account_id) . '" size="50" /></td>';
  echo '</tr>';
  echo '</table>';

  submit_button();

  echo '</form>';
  echo '<h2>Usage</h2>';
  echo '<ol>';
  echo '<li>Open your existing KissMyUpTime account or create a new one at <a href="https://kissmyuptime.com/create" target="_blank">https://kissmyuptime.com/create</a> - no personal data required.</li>';
  echo '<li>Go to <b>Manage targets &gt; Create target</b>, and copy your account identifier.</li>';
  echo '<li>Paste the account identifier above, then click <b>Save changes</b>. Optionally, open <a href="' . esc_url($kmut_txt_url) . '" target="_blank">' . esc_url($kmut_txt_url) . '</a> to verify the changes.</li>';
  echo '<li>Back in <b>Manage targets &gt; Create target</b>, set the host to <code>' . esc_html($site_host) . '</code>, customize any additional settings if needed, then click <b>Create target</b>.</li>';
  echo '<li>Go to <b>Create check</b>, provide a name and group, select the <code>' . esc_html($site_host) . '</code> target, customize any additional settings if needed, and click <b>Create check</b>.</li>';
  echo '</ol>';
  echo '<p>Done! Enjoy your completely free uptime monitoring service.</p>';
  echo '</div>';

}

// Add kmut_txt query variable

function kmut_query_vars ( $vars ) {
  $vars[] = 'kmut_txt';
  return $vars;
}

add_filter('query_vars', 'kmut_query_vars');

// Add rewrite rule on page load from /kmut.txt to a not empty kmut_txt query variable

function kmut_init_rewrite_rule () {
  add_rewrite_rule('^kmut\.txt$', 'index.php?kmut_txt=1', 'top');
}

add_action('init', 'kmut_init_rewrite_rule');

// Display kmut_account_id setting when /kmut.txt is accessed so the kmut_txt query variable is not empty

function kmut_parse_request ( $wp ) {
  if (
    ! empty($wp->query_vars['kmut_txt'])
  ) {
    $account_id = get_option('kmut_account_id');
    header('Content-Type: text/plain; charset=utf-8');
    echo esc_html($account_id ?: 'Account identifier not set.');
    exit;
  }
}

add_action('parse_request', 'kmut_parse_request');

// Prevent redirect from /kmut.txt to /kmut.txt/

function kmut_redirect_canonical ( $redirect_url, $requested_url ) {
  if (
    isset($_SERVER['REQUEST_URI']) &&
    strpos(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), '/kmut.txt') === 0
  ) {
    return false;
  } else {
    return $redirect_url;
  }
}

add_filter('redirect_canonical', 'kmut_redirect_canonical', 10, 2);

// Add rewrite rule on activation

function kmut_activation_hook () {
  kmut_init_rewrite_rule();
  flush_rewrite_rules();
}

register_activation_hook(__FILE__, 'kmut_activation_hook');

// Clear rewrite rule on deactivation

function kmut_deactivation_hook () {
  flush_rewrite_rules();
}

register_deactivation_hook(__FILE__, 'kmut_deactivation_hook');
