<?php 

// Exit if accessed directly for security
if (!defined('ABSPATH')) {
     exit;
}

/**
 * Main plugin class to handle functionality
 */
class kidsenal_kids_english_alphabet {

     /**
      * Constructor to hook actions and shortcode
      */
     public function __construct() {
          // Enqueue scripts and styles on the frontend
          add_action('wp_enqueue_scripts', [$this, 'kidsenal_enqueue_assets']);
          
          // Register shortcode to display alphabet buttons
          add_shortcode('kidsenal_english_alphabet_ajax', [$this, 'kidsenal_shortcode']);

          // Register AJAX actions for logged-in and guest users
          add_action('wp_ajax_kidsenal_alphabet_video', [$this, 'kidsenal_handle_ajax']);
          add_action('wp_ajax_nopriv_kidsenal_alphabet_video', [$this, 'kidsenal_handle_ajax']);

          // Admin notice on plugin activation (optional)
          add_action('admin_notices', function() {
               echo '<div class="notice notice-success is-dismissible">
                <p><strong>Kids English Alphabet Pronunciation Activated!</strong><br>
                Use this shortcode <code>[kidsenal_english_alphabet_ajax]</code> to display the alphabet buttons.</p>
               </div>';
          });
     }

     /**
      * Enqueue CSS and JS files on the frontend
      */
     public function kidsenal_enqueue_assets() {
         define('KIDSENAL_VERSION', '1.0.0');

         // Enqueue CSS file
         wp_enqueue_style('kidsenal_style', KIDSENAL_URL . 'assets/css/style.css', array(), KIDSENAL_VERSION);
         
         // Enqueue JS file with jQuery dependency, loaded in footer
         wp_enqueue_script('kidsenal_script', KIDSENAL_URL . 'assets/js/script.js', array('jquery'), KIDSENAL_VERSION, true);

         // Pass AJAX URL and nonce to JS
         wp_localize_script('kidsenal_script', 'kidsenalData', [
               'ajax_url' => admin_url('admin-ajax.php'),
               'nonce'    => wp_create_nonce('kidsenal_alphabet_nonce'),
         ]);
     }

     /**
      * Shortcode callback to output alphabet buttons
      * 
      * @return string HTML markup
      */
     public function kidsenal_shortcode() {
          ob_start();
          ?>
          <div class="kids-alphabet-container">
               <h2 class="alphabet-heading"> 🎵 Let's Learn the Alphabet! 🎵 </h2>

               <div class="alphabet-buttons">
                    <?php 
                    // Loop through letters A-Z to create buttons
                    foreach (range('A', 'Z') as $letter): ?>
                         <button class="alphabet-button" data-letter="<?php echo esc_attr(strtolower($letter)); ?>">
                              <?php echo esc_html($letter); ?>
                         </button>
                    <?php endforeach; ?>
               </div>

               <div id="video-container" class="alphabet-video"></div>
          </div>
          <?php
          return ob_get_clean();
     }

     /**
      * AJAX handler to return video HTML for the selected letter
      */
     public function kidsenal_handle_ajax() {
          // Verify nonce for security
          if (!isset($_POST['_ajax_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])), 'kidsenal_alphabet_nonce')) {
               wp_send_json_error('Invalid nonce');
          }

          // Check if letter parameter is set
          if (!isset($_POST['letter'])) {
               wp_send_json_error('Letter not set.');
          }

          // Sanitize input letter
          $letter = sanitize_text_field(wp_unslash($_POST['letter']));

          // Construct video file path and URL
          $video_path = KIDSENAL_PATH . "video/{$letter}.mp4";
          $video_url = KIDSENAL_URL . "video/{$letter}.mp4";

        

          // Generate video HTML markup
          ob_start();
          ?>
          <video src="<?php echo esc_url($video_url); ?>" type="video/mp4" autoplay controls hidden></video>
          <?php
          wp_send_json_success(ob_get_clean());
     }
}

?>
