<?php
if (!defined('ABSPATH')) {
    exit;
}

// Capability check
if (!current_user_can('manage_options')) {
    wp_die(esc_html_e('You do not have permission to access this page.', 'keygin-erp-sync'));
}

require_once KEYGIN_ERP_SYNC_PATH . 'includes/logs/class-logger.php';
$keygin_logger = Keygin_Logger::init();

/**
 * Clear today's logs
 */
if (
    isset($_POST['keygin_clear_logs']) &&
    isset($_POST['keygin_clear_logs_nonce']) &&
    current_user_can('manage_options') &&
    wp_verify_nonce(
        sanitize_text_field(wp_unslash($_POST['keygin_clear_logs_nonce'])),
        'keygin_clear_logs_action'
    )
) {
    $keygin_logger->clear_today_logs();
    echo '<div class="notice notice-success is-dismissible"><p>' .
        esc_html_e('Today’s logs have been cleared.', 'keygin-erp-sync') .
    '</p></div>';
}

/**
 * Get logs
 */
$logs = $keygin_logger->get_event_logs();

/**
 * Detect free limitations
 */
$keygin_upgrade_flag = false;
foreach ($logs as $log) {
    if (
        stripos($log['message'], 'limit') !== false ||
        stripos($log['message'], 'chunk') !== false ||
        stripos($log['message'], 'free') !== false
    ) {
        $keygin_upgrade_flag = true;
        break;
    }
}

/**
 * Load latest log file if empty
 */
if (empty($logs)) {
    $files = $keygin_logger->get_available_event_files();
    if (!empty($files)) {
        rsort($files);
        if (preg_match('/keygin-events-(\d{4}-\d{2}-\d{2})\.log$/', basename($files[0]), $m)) {
            $logs = $keygin_logger->get_event_logs($m[1]);
        }
    }
}

/**
 * Search
 */
$search = isset($_GET['s'])
    ? sanitize_text_field(wp_unslash($_GET['s']))
    : '';

if ($search) {
    $logs = array_filter($logs, function ($log) use ($search) {
        return stripos($log['message'], $search) !== false || stripos($log['type'], $search) !== false;
    });
}

/**
 * Pagination
 */
$per_page = 20;
$total     = count($logs);
$pages     = max(1, ceil($total / $per_page));
$page      = isset($_GET['paged'])
    ? max(1, intval(wp_unslash($_GET['paged'])))
    : 1;

$logs_page = array_slice($logs, ($page - 1) * $per_page, $per_page);

$settings  = (array) get_option('keygin_settings', []);
$frequency = sanitize_text_field($settings['frequency'] ?? '');

/**
 * Page param
 */
$page_slug = isset($_GET['page'])
    ? sanitize_text_field(wp_unslash($_GET['page']))
    : '';
?>

<div class="wrap keygin-logs">

    <div class="keygin-header-content">
        <div class="keygin-header-title">
            <div class="keygin-title">
                <h1><?php esc_html_e('Keygin Erp Sync', 'keygin-erp-sync'); ?></h1>
            </div>
            <p class="keygin-tagline">
                <?php esc_html_e('Automatic synchronization between WooCommerce and Contifico', 'keygin-erp-sync'); ?>
            </p>
        </div>
    </div>

    <p>
        <strong><?php esc_html_e('Synchronization frequency:', 'keygin-erp-sync'); ?></strong>
        <?php echo $frequency ? esc_html(ucfirst($frequency)) : esc_html_e('Not configured', 'keygin-erp-sync'); ?>
    </p>

     <?php if ($keygin_upgrade_flag): ?>
        <div class="notice notice-warning">
            <p>
                <?php esc_html_e(
                    'This synchronization process is currently limited to the first 50 products. For larger catalogs, additional synchronization capacity is available.',
                    'keygin-erp-sync'
                ); ?>
            </p>
        </div>
    <?php endif; ?>


    <!-- Search -->
    <form method="get" style="margin-bottom:15px;">
        <input type="hidden" name="page" value="<?php echo esc_attr($page_slug); ?>">

        <input
            type="search"
            name="s"
            value="<?php echo esc_attr($search); ?>"
            size="40"
            placeholder="<?php esc_attr_e('Search logs…', 'keygin-erp-sync'); ?>"
        >

        <button class="button"><?php esc_html_e('Search', 'keygin-erp-sync'); ?></button>

        <?php if ($search): ?>
            <a href="<?php echo esc_url(remove_query_arg(['s','paged'])); ?>" class="button-secondary">
                <?php esc_html_e('Clear search', 'keygin-erp-sync'); ?>
            </a>
        <?php endif; ?>
    </form>

    <!-- Filters -->
    <div class="keygin-log-filters" style="margin-bottom:15px;">
        <?php
        $filters = [
            'all'     => esc_html_e('All', 'keygin-erp-sync'),
            'success' => esc_html_e('Success', 'keygin-erp-sync'),
            'warning' => esc_html_e('Warnings', 'keygin-erp-sync'),
            'error'   => esc_html_e('Errors', 'keygin-erp-sync'),
            'info'    => esc_html_e('Info', 'keygin-erp-sync'),
        ];
        foreach ($filters as $type => $label) {
            echo '<button class="button filter-btn" data-type="' . esc_attr($type) . '">' . esc_html($label) . '</button> ';
        }
        ?>
    </div>

    <table class="widefat striped">
        <thead>
            <tr>
                <th><?php esc_html_e('Date', 'keygin-erp-sync'); ?></th>
                <th><?php esc_html_e('Type', 'keygin-erp-sync'); ?></th>
                <th><?php esc_html_e('Message', 'keygin-erp-sync'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php if (!$logs_page): ?>
                <tr><td colspan="3"><?php esc_html_e('No logs found.', 'keygin-erp-sync'); ?></td></tr>
            <?php else: foreach ($logs_page as $log): ?>
                <tr class="log-row <?php echo esc_attr($log['type']); ?>">
                    <td><?php echo esc_html($log['date']); ?></td>
                    <td><?php echo esc_html(ucfirst($log['type'])); ?></td>
                    <td><?php echo esc_html($log['message']); ?></td>
                </tr>
            <?php endforeach; endif; ?>
        </tbody>
    </table>

    <?php if ($pages > 1): ?>
        <div class="tablenav-pages" style="margin-top:15px;">
            <?php
            $base = remove_query_arg('paged');

            if ($page > 1) {
                echo '<a class="button" href="' . esc_url(add_query_arg('paged', $page - 1, $base)) . '">' .
                    esc_html_e('Previous', 'keygin-erp-sync') . '</a> ';
            }

            printf(
                '<span>%s</span>',
                esc_html(sprintf(esc_html_e('Page %1$d of %2$d', 'keygin-erp-sync'), $page, $pages))
            );

            if ($page < $pages) {
                echo ' <a class="button" href="' . esc_url(add_query_arg('paged', $page + 1, $base)) . '">' .
                    esc_html_e('Next', 'keygin-erp-sync') . '</a>';
            }
            ?>
        </div>
    <?php endif; ?>

    <form method="post" style="margin-top:15px;">
        <?php wp_nonce_field('keygin_clear_logs_action', 'keygin_clear_logs_nonce'); ?>
        <button type="submit" name="keygin_clear_logs" class="button button-secondary">
            <?php esc_html_e('Clear today’s logs', 'keygin-erp-sync'); ?>
        </button>
    </form>

</div>