<?php
if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__DIR__) . 'api/class-api.php';
require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';

class Keygin_Stock_Sync {

    private $keygin_api;
    private $settings;
    private $warehouse_id;

    /** @var Keygin_Logger|null */
    private $logger;

    public function __construct() {
        $this->settings     = (array) get_option('keygin_settings', []);
        $api_key            = sanitize_text_field($this->settings['api_key'] ?? '');
        $api_token          = sanitize_text_field($this->settings['api_token'] ?? '');
        $this->warehouse_id = sanitize_text_field($this->settings['warehouse'] ?? '');

        $this->keygin_api = new Keygin_API($api_key, $api_token);
        $this->logger     = class_exists('Keygin_Logger') ? Keygin_Logger::init() : null;

        add_action('woocommerce_order_status_completed', [$this, 'update_stock_on_order_complete']);
    }

    /**
     * Sends stock output to Keygin when a WooCommerce order is completed.
     */
    public function update_stock_on_order_complete($order_id) {

        if (!$this->logger) {
            return;
        }

        $order_id = absint($order_id);
        if (!$order_id) {
            return;
        }

        if (did_action('woocommerce_checkout_process') === 0 && !wp_doing_cron()) {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            $this->logger->debug(
                'info',
                sprintf(__('Order #%d not found.', 'keygin-erp-sync'), $order_id)
            );
            return;
        }

        if (empty($this->warehouse_id)) {
            $this->logger->debug(
                'info',
                __('No warehouse configured for stock outputs in Keygin.', 'keygin-erp-sync')
            );
            return;
        }

        $details = [];
        $untracked_products = [];

        foreach ($order->get_items() as $item) {
            $product = $item->get_product();

            if (!$product || !is_a($product, 'WC_Product')) {
                continue;
            }

            $product_id    = $product->get_id();
            $keygin_id     = sanitize_text_field(get_post_meta($product_id, '_keygin_id', true));
            $qty_purchased = (int) $item->get_quantity();

            if (empty($keygin_id)) {
                $untracked_products[] = $product_id;
                continue;
            }

            $details[] = [
                'producto_id' => $keygin_id,
                'cantidad'    => (string) $qty_purchased,
            ];
        }

        if (!empty($untracked_products) && !empty($details)) {
            $this->logger->event(
                'warning',
                sprintf(
                    __('Stock output NOT sent because order #%d contains non-synced products.', 'keygin-erp-sync'),
                    $order_id
                )
            );
            return;
        }

        if (empty($details)) {
            $this->logger->event(
                'warning',
                sprintf(
                    __('Stock output NOT sent because order #%d contains no synced products.', 'keygin-erp-sync'),
                    $order_id
                )
            );
            return;
        }

        $data = [
            'tipo'        => 'EGR',
            'fecha'       => gmdate('d/m/Y'),
            'bodega_id'   => $this->warehouse_id,
            'detalles'    => $details,
            'descripcion' => sprintf(__('WooCommerce order #%d stock output.', 'keygin-erp-sync'), $order_id),
        ];

        $response = $this->keygin_api->crear_egreso_inventario($data);

        $this->logger->event(
            $response ? 'success' : 'error',
            sprintf(
                $response
                    ? __('Stock output successfully created for order #%d.', 'keygin-erp-sync')
                    : __('Error creating stock output for order #%d.', 'keygin-erp-sync'),
                $order_id
            )
        );
    }

    /**
     * Sync stock from Keygin to WooCommerce.
     */
    public function sync_stock() {

        if (!$this->logger) {
            return false;
        }

        $response = $this->keygin_api->get_products();
        $products = is_array($response['data'] ?? null) ? $response['data'] : [];

        if (empty($products)) {
            $this->logger->debug(
                'error',
                __('No products found or error retrieving products for stock sync.', 'keygin-erp-sync')
            );
            return false;
        }

        foreach ($products as $product) {
            $this->update_local_stock($product);
        }

        return true;
    }

    /**
     * Update local WooCommerce stock.
     */
    private function update_local_stock($product_data) {

        if (empty($product_data['id'])) {
            return false;
        }

        $keygin_id = sanitize_text_field($product_data['id']);
        $new_stock = max(0, (int) ($product_data['cantidad_stock'] ?? 0));
        $sku       = sanitize_text_field($product_data['codigo'] ?? '');

        $query = new WP_Query([
            'post_type'      => 'product',
            'posts_per_page' => 1,
            'no_found_rows'  => true,
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'   => '_keygin_id',
                    'value' => $keygin_id,
                ],
            ],
        ]);

        $product_id = $query->posts[0] ?? null;
        wp_reset_postdata();

        if (!$product_id && $sku) {
            $product_id = wc_get_product_id_by_sku($sku);
            if ($product_id) {
                update_post_meta($product_id, '_keygin_id', $keygin_id);
            }
        }

        if (!$product_id) {
            return false;
        }

        $product = wc_get_product($product_id);
        if (!$product || !$product->managing_stock()) {
            return false;
        }

        $product->set_stock_quantity($new_stock);
        $product->save();

        return true;
    }
}