<?php

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__DIR__) . 'api/class-api.php';
require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';

class Keygin_Product_Sync {

    private $keygin_api;
    private $settings;
    private $sync_options;
    private $warehouse_id;
    private $logger;

    public function __construct() {
        $this->settings     = (array) get_option('keygin_settings', []);
        $this->sync_options = isset($this->settings['sync_options']) ? (array) $this->settings['sync_options'] : [];
        $this->warehouse_id = isset($this->settings['warehouse']) ? sanitize_text_field($this->settings['warehouse']) : null;

        $api_key   = isset($this->settings['api_key']) ? sanitize_text_field($this->settings['api_key']) : null;
        $api_token = isset($this->settings['api_token']) ? sanitize_text_field($this->settings['api_token']) : null;

        $this->keygin_api = new Keygin_API($api_key, $api_token);
        $this->logger     = class_exists('Keygin_Logger') ? Keygin_Logger::init() : null;
    }

    /**
     * Synchronize all products from Contifico.
     */
    public function sync_products() {

        if (function_exists('set_time_limit')) {
            @set_time_limit(0);
        }

        $response = $this->keygin_api->get_products();
        $products = (is_array($response) && isset($response['data'])) ? $response['data'] : $response;

        if (!is_array($products) || empty($products)) {
            $this->logger->event(
                'warning',
                __('No products found to synchronize from Contifico.', 'keygin-erp-sync')
            );
            return false;
        }

        /**
         * FREE LIMIT: 50 products
         */
        $FREE_LIMIT = 50;

        if (count($products) > $FREE_LIMIT) {
            $this->logger->event(
                'warning',
                __('Free version product limit reached (50 products).', 'keygin-erp-sync')
            );

            $products = array_slice($products, 0, $FREE_LIMIT);
        }

        $total_processed = 0;
        $chunk_size      = 50;

        foreach (array_chunk($products, $chunk_size) as $chunk_index => $chunk) {

            $this->logger->debug(
                'info',
                sprintf(
                    __('Processing batch %1$d of %2$d (products in batch: %3$d)', 'keygin-erp-sync'),
                    $chunk_index + 1,
                    ceil(count($products) / $chunk_size),
                    count($chunk)
                )
            );

            foreach ($chunk as $product) {
                try {
                    $res = $this->process_product($product);
                    if ($res !== false) {
                        $total_processed++;
                    }
                } catch (Throwable $e) {
                    $this->logger->debug(
                        'error',
                        sprintf(
                            __('Exception while processing product: %s', 'keygin-erp-sync'),
                            $e->getMessage()
                        )
                    );
                }
            }

            if (!defined('DOING_CRON')) {
                usleep(100000);
            }
        }

        $this->logger->debug(
            'success',
            sprintf(
                __('Product synchronization completed. Total processed: %d', 'keygin-erp-sync'),
                $total_processed
            )
        );

        return true;
    }

    /**
     * Process a single product.
     */
    private function process_product($product) {

        if (empty($product['id']) || empty($product['codigo'])) {
            $this->logger->debug(
                'error',
                __('Invalid product received from Contifico (missing ID or SKU).', 'keygin-erp-sync')
            );
            return false;
        }

        $keygin_id = sanitize_text_field($product['id']);
        $sku       = sanitize_text_field($product['codigo']);
        $name      = sanitize_text_field($product['nombre'] ?? __('Unnamed product', 'keygin-erp-sync'));
        $stock_raw = $product['cantidad_stock'] ?? null;
        $stock     = (is_numeric($stock_raw) || $stock_raw === '0') ? intval($stock_raw) : 0;

        $sync = $this->sync_options;

        $sync_name  = isset($sync['nombre']) && $this->truthy($sync['nombre']);
        $sync_stock = isset($sync['cantidad_stock']) && $this->truthy($sync['cantidad_stock']);

        $this->logger->debug(
            'info',
            sprintf(
                __('Processing product Keygin_ID=%1$s, SKU=%2$s', 'keygin-erp-sync'),
                $keygin_id,
                $sku
            )
        );

        $existing_product_id = wc_get_product_id_by_sku($sku);

        // Update existing product
        if ($existing_product_id) {

            $update_data = [];

            if ($sync_name) {
                $update_data['name'] = $name;
            }

            if ($sync_stock) {
                $update_data['stock'] = $stock;
            }

            $update_data['keygin_id'] = $keygin_id;

            if ($this->update_product($existing_product_id, $update_data)) {
                $this->logger->debug(
                    'info',
                    sprintf(
                        __('Product updated: SKU=%1$s, WC_ID=%2$d', 'keygin-erp-sync'),
                        $sku,
                        $existing_product_id
                    )
                );

                do_action('keygin_product_synced', $existing_product_id, $product);
                return $existing_product_id;
            }

            $this->logger->debug(
                'error',
                sprintf(
                    __('Error updating product: SKU=%1$s, WC_ID=%2$d', 'keygin-erp-sync'),
                    $sku,
                    $existing_product_id
                )
            );

            return false;
        }

        // Create new product
        $this->logger->debug(
            'info',
            sprintf(
                __('Creating new product: SKU=%s', 'keygin-erp-sync'),
                $sku
            )
        );

        $new_id = $this->create_product([
            'sku'       => $sku,
            'name'      => $name,
            'stock'     => $stock,
            'keygin_id' => $keygin_id,
        ]);

        if ($new_id) {
            $this->logger->debug(
                'info',
                sprintf(
                    __('Product created: SKU=%1$s, WC_ID=%2$d', 'keygin-erp-sync'),
                    $sku,
                    $new_id
                )
            );

            do_action('keygin_product_created', $new_id, $product);
            return $new_id;
        }

        $this->logger->debug(
            'error',
            sprintf(
                __('Error creating product: SKU=%s', 'keygin-erp-sync'),
                $sku
            )
        );

        return false;
    }

    private function create_product($data) {
        try {

            $sku       = sanitize_text_field($data['sku'] ?? '');
            $name      = sanitize_text_field($data['name'] ?? '');
            $stock     = isset($data['stock']) ? intval($data['stock']) : 0;
            $keygin_id = sanitize_text_field($data['keygin_id'] ?? '');

            if (empty($sku) || empty($name)) {
                throw new Exception(
                    __('Insufficient data to create product (missing SKU or name).', 'keygin-erp-sync')
                );
            }

            if (wc_get_product_id_by_sku($sku)) {
                $this->logger->debug(
                    'info',
                    sprintf(
                        __('Product not created, SKU already exists: %s', 'keygin-erp-sync'),
                        $sku
                    )
                );
                return false;
            }

            $product = new WC_Product_Simple();
            $product->set_name($name);
            $product->set_sku($sku);
            $product->set_manage_stock(true);
            $product->set_stock_quantity($stock);
            $product->set_stock_status($stock > 0 ? 'instock' : 'outofstock');

            $product_id = $product->save();

            if ($product_id && $keygin_id) {
                update_post_meta($product_id, '_keygin_id', $keygin_id);
            }

            return $product_id ?: false;

        } catch (Exception $e) {
            $this->logger->debug(
                'error',
                sprintf(
                    __('Error creating product: %s', 'keygin-erp-sync'),
                    $e->getMessage()
                )
            );
            return false;
        }
    }

    private function update_product($product_id, $data) {
        try {

            $product = wc_get_product($product_id);
            if (!$product) {
                throw new Exception(
                    sprintf(
                        __('Product not found: %d', 'keygin-erp-sync'),
                        $product_id
                    )
                );
            }

            $dirty = false;

            if (isset($data['name'])) {
                $product->set_name(sanitize_text_field($data['name']));
                $dirty = true;
            }

            if (isset($data['stock'])) {
                $product->set_manage_stock(true);
                $product->set_stock_quantity(intval($data['stock']));
                $product->set_stock_status(intval($data['stock']) > 0 ? 'instock' : 'outofstock');
                $dirty = true;
            }

            if ($dirty) {
                $product->save();
            }

            if (!empty($data['keygin_id'])) {
                $existing = get_post_meta($product_id, '_keygin_id', true);
                if ($existing !== $data['keygin_id']) {
                    update_post_meta(
                        $product_id,
                        '_keygin_id',
                        sanitize_text_field($data['keygin_id'])
                    );
                }
            }

            return true;

        } catch (Exception $e) {
            $this->logger->debug(
                'error',
                sprintf(
                    __('Error updating product %1$d: %2$s', 'keygin-erp-sync'),
                    $product_id,
                    $e->getMessage()
                )
            );
            return false;
        }
    }

    private function truthy($val) {
        if (is_bool($val)) return $val;
        if (is_numeric($val)) return intval($val) === 1;
        if (is_string($val)) {
            $v = strtolower($val);
            return in_array($v, ['1', 'true', 'on', 'yes'], true);
        }
        return false;
    }
}