<?php

if (!defined('ABSPATH')) {
    exit;
}

class Keygin_Logger {

    private static $instance = null;
    private $log_dir;

    private $event_log_file;
    private $debug_log_file;

    private function __construct() {

        $upload_dir = wp_upload_dir();

        if (!empty($upload_dir['error'])) {
            $this->write_log(
                null,
                'error',
                sprintf(
                    __('Error resolving wp_upload_dir(): %s', 'keygin-erp-sync'),
                    $upload_dir['error']
                )
            );
            return;
        }

        // Final directory: /wp-content/uploads/keygin-sync-logs/
        $this->log_dir = trailingslashit($upload_dir['basedir']) . 'keygin-sync-logs/';

        if (!file_exists($this->log_dir)) {
            wp_mkdir_p($this->log_dir);
        }

        $date = current_time('Y-m-d');

        $this->event_log_file = $this->log_dir . "keygin-events-{$date}.log";
        $this->debug_log_file = $this->log_dir . "keygin-debug-{$date}.log";

        $this->rotate_logs();
    }

    /** Singleton */
    public static function init() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __clone() {}
    public function __wakeup() {}

    /**
     * Important events (shown in admin panel)
     */
    public function event($type, $message) {
        $this->write_log($this->event_log_file, $type, $message);
    }

    /**
     * Internal debug logs (not shown in panel)
     */
    public function debug($type, $message = null) {

        // Allow calling debug('message')
        if ($message === null) {
            $message = $type;
            $type    = 'debug';
        }

        $this->write_log($this->debug_log_file, $type, $message);
    }

    /**
     * Write a log entry
     */
    private function write_log($file, $type, $message = '') {

        $allowed = ['success', 'warning', 'error', 'info', 'debug'];
        $type = in_array($type, $allowed, true) ? $type : 'info';

        $message = is_scalar($message)
            ? wp_kses_post($message)
            : esc_html(wp_json_encode($message));

        $timestamp = current_time('Y-m-d H:i:s');
        $entry = sprintf("[%s] [%s] %s\n", $timestamp, strtoupper($type), $message);

        if ($file) {
            file_put_contents($file, $entry, FILE_APPEND | LOCK_EX);
        }
    }

    /**
     * Read event logs (shown in admin panel)
     */
    public function get_event_logs($date = null) {

        $file = $date
            ? $this->log_dir . "keygin-events-{$date}.log"
            : $this->event_log_file;

        if (!file_exists($file)) {
            return [];
        }

        $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $parsed = [];

        foreach ($lines as $line) {
            if (preg_match('/^\[(.*?)\]\s+\[(.*?)\]\s+(.*)$/', $line, $m)) {
                $parsed[] = [
                    'date'    => $m[1],
                    'type'    => strtolower($m[2]),
                    'message' => $m[3],
                ];
            }
        }

        return $parsed;
    }

    /**
     * Get available event log files
     */
    public function get_available_event_files() {
        if (!is_dir($this->log_dir)) {
            return [];
        }

        $files = glob($this->log_dir . "keygin-events-*.log");
        return $files ? array_map('basename', $files) : [];
    }

    /**
     * Automatic rotation: delete logs older than 30 days
     */
    private function rotate_logs() {

        if (!is_dir($this->log_dir)) {
            return;
        }

        $files = glob($this->log_dir . "*.log");

        if (!$files) {
            return;
        }

        foreach ($files as $file) {
            if (filemtime($file) < strtotime('-30 days')) {
                wp_delete_file($file);
            }
        }
    }

    /**
     * Clear today's event log
     */
    public function clear_today_logs() {
        file_put_contents($this->event_log_file, '');
        $this->event(
            'info',
            __('Today’s event logs were manually cleared.', 'keygin-erp-sync')
        );
    }
}