<?php
if (!defined('ABSPATH')) exit;

require_once plugin_dir_path(__DIR__) . 'sync/class-product.php';
require_once plugin_dir_path(__DIR__) . 'sync/class-stock.php';
require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';

class Keygin_Scheduler {

    private $settings;
    private $sync_interval;

    /** @var Keygin_Logger|null */
    private $logger;

    public function __construct() {
        $this->settings      = (array) get_option('keygin_settings', []);
        $this->sync_interval = sanitize_text_field($this->settings['frequency'] ?? 'twicedaily');
        $this->logger        = class_exists('Keygin_Logger') ? Keygin_Logger::init() : null;

        $this->init_hooks();
    }

    /**
     * Load required dependencies
     */
    private function load_dependencies() {

        $base_path = plugin_dir_path(dirname(__DIR__));

        $files = [
            'includes/sync/class-product.php',
            'includes/sync/class-stock.php',
            'includes/logs/class-logger.php'
        ];

        foreach ($files as $file) {
            $path = $base_path . $file;

            if (!file_exists($path)) {
                $this->logger?->event(
                    'error',
                    sprintf(
                        esc_html_e('Required file missing: %s', 'keygin-erp-sync'),
                        $file
                    )
                );
                continue;
            }

            require_once $path;
        }

        return true;
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {

        $this->load_dependencies();

        add_filter('cron_schedules', [$this, 'add_custom_intervals']);
        add_action('keygin_sync_event', [$this, 'run_synchronization']);

        if (!class_exists('Keygin_Product_Sync') || !class_exists('Keygin_Stock_Sync')) {
            add_action('admin_notices', [$this, 'show_missing_dependencies_notice']);
        }
    }

    /**
     * Custom cron intervals
     */
    public function add_custom_intervals($schedules) {

        $schedules['keygin_twicedaily'] = [
            'interval' => 12 * HOUR_IN_SECONDS,
            'display'  => esc_html_e('Every 12 hours (Keygin)', 'keygin-erp-sync')
        ];

        $schedules['keygin_daily'] = [
            'interval' => DAY_IN_SECONDS,
            'display'  => esc_html_e('Daily (Keygin)', 'keygin-erp-sync')
        ];

        return $schedules;
    }

    /**
     * Schedule cron events
     */
    public function schedule_events() {

        $settings  = (array) get_option('keygin_settings', []);
        $frequency = sanitize_text_field($settings['frequency'] ?? 'twicedaily');
        $interval  = $this->map_interval($frequency);

        $hook = 'keygin_sync_event';

        // Remove previous schedule
        if ($timestamp = wp_next_scheduled($hook)) {
            wp_unschedule_event($timestamp, $hook);
        }

        wp_schedule_event(time(), $interval, $hook);

        if ($next = wp_next_scheduled($hook)) {
            update_option('keygin_next_sync', $next);
            $this->logger?->event(
                'success',
                sprintf(
                    esc_html_e('Automatic sync scheduled with frequency: %s', 'keygin-erp-sync'),
                    $frequency
                )
            );
        } else {
            $this->logger?->event(
                'error',
                esc_html_e('Failed to schedule automatic synchronization.', 'keygin-erp-sync')
            );
        }
    }

    /**
     * Map frequency to cron interval
     */
    private function map_interval($value) {

        $valid = [
            'twicedaily' => 'keygin_twicedaily',
            'daily'      => 'keygin_daily',
        ];

        return $valid[$value] ?? 'keygin_daily';
    }

    /**
     * Main synchronization process
     */
    public function run_synchronization() {

        $logger = Keygin_Logger::init();
        $start  = microtime(true);

        if (get_transient('keygin_sync_running')) {
            $logger->event(
                'warning',
                esc_html_e('Synchronization skipped: another process is already running.', 'keygin-erp-sync')
            );
            return;
        }

        set_transient('keygin_sync_running', true, 5 * MINUTE_IN_SECONDS);

        if (!get_option('keygin_settings')) {
            $logger->event(
                'error',
                esc_html_e('API not configured. Please complete credentials in settings.', 'keygin-erp-sync')
            );
            delete_transient('keygin_sync_running');
            return;
        }

        try {

            $product_sync = new Keygin_Product_Sync();
            $stock_sync   = new Keygin_Stock_Sync();

            $product_result = $product_sync->sync_products();
            $stock_result   = $stock_sync->sync_stock();

            if ($product_result === false && $stock_result === false) {

                $logger->event(
                    'error',
                    esc_html_e('Product and stock synchronization failed.', 'keygin-erp-sync')
                );

            } elseif ($product_result === false) {

                $logger->event(
                    'warning',
                    esc_html_e('Product sync failed, but stock was updated.', 'keygin-erp-sync')
                );

            } elseif ($stock_result === false) {

                $logger->event(
                    'warning',
                    esc_html_e('Stock sync failed, but products were updated.', 'keygin-erp-sync')
                );

            } else {

                update_option('keygin_last_sync', current_time('mysql'));
                $logger->event(
                    'success',
                    esc_html_e('Automatic synchronization completed successfully.', 'keygin-erp-sync')
                );
            }

        } catch (Throwable $e) {

            $logger->event(
                'error',
                sprintf(
                    esc_html_e('Synchronization exception: %s', 'keygin-erp-sync'),
                    $e->getMessage()
                )
            );

        } finally {

            delete_transient('keygin_sync_running');

            $duration = round(microtime(true) - $start, 2);
            $logger->event(
                'info',
                sprintf(
                    esc_html_e('Total execution time: %ss', 'keygin-erp-sync'),
                    $duration
                )
            );
        }
    }

    /**
     * Clear cron schedule
     */
    public function clear_schedule() {

        $hook = 'keygin_sync_event';

        wp_clear_scheduled_hook($hook);
        delete_option('keygin_next_sync');
    }

    /**
     * Admin notice if dependencies are missing
     */
    public function show_missing_dependencies_notice() {
        echo '<div class="notice notice-error"><p>';
        echo esc_html_e(
            'Keygin Erp Sync requires Keygin_Product_Sync and Keygin_Stock_Sync classes. Please verify the installation.',
            'keygin-erp-sync'
        );
        echo '</p></div>';
    }
}