<?php

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';

class Keygin_API {

    private $logger;
    private $api_key;
    private $api_token;

    /**
     * Base API URL
     * @todo Make endpoint configurable from settings
     */
    private $api_url = 'https://api.contifico.com/sistema/api/v1/';

    public function __construct($api_key = null, $api_token = null) {

        $this->api_key   = sanitize_text_field($api_key);
        $this->api_token = sanitize_text_field($api_token);
        $this->logger    = class_exists('Keygin_Logger') ? Keygin_Logger::init() : null;
    }

    /**
     * Generic HTTP request handler
     */
    private function request($endpoint, $method = 'GET', $data = []) {

        if (empty($this->api_key) || empty($this->api_token)) {
            $this->logger?->debug(
                'error',
                esc_html_e('API credentials are missing.', 'keygin-erp-sync')
            );
            return [
                'success' => false,
                'message' => esc_html_e('Missing API credentials.', 'keygin-erp-sync'),
                'data'    => [],
            ];
        }

        $url = trailingslashit($this->api_url) . ltrim($endpoint, '/');

        $args = [
            'method'  => strtoupper($method),
            'timeout' => 25,
            'headers' => $this->get_auth_headers(),
        ];

        if (!empty($data)) {
            $args['body'] = wp_json_encode($data);
        }

        $max_retries = 3;

        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {

            $response = wp_remote_request($url, $args);

            if (!is_wp_error($response)) {

                $status = wp_remote_retrieve_response_code($response);

                if ($status < 400) {
                    return [
                        'success' => true,
                        'message' => esc_html_e('OK', 'keygin-erp-sync'),
                        'data'    => $this->sanitize_response(
                            wp_remote_retrieve_body($response)
                        ) ?: [],
                    ];
                }

                switch ($status) {

                    case 401:
                        update_option('keygin_sync_enabled', false);
                        $this->logger?->event(
                            'error',
                            sprintf(
                                esc_html_e('Invalid API credentials when accessing endpoint: %s', 'keygin-erp-sync'),
                                esc_html($endpoint)
                            )
                        );
                        return [
                            'success' => false,
                            'message' => esc_html_e('Invalid API credentials.', 'keygin-erp-sync'),
                            'data'    => [],
                        ];

                    case 404:
                        $this->logger?->event(
                            'error',
                            sprintf(
                                esc_html_e('API endpoint not found: %s', 'keygin-erp-sync'),
                                esc_html($endpoint)
                            )
                        );
                        return [
                            'success' => false,
                            'message' => esc_html_e('Endpoint not found.', 'keygin-erp-sync'),
                            'data'    => [],
                        ];

                    case 500:
                        $this->logger?->event(
                            'error',
                            sprintf(
                                esc_html_e('Server error (500) when accessing endpoint: %s', 'keygin-erp-sync'),
                                esc_html($endpoint)
                            )
                        );
                        break;

                    default:
                        $this->logger?->event(
                            'error',
                            sprintf(
                                esc_html_e('HTTP error %d when accessing endpoint: %s', 'keygin-erp-sync'),
                                (int) $status,
                                esc_html($endpoint)
                            )
                        );
                        break;
                }
            }

            if ($attempt < $max_retries) {
                sleep(min(pow(2, $attempt), 4));
            }
        }

        $this->logger?->event(
            'error',
            sprintf(
                esc_html_e('Unable to connect to endpoint %s after %d attempts.', 'keygin-erp-sync'),
                esc_html($endpoint),
                (int) $max_retries
            )
        );

        return [
            'success' => false,
            'message' => esc_html_e('Connection failed.', 'keygin-erp-sync'),
            'data'    => [],
        ];
    }

    /**
     * Retrieve warehouses from Contifico
     */
    public function get_warehouses() {
        return $this->request('bodega/', 'GET');
    }

    /**
     * Test API connection
     */
    public function test_connection() {

        $response = $this->get_warehouses();

        return [
            'success' => !empty($response['data']),
        ];
    }

    /**
     * Sanitize API JSON response
     */
    private function sanitize_response($response) {

        $data = json_decode($response, true);

        return (json_last_error() === JSON_ERROR_NONE && is_array($data))
            ? $data
            : false;
    }

    /**
     * Retrieve products from Contifico
     */
    public function get_products() {

        $response = $this->request('producto/', 'GET');

        if (empty($response['success'])) {
            $this->logger?->debug(
                'error',
                sprintf(
                    esc_html_e('Failed to retrieve products: %s', 'keygin-erp-sync'),
                    sanitize_text_field($response['message'] ?? '')
                )
            );
            return [];
        }

        $products = $response['data'];

        if (empty($products)) {
            $this->logger?->event(
                'info',
                esc_html_e('API returned an empty product list.', 'keygin-erp-sync')
            );
            return [];
        }

        $this->logger?->debug(
            'info',
            sprintf(
                esc_html_e('Products received from API: %d', 'keygin-erp-sync'),
                count($products)
            )
        );

        return $products;
    }

    /**
     * Create inventory output (WooCommerce order)
     */
    public function crear_egreso_inventario($data) {

        if (empty($data['bodega_id']) || empty($data['detalles'])) {
            $this->logger?->debug(
                'error',
                esc_html_e('Inventory output cannot be created: missing required data.', 'keygin-erp-sync')
            );
            return false;
        }

        $response = $this->request('movimiento-inventario/', 'POST', $data);

        if (empty($response['success'])) {
            $this->logger?->debug(
                'error',
                sprintf(
                    esc_html_e('Failed to create inventory output: %s', 'keygin-erp-sync'),
                    sanitize_text_field($response['message'] ?? '')
                )
            );
            return false;
        }

        return true;
    }

    /**
     * Authentication headers
     */
    private function get_auth_headers() {

        return [
            'User-Agent'   => 'Keygin-ERP-Sync/' . KEYGIN_ERP_SYNC_VERSION,
            'Authorization'=> $this->api_key,
            'Api-Token'    => $this->api_token,
            'Accept'       => 'application/json',
            'Content-Type' => 'application/json',
            'Connection'   => 'close',
        ];
    }
}