<?php
/**
 * Keygin Admin Settings Class
 *
 * @package Keygin_ERP_Sync
 */

if (!defined('ABSPATH')) {
    exit;
}

class Keygin_Admin_Settings {

    public function __construct() {}

    public function init() {
        add_action('admin_menu', [$this, 'register_menus']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('wp_ajax_keygin_test_connection', [$this, 'ajax_test_connection']);
        add_action('admin_post_keygin_save_settings', [$this, 'save_settings']);
    }

    public function register_menus() {
        $capability = 'manage_options';
        $slug       = 'keygin-erp-sync';

        add_menu_page(
            esc_html__('Keygin Erp Sync', 'keygin-erp-sync'),
            esc_html__('Keygin Erp Sync', 'keygin-erp-sync'),
            $capability,
            $slug,
            [$this, 'render_settings_page'],
            'dashicons-update-alt',
            26
        );

        add_submenu_page(
            $slug,
            esc_html__('Settings', 'keygin-erp-sync'),
            esc_html__('Settings', 'keygin-erp-sync'),
            $capability,
            'keygin-settings',
            [$this, 'render_credentials_page']
        );
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html_e('Access denied.', 'keygin-erp-sync'));
        }

        require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';
        include plugin_dir_path(__DIR__) . 'views/logs-settings-page.php';
    }

    public function render_credentials_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html_e('Access denied.', 'keygin-erp-sync'));
        }

        $settings   = (array) get_option('keygin_settings', []);
        $warehouses = get_transient('keygin_warehouses');

        if (!is_array($warehouses)) {
            $warehouses = [];
        }

        $GLOBALS['keygin_settings']   = $settings;
        $GLOBALS['keygin_warehouses'] = $warehouses;

        include plugin_dir_path(__DIR__) . 'views/admin-settings-page.php';
    }

    public function enqueue_admin_scripts($hook) {
        wp_enqueue_script(
            'keygin-admin-script',
            plugins_url('assets/js/admin.js', dirname(__FILE__, 2)),
            ['jquery'],
            KEYGIN_ERP_SYNC_VERSION,
            true
        );

        wp_localize_script(
            'keygin-admin-script',
            'keyginAdmin',
            [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('keygin_admin_nonce'),
                'sync_in_progress' => (bool) get_transient('keygin_sync_running'),
                'i18n' => [
                    'testing_connection' => esc_html__('Testing connection...', 'keygin-erp-sync'),
                    'connection_success' => esc_html__('Connection successful.', 'keygin-erp-sync'),
                    'connection_error'   => esc_html__('Connection error.', 'keygin-erp-sync'),
                    'missing_fields'     => esc_html__('Please complete all required fields.', 'keygin-erp-sync'),
                    'unexpected_error'  => esc_html__('Unexpected server communication error.', 'keygin-erp-sync'),
                    'select_warehouse'  => esc_html__('Select a warehouse', 'keygin-erp-sync'),
                    'loading_warehouses'=> esc_html__('Loading warehouses...', 'keygin-erp-sync'),
                    'no_warehouses'     => esc_html__('No warehouses found', 'keygin-erp-sync'),
                    'leave_page_warning'=> esc_html__('Leave page? Changes may not be saved.', 'keygin-erp-sync'),
                    'test_connection_first' => esc_html__('Please test the connection first before saving.', 'keygin-erp-sync'),
                ],
            ]
        );
    }

    public function enqueue_styles($hook) {
        if (strpos($hook, 'keygin') === false) {
            return;
        }

        wp_enqueue_style(
            'keygin-admin-style',
            plugins_url('assets/css/admin.css', dirname(__FILE__, 2)),
            [],
            KEYGIN_ERP_SYNC_VERSION
        );
    }

    public function ajax_test_connection() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            wp_send_json_error(['message' => esc_html_e('Invalid request method.', 'keygin-erp-sync')]);
            exit;
        }

        check_ajax_referer('keygin_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'message' => esc_html_e('You do not have permission.', 'keygin-erp-sync')
            ]);
            exit;
        }

        $api_key   = sanitize_text_field(wp_unslash($_POST['api_key'] ?? ''));
        $api_token = sanitize_text_field(wp_unslash($_POST['api_token'] ?? ''));

        if (empty($api_key) || empty($api_token)) {
            wp_send_json_error([
                'message' => esc_html_e('Missing API credentials.', 'keygin-erp-sync')
            ]);
            exit;
        }

        require_once plugin_dir_path(__DIR__) . 'api/class-api.php';

        $api        = new Keygin_API($api_key, $api_token);
        $connection = $api->test_connection();

        if (empty($connection['success'])) {
            wp_send_json_error([
                'message' => esc_html_e('Error connecting to API.', 'keygin-erp-sync')
            ]);
            exit;
        }

        $response = $api->get_warehouses();

        if (empty($response['success']) || !is_array($response['data'])) {
            wp_send_json_error([
                'message' => esc_html_e('Error fetching warehouses.', 'keygin-erp-sync')
            ]);
            exit;
        }

        $warehouses = array_map(static function ($warehouse) {
            return [
                'id'   => $warehouse['id'] ?? '',
                'name' => $warehouse['nombre'] ?? ('Warehouse ' . ($warehouse['id'] ?? '')),
            ];
        }, $response['data']);

        set_transient('keygin_warehouses', $warehouses, 12 * HOUR_IN_SECONDS);

        wp_send_json_success([
            'message'    => esc_html_e('Connection successful.', 'keygin-erp-sync'),
            'warehouses' => $warehouses
        ]);
        exit;
    }

    public function save_settings() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            wp_die(esc_html_e('Invalid request method.', 'keygin-erp-sync'));
        }

        if (
            !current_user_can('manage_options') ||
            empty($_POST['keygin_nonce']) ||
            !wp_verify_nonce(
                sanitize_text_field(wp_unslash($_POST['keygin_nonce'])),
                'keygin_save_settings'
            )
        ) {
            wp_die(esc_html_e('Access denied.', 'keygin-erp-sync'));
        }

        $settings = (array) get_option('keygin_settings', []);

        $settings['api_key']        = sanitize_text_field(wp_unslash($_POST['keygin_api_key'] ?? ''));
        $settings['api_token']      = sanitize_text_field(wp_unslash($_POST['keygin_api_token'] ?? ''));
        $settings['warehouse']      = sanitize_text_field(wp_unslash($_POST['keygin_warehouse'] ?? ''));
        $settings['warehouse_name'] = sanitize_text_field(wp_unslash($_POST['keygin_warehouse_name'] ?? ''));

        $allowed_frequencies = ['twicedaily', 'daily'];
        $frequency = sanitize_text_field(wp_unslash($_POST['keygin_frequency'] ?? ''));
        $settings['frequency'] = in_array($frequency, $allowed_frequencies, true) ? $frequency : 'daily';

        $sync_options = wp_unslash($_POST['keygin_sync'] ?? []);
        $sync_options['codigo'] = 1;
        $settings['sync_options'] = array_map('sanitize_text_field', (array) $sync_options);

        update_option('keygin_settings', $settings);

        if (class_exists('Keygin_Scheduler')) {
            (new Keygin_Scheduler())->schedule_events();
        }

        require_once plugin_dir_path(__DIR__) . 'logs/class-logger.php';
        Keygin_Logger::init()->event(
            'success',
            esc_html_e('Settings saved successfully.', 'keygin-erp-sync')
        );

        wp_safe_redirect(
            add_query_arg(
                'keygin_status',
                'success',
                admin_url('admin.php?page=keygin-erp-sync')
            )
        );
        exit;
    }
}