<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Adds a product list column with a traffic-light indicator
 * showing whether the product is linked to Keygin (_keygin_id).
 */
class Keygin_Admin_Product_Indicator {

    private static $instance = null;

    public static function init() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {

        // Add column
        add_filter( 'manage_edit-product_columns', [ $this, 'add_product_column' ], 15 );
        add_action( 'manage_product_posts_custom_column', [ $this, 'render_product_column' ], 10, 2 );

        // Load admin assets
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
    }

    public function add_product_column( $columns ) {

        $new_columns = [];

        foreach ( $columns as $key => $label ) {
            $new_columns[ $key ] = $label;

            if ( 'name' === $key ) {
                $new_columns['keygin_monitor'] = esc_html_e( 'Monitoring', 'keygin-erp-sync' );
            }
        }

        if ( ! isset( $new_columns['keygin_monitor'] ) ) {
            $new_columns['keygin_monitor'] = esc_html_e( 'Monitoring', 'keygin-erp-sync' );
        }

        return $new_columns;
    }

    public function render_product_column( $column, $post_id ) {

        if ( 'keygin_monitor' !== $column ) {
            return;
        }

        // Capability check (required by Plugin Review)
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $keygin_id = get_post_meta( $post_id, '_keygin_id', true );

        if ( ! empty( $keygin_id ) ) {
            printf(
                '<span class="keygin-indicator keygin-indicator--ok" title="%s"></span>',
                esc_attr(
                    sprintf(
                        /* translators: %s is the Keygin product ID */
                        esc_html_e( 'Linked to Keygin (ID: %s)', 'keygin-erp-sync' ),
                        $keygin_id
                    )
                )
            );
        } else {
            printf(
                '<span class="keygin-indicator keygin-indicator--nok" title="%s"></span>',
                esc_attr(
                    esc_html_e( 'Not linked to Keygin', 'keygin-erp-sync' )
                )
            );
        }
    }

    /**
     * Enqueue CSS for product indicator column
     */
    public function enqueue_admin_assets( $hook ) {

        if ( 'edit.php' !== $hook ) {
            return;
        }

        $post_type = isset( $_GET['post_type'] )
            ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) )
            : '';

        if ( 'product' !== $post_type ) {
            return;
        }

        // Capability check
        if ( ! current_user_can( 'edit_products' ) ) {
            return;
        }

        // Main admin styles
        wp_enqueue_style(
            'keygin-admin-css',
            KEYGIN_ERP_SYNC_URL . 'assets/css/admin.css',
            [],
            KEYGIN_ERP_SYNC_VERSION
        );

        // Product indicator styles
        wp_enqueue_style(
            'keygin-product-indicator',
            KEYGIN_ERP_SYNC_URL . 'assets/css/product-indicator.css',
            [ 'keygin-admin-css' ],
            KEYGIN_ERP_SYNC_VERSION
        );
    }
}

/**
 * Init indicator after plugins loaded
 */
add_action(
    'plugins_loaded',
    static function () {
        if ( class_exists( 'Keygin_Admin_Product_Indicator' ) ) {
            Keygin_Admin_Product_Indicator::init();
        }
    }
);