<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Keygin_Admin_Notices {

    public function __construct() {
        add_action( 'admin_notices', [ $this, 'check_unlinked_products' ] );
        add_action( 'pre_get_posts', [ $this, 'filter_unlinked_products' ] );
    }

    /**
     * Display admin notice if there are products without _keygin_id
     */
    public function check_unlinked_products() {

        // Ensure admin screen function exists
        if ( ! function_exists( 'get_current_screen' ) ) {
            return;
        }

        $screen = get_current_screen();

        // Show only on product list or WooCommerce settings
        if (
            ! $screen ||
            ! in_array(
                $screen->id,
                [ 'edit-product', 'product', 'woocommerce_page_wc-settings' ],
                true
            )
        ) {
            return;
        }

        // Capability check
        if ( ! current_user_can( 'edit_products' ) ) {
            return;
        }

        global $wpdb;

        // Use cache to avoid frequent DB queries
        $cache_key = 'keygin_unlinked_products';
        $missing   = wp_cache_get( $cache_key, 'keygin' );

        if ( $missing === false ) {

            $meta_key = '_keygin_id';

            // Find published products without _keygin_id
            $query = $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 LEFT JOIN {$wpdb->postmeta} pm
                    ON pm.post_id = p.ID AND pm.meta_key = %s
                 WHERE p.post_type = %s
                   AND p.post_status = %s
                   AND (pm.meta_value IS NULL OR pm.meta_value = '')
                 LIMIT 200",
                $meta_key,
                'product',
                'publish'
            );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $missing = $wpdb->get_col( $query );

            // Cache for 10 minutes
            wp_cache_set(
                $cache_key,
                $missing,
                'keygin',
                10 * MINUTE_IN_SECONDS
            );
        }

        if ( empty( $missing ) ) {
            return;
        }

        $count      = count( $missing );
        $filter_url = admin_url( 'edit.php?post_type=product&keygin_unlinked=1' );
        ?>

        <div class="notice notice-warning is-dismissible keygin-warning">
            <p>
                <strong><?php esc_html_e( 'Keygin ERP Sync:', 'keygin-erp-sync' ); ?></strong>
                <?php
                /* translators: %d is the number of products without _keygin_id */
                printf(
                    esc_html_e(
                        '%d products are not linked (_keygin_id is missing or empty).',
                        'keygin-erp-sync'
                    ),
                    (int) $count
                );
                ?>
            </p>
            <p>
                <a href="<?php echo esc_url( $filter_url ); ?>" class="button button-primary">
                    <?php esc_html_e( 'View unlinked products', 'keygin-erp-sync' ); ?>
                </a>
            </p>
        </div>

        <?php
    }

    /**
     * Filter product list to show only unlinked products
     */
    public function filter_unlinked_products( $query ) {

        if ( ! is_admin() || ! $query->is_main_query() ) {
            return;
        }

        // Capability check
        if ( ! current_user_can( 'edit_products' ) ) {
            return;
        }

        if ( $query->get( 'post_type' ) !== 'product' ) {
            return;
        }

        $flag = isset( $_GET['keygin_unlinked'] )
            ? sanitize_text_field( wp_unslash( $_GET['keygin_unlinked'] ) )
            : '';

        if ( $flag !== '1' ) {
            return;
        }

        // Apply meta query filter
        $query->set(
            'meta_query',
            [
                'relation' => 'OR',
                [
                    'key'     => '_keygin_id',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key'     => '_keygin_id',
                    'value'   => '',
                    'compare' => '=',
                ],
            ]
        );
    }
}