jQuery(document).ready(function ($) {
    const form = $('#keygin-settings-form');
    const saveButton = $('#keygin-save-button');
    const testButton = $('#keygin-test-connection');
    const result = $('#keygin-connection-result');
    const selectWarehouse = $('#keygin_warehouse');
    const warehouseNameInput = $('#keygin_warehouse_name');
    const filters = $('.filter-btn');
    const rows = $('.log-row');

    let formChanged = false;
    let isConnectionValid = false;

    // Disable save button on load
    saveButton.prop('disabled', true);

    // Detect form changes
    form.on('change input', 'input, select, textarea', function () {
        const fieldId = $(this).attr('id');
        formChanged = true;

        // If credentials change, invalidate connection
        if (fieldId === 'keygin_api_key' || fieldId === 'keygin_api_token') {
            isConnectionValid = false;
            saveButton.prop('disabled', true);
            return;
        }

        saveButton.prop('disabled', false);
    });

    // Warn before leaving if there are unsaved changes
    $(window).on('beforeunload', function () {
        if (formChanged) {
            return keyginAdmin.i18n.leave_page_warning;
        }
    });

    // Form submit
    form.on('submit', function (e) {
        const apiKey = $('#keygin_api_key').val();
        const apiToken = $('#keygin_api_token').val();
        const warehouse = $('#keygin_warehouse').val();
        const frequency = $('#keygin_frequency').val();
        
        // Validación mejorada
        const missingFields = [];
        if (!apiKey) missingFields.push('API Key');
        if (!apiToken) missingFields.push('API Token');
        if (!warehouse) missingFields.push('Warehouse');
        if (!frequency) missingFields.push('Frequency');
        
        if (missingFields.length > 0) {
            e.preventDefault();
            
            showValidationNotice('error', keyginAdmin.i18n.missing_fields);
            
            formChanged = false;
            saveButton.prop('disabled', false);
            return;
        }
        
        if (!isConnectionValid) {
            e.preventDefault();
            
            showValidationNotice('warning', keyginAdmin.i18n.test_connection_first);
            return;
        }
        
        formChanged = false;
        saveButton.prop('disabled', false);
    });

    // Test API connection
    testButton.on('click', function () {
        const apiKey = $('#keygin_api_key').val();
        const apiToken = $('#keygin_api_token').val();

        if (!apiKey || !apiToken) {
            result.text(keyginAdmin.i18n.missing_fields).css('color', 'red');
            resetWarehouseSelect();
            isConnectionValid = false;
            saveButton.prop('disabled', true);
            return;
        }

        // Mostrar loading en el botón
        const originalText = testButton.text();
        testButton.prop('disabled', true).text(keyginAdmin.i18n.testing_connection);
        
        result.text(keyginAdmin.i18n.testing_connection).css('color', '#666');
        selectWarehouse.empty().append(
            $('<option>', {
                value: '',
                text: keyginAdmin.i18n.loading_warehouses,
                disabled: true,
                selected: true
            })
        );

        $.ajax({
            url: keyginAdmin.ajax_url,
            type: 'POST',
            data: {
                action: 'keygin_test_connection',
                nonce: keyginAdmin.nonce,
                api_key: apiKey,
                api_token: apiToken
            },
            dataType: 'json'
        })
        .done(function (response) {
            if (response.success) {
                const warehouses = response.data && response.data.warehouses ? response.data.warehouses : [];
                
                result.text(
                    response.data && response.data.message ? response.data.message : keyginAdmin.i18n.connection_success
                ).css('color', 'green');

                isConnectionValid = true;
                formChanged = true;
                saveButton.prop('disabled', false);

                loadWarehouses(warehouses);
            } else {
                const errorMessage = response.data && response.data.message 
                    ? response.data.message 
                    : keyginAdmin.i18n.connection_error;
                handleConnectionError(errorMessage);
            }
        })
        .fail(function (jqXHR, textStatus, errorThrown) {
            handleConnectionError(keyginAdmin.i18n.unexpected_error);
        })
        .always(function() {
            testButton.prop('disabled', false).text(originalText);
        });
    });

    /**
     * Update warehouse name input on selection change
    */

    selectWarehouse.on('change', function () {
        const selectedOption = this.options[this.selectedIndex];
        warehouseNameInput.val(selectedOption.textContent.trim());
    });

    /**
     * Filters
     */
    if (filters.length && rows.length) {
        filters.on('click', function() {
            const type = $(this).data('type') || 'all';
            
            rows.each(function() {
                const $row = $(this);
                if (type === 'all' || $row.hasClass(type)) {
                    $row.show();
                } else {
                    $row.hide();
                }
            });
        });
    }

    /* -------------------------
     * Helper functions
     * ------------------------- */

    function showValidationNotice(type, message) {
        $('.keygin-validation-notice').remove();
        
        const noticeClass = type === 'error' ? 'notice-error' : 'notice-warning';
        const $notice = $('<div>', {
            'class': 'notice ' + noticeClass + ' is-dismissible keygin-validation-notice',
            'css': { 'margin': '10px 0' }
        }).append($('<p>').text(message));
        
        $('.keygin-settings h1').after($notice);
        
        setTimeout(() => {
            $notice.fadeOut(300, function() {
                $(this).remove();
            });
        }, 8000);
    }

    function resetWarehouseSelect() {
        selectWarehouse.empty().append(
            $('<option>', {
                value: '',
                text: keyginAdmin.i18n.select_warehouse,
                disabled: true,
                selected: true
            })
        );
    }

    function loadWarehouses(warehouses) {
        resetWarehouseSelect();

        if (warehouses && warehouses.length > 0) {
            warehouses.forEach(function (warehouse) {
                selectWarehouse.append(
                    $('<option>', {
                        value: warehouse.id,
                        text: warehouse.name
                    })
                );
            });
        } else {
            selectWarehouse.append(
                $('<option>', {
                    value: '',
                    text: keyginAdmin.i18n.no_warehouses,
                    disabled: true
                })
            );
        }
    }

    function handleConnectionError(message) {
        result.text(
            message || keyginAdmin.i18n.connection_error
        ).css('color', 'red');

        isConnectionValid = false;
        saveButton.prop('disabled', true);
        resetWarehouseSelect();
    }
});