<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adiciona página de configurações do plugin
 */
add_action( 'admin_menu', 'kpco_add_settings_page' );
function kpco_add_settings_page() {
	add_options_page(
		esc_html__( 'KeedPay Settings', 'keedpay-checkout-for-woocommerce' ),
		esc_html__( 'KeedPay Checkout', 'keedpay-checkout-for-woocommerce' ),
		'manage_options',
		'kpco-settings',
		'kpco_render_settings_page'
	);
}

/**
 * Renderiza a página de configurações
 */
function kpco_render_settings_page() {
	?>
	<div class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
		<form method="post" action="options.php">
			<?php
			settings_fields( 'kpco_settings' );
			do_settings_sections( 'kpco-settings' );
			submit_button();
			?>
		</form>
	</div>
	<?php
}

/**
 * Registra as configurações
 */
add_action( 'admin_init', 'kpco_register_settings' );
function kpco_register_settings() {
	register_setting( 'kpco_settings', 'kpco_enable_redirect', array(
		'type'              => 'boolean',
		'default'           => true,
		'sanitize_callback' => 'rest_sanitize_boolean',
	) );

	add_settings_section(
		'kpco_main_section',
		esc_html__( 'Redirect Settings', 'keedpay-checkout-for-woocommerce' ),
		'kpco_main_section_callback',
		'kpco-settings'
	);

	add_settings_field(
		'kpco_enable_redirect',
		esc_html__( 'Enable Redirect', 'keedpay-checkout-for-woocommerce' ),
		'kpco_enable_redirect_callback',
		'kpco-settings',
		'kpco_main_section'
	);
}

/**
 * Callback da seção principal
 */
function kpco_main_section_callback() {
	echo '<p>' . esc_html__( 'Configure the automatic redirect to KeedPay custom checkout.', 'keedpay-checkout-for-woocommerce' ) . '</p>';
}

/**
 * Callback do campo de ativação
 */
function kpco_enable_redirect_callback() {
    $enabled = get_option( 'kpco_enable_redirect', true );
    ?>
    <label>
        <input type="checkbox"
               name="kpco_enable_redirect"
               value="1"
               <?php checked( $enabled, true ); ?> />
        <?php esc_html_e( 'Automatically redirect customers to the custom KeedPay checkout', 'keedpay-checkout-for-woocommerce' ); ?>
    </label>
    <p class="description">
        <?php esc_html_e( 'When enabled, customers will be automatically redirected to https://app.keedpay.com.br/checkout when they finish the purchase.', 'keedpay-checkout-for-woocommerce' ); ?>
    </p>
    <?php
}

/**
 * Redireciona para o checkout customizado
 */
add_action( 'template_redirect', 'kpco_redirect_to_custom_checkout' );
function kpco_redirect_to_custom_checkout() {
	if ( ! get_option( 'kpco_enable_redirect', true ) ) {
		return;
	}

	if ( ! is_checkout() || is_wc_endpoint_url( 'order-received' ) ) {
		return;
	}

	if ( ! WC()->cart || WC()->cart->is_empty() ) {
		return;
	}

	$plan_ids = kpco_get_plan_ids_from_cart();
	if ( empty( $plan_ids ) ) {
		wc_add_notice(
    		esc_html__( 'The products in your cart do not have any configured plans. Please contact support.', 
    		'keedpay-checkout-for-woocommerce' ),
    		'error'
		);
		return;
	}

	$base_url = 'https://app.keedpay.com.br/checkout/' . rawurlencode( $plan_ids[0] );

	array_shift( $plan_ids );
	$custom_checkout_url = $base_url;

	if ( ! empty( $plan_ids ) ) {
		$query = http_build_query( array( 'cart_items' => $plan_ids ), '', '&' );
		$query = preg_replace( '/%5B\d+%5D=/', '=', $query );
		$custom_checkout_url = $custom_checkout_url . '?' . $query;
	}

	wp_redirect( $custom_checkout_url );
	exit;
}

/**
 * Obtém todos os plan_ids do carrinho
 */
function kpco_get_plan_ids_from_cart() {
	$cart_items = WC()->cart->get_cart();
	$plan_ids   = array();

	foreach ( $cart_items as $cart_item ) {
		$product  = $cart_item['data'];
		$quantity = isset( $cart_item['quantity'] ) ? intval( $cart_item['quantity'] ) : 1;
		$plan_id  = null;

		if ( $product->is_type( 'variation' ) ) {
			$variation_id = $cart_item['variation_id'];
			$plan_id      = get_post_meta( $variation_id, '_kpco_plan_id', true );
		}

		if ( empty( $plan_id ) ) {
			$product_id = $cart_item['product_id'];
			$plan_id    = get_post_meta( $product_id, '_kpco_plan_id', true );
		}

		if ( !empty( $plan_id ) ) {
			for ( $i = 0; $i < $quantity; $i++ ) {
				$plan_ids[] = $plan_id;
			}
		}
	}

	return $plan_ids;
}

/**
 * Modifica o botão "Finalizar compra"
 */
add_filter( 'woocommerce_get_checkout_url', 'kpco_custom_checkout_url' );
function kpco_custom_checkout_url( $checkout_url ) {
	if ( WC()->cart && ! WC()->cart->is_empty() ) {
		$plan_ids = kpco_get_plan_ids_from_cart();

		if ( empty( $plan_ids ) ) {
			return $checkout_url;
		}

		$base_url = 'https://app.keedpay.com.br/checkout/' . rawurlencode( $plan_ids[0] );

		array_shift( $plan_ids );
		$custom_checkout_url = $base_url;

		if ( ! empty( $plan_ids ) ) {
			$query = http_build_query( array( 'cart_items' => $plan_ids ), '', '&' );
			$query = preg_replace( '/%5B\d+%5D=/', '=', $query );
			$custom_checkout_url = $custom_checkout_url . '?' . $query;
		}

		return esc_url_raw( $custom_checkout_url );
	}

	return $checkout_url;
}
