<?php
/**
 * Plugin Name:       Kainoto Simple RSS Importer
 * Description:       A simple plugin to import posts from an RSS feed file with full date, category, and content cleaning support.
 * Version:           1.7
 * Author:            Kainoto
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Tested up to:      6.8
 */

// Don't allow direct access to the file.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * 1. CREATE THE ADMIN MENU PAGE
 */
function ksri_add_admin_menu() {
    add_menu_page( 'Kainoto RSS Importer', 'Kainoto RSS Importer', 'manage_options', 'kainoto_simple_rss_importer', 'ksri_importer_page_html', 'dashicons-upload', 25 );
}
add_action( 'admin_menu', 'ksri_add_admin_menu' );

/**
 * 2. DISPLAY THE IMPORTER PAGE HTML
 */
function ksri_importer_page_html() {
    if ( ! current_user_can( 'manage_options' ) ) { return; }
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        <p>Upload your RSS/XML feed file to import or update posts. Use the options below to clean the content during import.</p>

        <form action="" method="post" enctype="multipart/form-data">
            <?php wp_nonce_field( 'ksri_import_action', 'ksri_nonce' ); ?>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row"><label for="rss-file">Feed File</label></th>
                        <td><input type="file" id="rss-file" name="rss_file" accept=".xml,.txt" required>
                        <p class="description">Choose the RSS/XML file from your computer.</p></td>
                    </tr>
                    <tr>
                        <th scope="row">Taxonomy Handling</th>
                        <td>
                            <fieldset>
                                <legend class="screen-reader-text"><span>Taxonomy Handling</span></legend>
                                <select name="taxonomy_handling" id="taxonomy-handling">
                                    <option value="none" selected>Do not import categories/tags</option>
                                    <option value="category">Import as Categories</option>
                                    <option value="post_tag">Import as Tags</option>
                                </select>
                                <p class="description">Choose how to handle the <code>&lt;category&gt;</code> fields from the RSS feed.</p>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Content Cleaning</th>
                        <td>
                            <fieldset>
                                <legend class="screen-reader-text"><span>Content Cleaning</span></legend>
                                <p><label for="remove-attributes"><input type="checkbox" name="remove_attributes" id="remove-attributes" value="1" checked><strong>Remove all style and class attributes</strong> from HTML tags.</label></p>
                                <p><label for="tags-to-remove"><strong>Remove Specific Tags</strong> (but keep their text)</label><br>
                                <input type="text" id="tags-to-remove" name="tags_to_remove" class="regular-text" placeholder="span,a,b,p">
                                <p class="description">Enter comma-separated tags to remove completely.</p></p>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Import Options</th>
                        <td>
                            <fieldset>
                                <label for="overwrite-duplicates"><input type="checkbox" name="overwrite_duplicates" id="overwrite-duplicates" value="1"><strong>Overwrite existing posts</strong> if a post with the same title is found.</label>
                                <p class="description">If unchecked, posts with duplicate titles will be skipped.</p>
                            </fieldset>
                        </td>
                    </tr>
                </tbody>
            </table>
            <?php submit_button( 'Import Feed', 'primary', 'ksri_submit' ); ?>
        </form>

        <hr>

        <div id="plugin-disclaimer" style="margin-top: 20px;">
            <a href="https://kainoto.com" target="_blank" rel="noopener">Visit Kainoto.com</a>
            <h2>Disclaimer and Important Information</h2>
            <p>Please read the following points carefully before using this plugin.</p>
            <ol>
                <li><strong>Risk of Content Modification:</strong> This plugin directly interacts with your website's database. If you use the "Overwrite existing posts" option, it will permanently replace content. <strong>Always back up your website before running an import.</strong> The author is not responsible for any unintended content changes or data loss.</li>
                <li><strong>No Guaranteed Support:</strong> This plugin is provided "as-is" without any warranties. We do not promise to provide support or release updates. However, we will do our best to help with issues when time and resources allow.</li>
                <li><strong>Server Constraints:</strong> Importing very large RSS files may fail due to your server's execution time or memory limits. If you encounter errors, we recommend splitting your feed into smaller files and importing them in batches.</li>
                <li><strong>Limitation of Liability:</strong> By using this plugin, you agree that the author is not liable for any damages that may result from its use. You assume full responsibility for using this tool on your system.</li>
            </ol>
        </div>
    </div>
    <?php
}

/**
 * 3. HANDLE THE FILE UPLOAD AND IMPORT PROCESS
 */
function ksri_handle_file_upload() {
    if ( ! isset( $_POST['ksri_submit'] ) || ! current_user_can( 'manage_options' ) ) { return; }

    // Sanitize and validate nonce to satisfy strict linters.
    $nonce = isset( $_POST['ksri_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['ksri_nonce'] ) ) : '';
    if ( ! wp_verify_nonce( $nonce, 'ksri_import_action' ) ) {
        echo '<div class="notice notice-error"><p>Security check failed. Please try again.</p></div>';
        return;
    }

    // Check for file upload errors.
    if ( ! isset( $_FILES['rss_file']['error'] ) || is_array( $_FILES['rss_file']['error'] ) || $_FILES['rss_file']['error'] !== UPLOAD_ERR_OK ) {
        echo '<div class="notice notice-error"><p>Error: No file was uploaded or there was an upload error.</p></div>';
        return;
    }

    // Securely get the temporary file path.
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $file_path = isset( $_FILES['rss_file']['tmp_name'] ) ? wp_unslash( $_FILES['rss_file']['tmp_name'] ) : '';
    if ( ! is_uploaded_file( $file_path ) ) {
        echo '<div class="notice notice-error"><p>Error: Invalid file upload.</p></div>';
        return;
    }

    // Read file content.
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
    $file_content = file_get_contents( $file_path );

    libxml_use_internal_errors( true );
    $xml = simplexml_load_string( $file_content );
    if ( $xml === false ) {
        echo '<div class="notice notice-error"><p>Error: Could not parse XML. Please check file format.</p></div>';
        return;
    }

    $imported_count = 0; $updated_count = 0; $skipped_count = 0;
    
    // Sanitize all POST data.
    $overwrite = isset( $_POST['overwrite_duplicates'] );
    $remove_attributes = isset( $_POST['remove_attributes'] );
    $tags_to_remove_str = isset( $_POST['tags_to_remove'] ) ? sanitize_text_field( wp_unslash( $_POST['tags_to_remove'] ) ) : '';
    $taxonomy_handling = isset( $_POST['taxonomy_handling'] ) ? sanitize_text_field( wp_unslash( $_POST['taxonomy_handling'] ) ) : 'none';

    foreach ( $xml->channel->item as $item ) {
        $title = (string) $item->title;
        $description = (string) $item->description;
        $pubDate = (string) $item->pubDate;

        $clean_content = html_entity_decode( $description, ENT_QUOTES, 'UTF-8' );
        $clean_content = preg_replace( '/\s+/', ' ', $clean_content );
        if ( $remove_attributes ) { $clean_content = preg_replace( '/\s(style|class)=["\'][^"\']*["\']/', '', $clean_content ); }
        if ( ! empty( $tags_to_remove_str ) ) {
            $tags_array = array_map( 'trim', explode( ',', $tags_to_remove_str ) );
            $tags_for_regex = implode( '|', array_filter( $tags_array ) );
            if ( ! empty( $tags_for_regex ) ) { $clean_content = preg_replace( '#</?(' . $tags_for_regex . ')[^>]*>#i', '', $clean_content ); }
        }
        $clean_content = trim( $clean_content );

        $post_date_gmt = gmdate( 'Y-m-d H:i:s', strtotime( $pubDate ) );
        $post_date = get_date_from_gmt( $post_date_gmt );

        // Modern way to check for existing post by title (replaces deprecated get_page_by_title).
        $post_query = new WP_Query( [
            'title' => $title,
            'post_type' => 'post',
            'post_status' => 'any, trash, auto-draft',
            'posts_per_page' => 1,
            'no_found_rows' => true,
        ] );
        
        $existing_post = $post_query->have_posts() ? $post_query->posts[0] : null;

        $post_data = [ 'post_content' => $clean_content, 'post_date' => $post_date, 'post_date_gmt' => $post_date_gmt ];
        $post_id = 0;

        if ( $existing_post ) {
            if ( $overwrite ) {
                $post_data['ID'] = $existing_post->ID;
                wp_update_post( $post_data );
                $post_id = $existing_post->ID;
                $updated_count++;
            } else {
                $skipped_count++;
                continue;
            }
        } else {
            $post_data['post_title'] = wp_strip_all_tags( $title );
            $post_data['post_status'] = 'publish';
            $post_data['post_author'] = get_current_user_id();
            $new_post_id = wp_insert_post( $post_data );
            if ( ! is_wp_error( $new_post_id ) ) {
                $post_id = $new_post_id;
                $imported_count++;
            }
        }

        if ( $post_id > 0 && 'none' !== $taxonomy_handling ) {
            $term_names = [];
            foreach ( $item->category as $category ) {
                $term_names[] = (string) $category;
            }
            if ( ! empty( $term_names ) ) {
                wp_set_post_terms( $post_id, $term_names, $taxonomy_handling, false );
            }
        }
    }

    $message = sprintf(
        '<strong>Import complete!</strong><br>Imported: %s new posts.<br>Updated: %s existing posts.<br>Skipped: %s duplicate posts.',
        esc_html( $imported_count ),
        esc_html( $updated_count ),
        esc_html( $skipped_count )
    );

    echo '<div class="notice notice-success is-dismissible"><p>' . wp_kses_post( $message ) . '</p></div>';
}
add_action( 'admin_init', 'ksri_handle_file_upload' );