<?php
/**
 * Plugin Name: Kailash Recent Post Pages
 * Description: Display recent posts in a responsive grid with featured images and a "Read More" link. Use the shortcode [krpp_shortcode] or enable auto-display from settings.
 * Version:     1.0.0
 * Author:      Kailash Chand
 * Author URI:  https://profiles.wordpress.org/kailash077/
 * Text Domain: kailash-recent-post-pages
 * License:     GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'KRPP_VERSION', '1.0.0' );

/** Default Options */
function krpp_defaults() {
    return array(
        'posts_per_page'        => 6,
        'columns'               => 3,
        'show_image'            => 1,
        'show_readmore'         => 1,
        'readmore_text'         => __( 'Read More', 'kailash-recent-post-pages' ),
        'auto_display_location' => 'none',
        'auto_display_margin'   => '50px',
        'auto_display_bgcolor'  => '#f9f9f9',
        'auto_display_padding'  => '20px',
    );
}

/** Activation Hook */
function krpp_activate() {
    $defaults = krpp_defaults();
    foreach ( $defaults as $key => $val ) {
        $opt_key = 'krpp_' . $key;
        if ( get_option( $opt_key, null ) === null ) {
            add_option( $opt_key, $val );
        }
    }
}
register_activation_hook( __FILE__, 'krpp_activate' );

/** Enqueue Frontend Styles */
function krpp_enqueue_styles() {
    wp_enqueue_style(
        'krpp-style',
        plugin_dir_url( __FILE__ ) . 'style.css',
        array(),
        KRPP_VERSION
    );
}
add_action( 'wp_enqueue_scripts', 'krpp_enqueue_styles' );

/** Enqueue Color Picker for Admin */
function krpp_admin_enqueue_scripts($hook) {
    if ( $hook !== 'settings_page_krpp-settings' ) return;
    wp_enqueue_style( 'wp-color-picker' );
    wp_enqueue_script( 'wp-color-picker' );
}
add_action( 'admin_enqueue_scripts', 'krpp_admin_enqueue_scripts' );

/** Shortcode */
function krpp_shortcode( $atts = array() ) {
    $defaults = array(
        'posts'    => (int) get_option( 'krpp_posts_per_page', 6 ),
        'columns'  => (int) get_option( 'krpp_columns', 3 ),
        'show_image'    => (int) get_option( 'krpp_show_image', 1 ),
        'show_readmore' => (int) get_option( 'krpp_show_readmore', 1 ),
        'readmore_text' => get_option( 'krpp_readmore_text', __( 'Read More', 'kailash-recent-post-pages' ) ),
        'exclude' => 0, // ID to exclude
    );

    $a = shortcode_atts( $defaults, $atts, 'krpp_shortcode' );

    $posts_per_page = max( 1, (int) $a['posts'] );
    $columns        = in_array( (int) $a['columns'], array(1,2,3,4,6), true ) ? (int) $a['columns'] : 3;
    $show_image     = (int) $a['show_image'] ? 1 : 0;
    $show_readmore  = (int) $a['show_readmore'] ? 1 : 0;
    $readmore_text  = sanitize_text_field( $a['readmore_text'] );
    $exclude_id     = (int) $a['exclude'];

    $args = array(
        'post_type'           => 'post',
        'post_status'         => 'publish',
        'posts_per_page'      => $posts_per_page,
        'ignore_sticky_posts' => true,
        'no_found_rows'       => true,
    );

    // Exclude current post if provided or if on single post page
    if ( $exclude_id > 0 ) {
        $args['post__not_in'] = array( $exclude_id );
    } elseif ( is_singular('post') ) {
        $args['post__not_in'] = array( get_the_ID() );
    }

    $q = new WP_Query( $args );

    if ( ! $q->have_posts() ) {
        return '<div class="krpp-recent-posts-grid">' . esc_html__( 'No posts found.', 'kailash-recent-post-pages' ) . '</div>';
    }

    ob_start();
    printf( '<div class="krpp-recent-posts-grid columns-%d">', esc_attr( $columns ) );

    while ( $q->have_posts() ) : $q->the_post();
        echo '<div class="krpp-post-item">';
        if ( $show_image && has_post_thumbnail() ) {
            echo '<a href="' . esc_url( get_permalink() ) . '" rel="noopener">';
            the_post_thumbnail( 'medium', array( 'class' => 'krpp-thumb', 'loading' => 'lazy' ) );
            echo '</a>';
        }
        echo '<h3 class="krpp-title"><a href="' . esc_url( get_permalink() ) . '">' . esc_html( get_the_title() ) . '</a></h3>';

        $excerpt = get_the_excerpt();
        if ( empty( $excerpt ) ) {
            $excerpt = wp_trim_words( wp_strip_all_tags( get_the_content() ), 20 );
        }
        echo '<div class="krpp-excerpt">' . wp_kses_post( $excerpt ) . '</div>';

        if ( $show_readmore ) {
            echo '<a class="krpp-readmore" href="' . esc_url( get_permalink() ) . '">' . esc_html( $readmore_text ) . '</a>';
        }
        echo '</div>';
    endwhile;
    wp_reset_postdata();
    echo '</div>';

    return ob_get_clean();
}
add_shortcode( 'krpp_shortcode', 'krpp_shortcode' );

/** Auto-Display Before Footer with Current Post Exclusion Fix */
function krpp_auto_display_before_footer() {
    if ( is_admin() ) return;

    $location = get_option( 'krpp_auto_display_location', 'none' );
    if ( $location === 'none' ) return;

    static $already_added = false;
    if ( $already_added ) return;
    $already_added = true;

    $show = false;
    if ( $location === 'post' && is_singular('post') ) {
        $show = true;
    } elseif ( $location === 'page' && is_singular('page') ) {
        $show = true;
    } elseif ( $location === 'all' && ( is_singular('post') || is_singular('page') ) ) {
        $show = true;
    }

    if ( $show ) {
        global $wp_query;
        $current_post_id = isset($wp_query->post->ID) ? $wp_query->post->ID : 0;

        $margin_top  = get_option( 'krpp_auto_display_margin', '50px' );
        $bg_color    = get_option( 'krpp_auto_display_bgcolor', '#f9f9f9' );
        $padding     = get_option( 'krpp_auto_display_padding', '20px' );

        echo '<div class="krpp-auto-display-wrapper" style="
                margin-top:'.esc_attr($margin_top).';
                background:'.esc_attr($bg_color).';
                padding:'.esc_attr($padding).';
                border-radius:8px;
            ">';
        echo do_shortcode('[krpp_shortcode exclude="' . esc_attr($current_post_id) . '"]');
        echo '</div>';
    }
}
add_action( 'get_footer', 'krpp_auto_display_before_footer', 10 );

/** Register Settings */
function krpp_register_settings() {
    register_setting( 'krpp_settings', 'krpp_posts_per_page', array( 'type'=>'integer','sanitize_callback'=>'absint','default'=>6 ) );
    register_setting( 'krpp_settings', 'krpp_columns', array( 'type'=>'integer','sanitize_callback'=>'absint','default'=>3 ) );
    register_setting( 'krpp_settings', 'krpp_show_image', array( 'type'=>'integer','sanitize_callback'=>'absint','default'=>1 ) );
    register_setting( 'krpp_settings', 'krpp_show_readmore', array( 'type'=>'integer','sanitize_callback'=>'absint','default'=>1 ) );
    register_setting( 'krpp_settings', 'krpp_readmore_text', array( 'type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__( 'Read More', 'kailash-recent-post-pages' ) ) );
    register_setting( 'krpp_settings', 'krpp_auto_display_location', array( 'type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'none' ) );
    register_setting( 'krpp_settings', 'krpp_auto_display_margin', array( 'type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'50px' ) );
    register_setting( 'krpp_settings', 'krpp_auto_display_bgcolor', array( 'type'=>'string','sanitize_callback'=>'sanitize_hex_color','default'=>'#f9f9f9' ) );
    register_setting( 'krpp_settings', 'krpp_auto_display_padding', array( 'type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'20px' ) );
}
add_action( 'admin_init', 'krpp_register_settings' );

/** Settings Page */
function krpp_add_settings_page() {
    add_options_page(
        __( 'Recent Posts', 'kailash-recent-post-pages' ),
        __( 'Recent Posts', 'kailash-recent-post-pages' ),
        'manage_options',
        'krpp-settings',
        'krpp_render_settings_page'
    );
}
add_action( 'admin_menu', 'krpp_add_settings_page' );

/** Render Settings Page */
function krpp_render_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) return; ?>
    <div class="wrap">
        <h1><?php echo esc_html__( 'Kailash Recent Post Pages – Settings', 'kailash-recent-post-pages' ); ?></h1>
        <form method="post" action="options.php">
            <?php settings_fields( 'krpp_settings' ); ?>
            <table class="form-table">
                <tr>
                    <th><label for="krpp_posts_per_page"><?php esc_html_e( 'Number of posts', 'kailash-recent-post-pages' ); ?></label></th>
                    <td><input type="number" min="1" id="krpp_posts_per_page" name="krpp_posts_per_page" value="<?php echo esc_attr( get_option( 'krpp_posts_per_page', 6 ) ); ?>" /></td>
                </tr>
                <tr>
                    <th><label for="krpp_columns"><?php esc_html_e( 'Number of columns', 'kailash-recent-post-pages' ); ?></label></th>
                    <td>
                        <select id="krpp_columns" name="krpp_columns">
                            <?php
                            $cols = array(1,2,3,4,6);
                            $current = (int) get_option( 'krpp_columns', 3 );
                            foreach ( $cols as $c ) {
                                printf( '<option value="%1$d"%2$s>%1$d</option>', esc_attr( $c ), selected( $current, $c, false ) );
                            }
                            ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Show featured image', 'kailash-recent-post-pages' ); ?></th>
                    <td><label><input type="checkbox" name="krpp_show_image" value="1" <?php checked( 1, get_option( 'krpp_show_image', 1 ) ); ?> /> <?php esc_html_e( 'Enable featured images', 'kailash-recent-post-pages' ); ?></label></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Show "Read More" link', 'kailash-recent-post-pages' ); ?></th>
                    <td><label><input type="checkbox" name="krpp_show_readmore" value="1" <?php checked( 1, get_option( 'krpp_show_readmore', 1 ) ); ?> /> <?php esc_html_e( 'Enable Read More link', 'kailash-recent-post-pages' ); ?></label></td>
                </tr>
                <tr>
                    <th><label for="krpp_readmore_text"><?php esc_html_e( 'Read More text', 'kailash-recent-post-pages' ); ?></label></th>
                    <td><input type="text" id="krpp_readmore_text" name="krpp_readmore_text" value="<?php echo esc_attr( get_option( 'krpp_readmore_text', __( 'Read More', 'kailash-recent-post-pages' ) ) ); ?>" /></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Auto-display location', 'kailash-recent-post-pages' ); ?></th>
                    <td>
                        <?php $location = get_option( 'krpp_auto_display_location', 'none' ); ?>
                        <select name="krpp_auto_display_location">
                            <option value="none" <?php selected( $location, 'none' ); ?>><?php esc_html_e( 'Do not auto-display', 'kailash-recent-post-pages' ); ?></option>
                            <option value="post" <?php selected( $location, 'post' ); ?>><?php esc_html_e( 'Show on Posts', 'kailash-recent-post-pages' ); ?></option>
                            <option value="page" <?php selected( $location, 'page' ); ?>><?php esc_html_e( 'Show on Pages', 'kailash-recent-post-pages' ); ?></option>
                            <option value="all" <?php selected( $location, 'all' ); ?>><?php esc_html_e( 'Show on All (Posts + Pages)', 'kailash-recent-post-pages' ); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th><label for="krpp_auto_display_margin"><?php esc_html_e( 'Spacing above section', 'kailash-recent-post-pages' ); ?></label></th>
                    <td><input type="text" id="krpp_auto_display_margin" name="krpp_auto_display_margin" value="<?php echo esc_attr( get_option( 'krpp_auto_display_margin', '50px' ) ); ?>" /></td>
                </tr>
                <tr>
                    <th><label for="krpp_auto_display_bgcolor"><?php esc_html_e( 'Background color', 'kailash-recent-post-pages' ); ?></label></th>
                    <td><input type="text" id="krpp_auto_display_bgcolor" name="krpp_auto_display_bgcolor" value="<?php echo esc_attr( get_option( 'krpp_auto_display_bgcolor', '#f9f9f9' ) ); ?>" class="krpp-color-field" /></td>
                </tr>
                <tr>
                    <th><label for="krpp_auto_display_padding"><?php esc_html_e( 'Wrapper padding', 'kailash-recent-post-pages' ); ?></label></th>
                    <td><input type="text" id="krpp_auto_display_padding" name="krpp_auto_display_padding" value="<?php echo esc_attr( get_option( 'krpp_auto_display_padding', '20px' ) ); ?>" /></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
        <p><?php echo esc_html__( 'Use the shortcode', 'kailash-recent-post-pages' ); ?> <code>[krpp_shortcode]</code></p>
    </div>
    <script>
    jQuery(document).ready(function($){
        $('.krpp-color-field').wpColorPicker();
    });
    </script>
<?php }