import {useBlockProps, InnerBlocks, RichText} from '@wordpress/block-editor';
import {__} from '@wordpress/i18n';
import {buildDescriptionProperties} from "./tools";

export default function save({attributes}) {
    const blockProps = useBlockProps.save();
    const {
        uniqueId,
        description,
        ariaLabel,
        instructions,
        backgroundColor,
        style,
    } = attributes;

    // Generate a unique ID for descriptions
    const descriptionId = `${uniqueId}--description`;
    const toolbarDescriptionId = `${uniqueId}--toolbar-description`;

    const finalAriaLabel = ariaLabel || __('Before and After comparison', 'juz-dualview');
    const finalInstructions = instructions || __('Move slider or use left and right arrow keys to compare content. Left shows "Before", right shows "After".', 'juz-dualview');

    // Create an object for the main container attributes
    const containerAttributes = {
        className: "juz-dualview",
        id: uniqueId,
        role: "region",
        "aria-label": finalAriaLabel,
    };

    const {descriptionClassName, descriptionStyle} = buildDescriptionProperties({
        backgroundColor,
        style,
        blockProps
    });

    // Only add aria-describedby if there's a description
    if (description) {
        containerAttributes["aria-describedby"] = descriptionId;
    }

    return (
        <div className={blockProps.className}>
            <div {...containerAttributes}>
                <InnerBlocks.Content/>
                <input
                    type="range"
                    className="juz-dualview__range"
                    defaultValue={50}
                    min={0}
                    max={100}
                    step={1}
                    role="slider"
                    aria-label={finalInstructions}
                    aria-valuemin={0}
                    aria-valuemax={0}
                    aria-valuenow={50}
                    aria-valuetext="50%"
                />
            </div>
            {/* Description area */}
            {description && (
                <div id={descriptionId} className={descriptionClassName} style={descriptionStyle}>
                    <RichText.Content
                        tagName="p"
                        value={description}
                        className="juz-dualview__description-text"
                    />
                </div>
            )}
            {/* Hidden description for toolbar relationship */}
            <div id={toolbarDescriptionId} className="juz-dualview__toolbar-description">
                {__('Use the controls below to adjust the comparison view.', 'juz-dualview')}
            </div>
            <div
                className="juz-dualview__toolbar"
                role="toolbar"
                aria-controls={uniqueId}
                aria-label={__('Control options', 'juz-dualview')}
                aria-describedby={toolbarDescriptionId}
            >
                <button
                    className="juz-dualview__show-before"
                    aria-label={__('Show only "Before" content (hides "After" view)', 'juz-dualview')}
                >
                    {__('Show "Before"', 'juz-dualview')}
                </button>
                <button
                    className="juz-dualview__show-after"
                    aria-label={__('Show only "After" content (hides "Before" view)', 'juz-dualview')}
                >
                    {__('Show "After"', 'juz-dualview')}
                </button>
            </div>
        </div>
    );
}
