import {useBlockProps, InnerBlocks, RichText, InspectorControls} from '@wordpress/block-editor';
import {__} from '@wordpress/i18n';
import {useEffect} from '@wordpress/element';
import {useInstanceId} from '@wordpress/compose';
import {TextControl, PanelBody} from '@wordpress/components';
import {buildDescriptionProperties} from "./tools";

export default function Edit({attributes, setAttributes}) {
    const blockProps = useBlockProps();
    const generatedUniqueId = useInstanceId(Edit, "juz-dualview-");
    const {
        uniqueId,
        description,
        ariaLabel,
        instructions,
        backgroundColor,
        style,
        preview
    } = attributes;

    if (preview) {
        return (
            <div className="juz-dualview__preview">
                <img src={juz_dualview_preview.preview_image} alt={__('JuZ DualView Preview', 'juz-dualview')}/>
            </div>
        );
    }

    // Save uniqueId as an attribute if it hasn't been set yet
    useEffect(() => {
        if (!uniqueId) {
            setAttributes({uniqueId: generatedUniqueId});
        }
    }, [uniqueId, generatedUniqueId]);

    // Generate a unique ID for the description
    const descriptionId = `${uniqueId}--description`;

    const descriptionProps = buildDescriptionProperties({
        backgroundColor,
        style,
        blockProps
    });

    const parentBlockProps = {...blockProps};
    delete parentBlockProps.style;

    // Handle description changes
    const onDescriptionChange = (newDescription) => {
        setAttributes({description: newDescription});
    };

    const innerBlocksTemplate = [
        [
            'core/group',
            {
                className: "juz-dualview__before",
                lock: {
                    remove: true,
                    move: true,
                },
                metadata: {
                    name: __('Before', 'juz-dualview')
                }
            },
            [
                [
                    'core/paragraph',
                    {
                        placeholder: __('Add "Before" content here...', 'juz-dualview'),
                    },
                ],
            ],
        ],
        [
            'core/group',
            {
                className: "juz-dualview__after",
                lock: {
                    remove: true,
                    move: true
                },
                metadata: {
                    name: __('After', 'juz-dualview')
                }
            },
            [
                [
                    'core/paragraph',
                    {
                        placeholder: __('Add "After" content here...', 'juz-dualview'),
                    },
                ],
            ],
        ],
    ];

    return (
        <>
            <InspectorControls>
                <PanelBody
                    title={__('JuZ DualView Accessibility Settings', 'juz-dualview')}
                    initialOpen={true}
                >
                    <TextControl
                        label={__('Main accessibility description', 'juz-dualview')}
                        value={ariaLabel}
                        onChange={newAriaLabel => setAttributes({ariaLabel: newAriaLabel})}
                        placeholder={__('Before and After comparison', 'juz-dualview')}
                        help={__(
                            'This label describes the purpose of the before/after comparison for screen reader users. Example: "Website redesign comparison showing old and new versions" or "Product transformation before and after treatment". Leave empty to use the default label.', 'juz-dualview'
                        )}
                    />
                    <TextControl
                        label={__('Slider accessibility description', 'juz-dualview')}
                        value={instructions}
                        onChange={newInstructions => setAttributes({instructions: newInstructions})}
                        placeholder={__('Move slider or use left and right arrow keys to compare content. Left shows "Before", right shows "After".', 'juz-dualview')}
                        help={__(
                            'Instructions for screen reader users on how to use the slider. Example: "Use left and right arrow keys to compare before and after content. Left shows Before view, right shows After view." Leave empty to use the default instructions.', 'juz-dualview'
                        )}
                    />
                </PanelBody>
            </InspectorControls>
            <div {...parentBlockProps}>
                <div
                    className="juz-dualview"
                    role="region"
                    aria-label={ariaLabel}
                    aria-describedby={descriptionId}
                >
                    <InnerBlocks
                        template={innerBlocksTemplate}
                        templateLock={false}
                        allowedBlocks={true}
                    />
                </div>

                {/* Description area */}
                <div id={descriptionId} className={descriptionProps.descriptionClassName}
                     style={descriptionProps.descriptionStyle}>
                    <RichText
                        tagName="p"
                        value={description || __('Add a description for the comparison...', 'juz-dualview')}
                        onChange={onDescriptionChange}
                        placeholder={__('Add a description for the comparison...', 'juz-dualview')}
                        className="juz-dualview__description-text"
                    />
                </div>
            </div>
        </>
    );
}
