<?php
namespace WPJT\Modules;

use WPJT\Classes\JT_Locale;
use WPJT\Classes\JT_Path;
use WPJT\Classes\JT_String;
use WPJT\Helper\JT_Util;

class JT_Translator {
    /**
     * Terjemahkan HTML
     */
    public static function translate_html(string $html, string $str_locale): string {
        $locale = wpjt_get_locale($str_locale);
        
        if($html == "") return $html;
        
        $xml = self::translate_xml($html, $locale->code);
        if(null !== $xml) return $xml;

        $dom = new \DOMDocument();
        libxml_use_internal_errors(true); // suppress warnings for invalid HTML
        @$dom->loadHTML($html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();        
        
        // pastikan HTML lengkap (ada head dan body)
        $xpath = new \DOMXPath($dom);
        if (!(
            ($xpath->query('/html')->length > 0) && 
            ($xpath->query('/html/head')->length > 0) && 
            ($xpath->query('/html/body')->length > 0)
        )) return $html;

        $htmlTag = $dom->getElementsByTagName('html')->item(0);

        if ($htmlTag) {
            $htmlTag->setAttribute('lang', $locale->code);
        }    
        
        /** hreflang for SEO */
        self::generate_hreflang($dom);

        /** translate document title */
        $title_node = $xpath->query('//title')->item(0);
        if ($title_node) {
            /** separator default pakai en dash, sesuai dengan bawaan WordPress */
            $sep = apply_filters('document_title_separator', "\u{2013}"); 

            $title = $title_node->nodeValue;
            $title_parts = explode($sep, $title);
            $title_parts = array_map(function($value) use($locale)  {
                return JT_String::get_instance($value)->value($locale->code) ?? $value;
            }, $title_parts);            
            $title = implode($sep, $title_parts);
            $title_node->nodeValue = $title;
        }        

        // Translate URL
        $nodes = $xpath->query('(
            //a[not(' . self::get_excluded_element() . ')]/@href
            |
            //link[not(' . self::get_excluded_element() . ')]/@href
        )');

        foreach ($nodes as $node) {
            if (trim($node->nodeValue) !== '' && !self::should_skip_string(trim($node->nodeValue))) {
                $node->nodeValue = self::translate_url($node->nodeValue, $locale->code);
            }
        }
        
        // Translate String
        $nodes = $xpath->query('//text()[ not ' . self::get_excluded_element() . ' and normalize-space()]');
        foreach ($nodes as $node) {
            if (trim($node->nodeValue) !== '' && !self::should_skip_string(trim($node->nodeValue))) {
                $node->nodeValue = JT_String::get_instance($node->nodeValue)->value($locale->code) ?? $node->nodeValue;
            }
        }
        
        // Set RTL/LTR
        if($locale->rtl){
            $rtl_nodes = $xpath->query("//*[not(self::html) and not(self::head) and not(self::meta) and not(self::title)]");
            
            /** @var DOMNodeList $rtl_nodes  */
            foreach ($rtl_nodes as $rtl_node) {                                
                $rtl_node->setAttribute('dir', 'rtl');
            }        
        } else {
            $ltr_nodes = $xpath->query("//*[@dir='rtl']");

            /** @var DOMNodeList $ltr_nodes */
            foreach ($ltr_nodes as $ltr_node) {                                
                $ltr_node->setAttribute('dir', 'ltr');
            }
        }
  
        return $dom->saveHTML();
    }

    protected static function generate_hreflang(\DOMDocument &$dom){
        $head = $dom->getElementsByTagName('head')->item(0);
        $path = isset($_SERVER['REQUEST_URI']) ? sanitize_url( wp_unslash($_SERVER['REQUEST_URI']) ) : '';

        $hreflang_tag = function($url, $lang) use ($dom) {
            $link = $dom->createElement('link');
            $link->setAttribute('rel', 'alternate');
            $link->setAttribute('href', $url);
            $link->setAttribute('hreflang', $lang);
            return $link;
        };

        $locales = [];

        // periksa apakah lang codenya lebih dari satu
        $spesific_lang_codes = [];
        foreach (wpjt_get_locale_codes() as $locale_code) {
            $locale = wpjt_get_locale($locale_code);
            $locales[] = $locale;
            $spesific_lang_codes[ $locale->lang ] = isset( $spesific_lang_codes[ $locale->lang ] ) ? true:false;            
        }

        foreach ($locales as $locale) {
            if($locale->is_source) {
                $url = home_url($path);
            } else {
                $url = home_url('/' . $locale->slug .  $path);
            }

            $hreflang = $locale->lang; // defalutnya adalah lang code
            if( $spesific_lang_codes[$locale->lang] ) { //jika lang code lebih dari satu
                $hreflang = $locale->code;
                $hreflang = apply_filters('wpjt_hreflang_code', $hreflang, $locale->code);
            }

            $x_default = $locale->is_source;
            if(apply_filters('wpjt_hreflang_default', $x_default, $locale->code)){
                $x_default_url = $url;
            }

            $head->appendChild( $hreflang_tag($url, $hreflang) ); 
        }

        if(isset($x_default_url)){
            $head->appendChild( $hreflang_tag($x_default_url, 'x-default') ); 
        }
    }

    protected static function get_excluded_element(): string{
        $default_elements = [
            'tag' => [
                'script',
                'style',
                'noscript',
                'title'
            ],
            'id' => [
                'wpadminbar'
            ],
            'class' => [
                'wp-block-comment-content',
                'wpjt-language-switcher'
            ]
        ];

        $elements = apply_filters('wpjt_exclude_elements', $default_elements);

        $excludes = [];
        foreach($elements as $key=>$value){
            if($key == 'tag'){
                foreach($value as $tag){
                    $excludes[] = 'ancestor::' . $tag;
                }                
            }

            if($key == 'id'){
                foreach($value as $id){
                    $excludes[] = 'ancestor::*[@id="' . $id . '"]';
                }                
            }
            
            if($key == 'class'){
                foreach($value as $class){
                    if(is_array($class)){
                        $classes = [];
                        foreach($class as $_class){
                            $classes[]= 'contains(@class, "' . $_class . '")';
                        }

                        if($classes) {
                            $excludes[] = 'ancestor::*[' . implode(' and ', $classes) . ']';
                        }                    
                    }else{
                        $excludes[] = 'ancestor::*[contains(@class, "' . $class . '")]';    
                    }
                    
                }            
            }
        }
        return '(' . implode(' or ', $excludes) . ')';        
    }

    protected static function translate_xml(string $xml, string $locale_code): string|null {
        if ($xml === '') return $xml;
        
        $dom = new \DOMDocument();
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput = false;

        // Hindari warning saat parsing XML
        libxml_use_internal_errors(true);
        $loaded = $dom->loadXML(mb_convert_encoding($xml, 'UTF-8', 'UTF-8'));                
        libxml_clear_errors();
        
        if(!$loaded) return null; // jika bukan XML return false agar dilanjut parsing html
        if(!self::is_sitemap_dom($dom)) return $xml; // untuk XML baru bisa translate sitemap. yang lainnya menyusul

        $locs = $dom->getElementsByTagName('loc');
        foreach ($locs as $loc) {
            $href = trim($loc->nodeValue);
            $loc->nodeValue = self::translate_url($href, $locale_code);
        }

        // Translate href di <a> tag (jika ada, walaupun di XML ini tidak selalu umum)
        $anchors = $dom->getElementsByTagName('a');
        foreach ($anchors as $a) {
            if (strpos($a->getAttribute('class'), 'wpjt-lang-block') !== false) {
                continue;
            }

            $href = $a->getAttribute('href');
            $a->setAttribute('href', self::translate_url($href, $locale_code));
        }

        // Translate semua text node
        $xpath = new \DOMXPath($dom);
        $nodes = $xpath->query('//text()[normalize-space()]');

        foreach ($nodes as $node) {
            $text = trim($node->nodeValue);
            if ($text !== '' && !self::should_skip_string($text)) {
                $node->nodeValue = JT_String::get_instance($node->nodeValue)->value($locale_code);
            }
        }

        return $dom->saveXML();
    }

    /**
     * @param DOMDocument $dom 
     * @return bool 
     */
    protected static function is_sitemap_dom($dom): bool{
        $root = $dom->documentElement;

        if (!$root) return false;

        $root_tag = $root->tagName;
        $namespace = $root->namespaceURI;

        if (
            in_array($root_tag, ['urlset', 'sitemapindex']) &&
            $namespace === 'http://www.sitemaps.org/schemas/sitemap/0.9'
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param string $url 
     * @param string $locale_code 
     * @return string 
     */
    public static function translate_url(string $url, string $locale_code): string{        
        /** saat ini hanya mentranslate internal link
         * untuk translate external link nanti menyusul insya Allah
         */
        if(!JT_Util::is_internal_link($url)){
            return $url;
        }

        $parsed = wp_parse_url($url);
        $path = isset($parsed['path']) ? $parsed['path'] : '' ;

        if(self::should_skip_path($path)) return $url;

        $path = JT_Path::get_instance($path)->value($locale_code);        

        $query = isset($parsed['query']) ? '?' . $parsed['query'] : '';

        return home_url($path). $query ;
    }
 
    /**
     * Skip teks yang tidak perlu diterjemahkan
     * regex: apakah $text hanya mengandung karakter angka dan/atau non-word character
     * @param string $string 
     * @return bool
     */
    private static function should_skip_string(string $string): bool {
        // $is_skip_text = preg_match('/^[\d\W]+$/', $string) || strlen($string) < 3;
        // $is_skip_text = strlen($string) < 3;
        return apply_filters('wpjt_should_skip_string', false, $string);
    }

    /**
     * Skip path yang tidak perlu diterjemahkan
     * @param string $path 
     * @return bool 
     */
    protected static function should_skip_path(string $path): bool{
        $skip_prefixes = [
            '/wp-login.php', 
            '/wp-admin/', 
            '/wp-content/', 
            '/wp-includes/', 
            '/wp-json/'
        ];

        foreach($skip_prefixes as $skip_prefix) {
            if( substr($path, 0, strlen($skip_prefix)) === $skip_prefix ) {
                return apply_filters('wpjt_should_skip_path', true, $path);
            }
        }

        return apply_filters('wpjt_should_skip_path', false, $path);;
    }

    public static function html_translate_http_header(){        
        $active_code = JT_Locale::get_active_code();
        $source_code = JT_Locale::get_source_code();

        $headers = headers_list();
        $header_links = [];
        
        // Kumpulkan semua link yang ada
        foreach ($headers as $header) {
            if (stripos($header, 'Link:') === 0) {
                $links = explode(', ', substr($header, 6));
                foreach ($links as $link) {
                    $header_links[] = $link;
                }
            }
        }
        
        if (empty($header_links)) {
            return false;
        }

        header_remove('Link');
        foreach ($header_links as $link) {
            $parts = explode('; ', $link);
            $uri = trim($parts[0], '<>');

            /** tidak support REST API jadi remove link yang mengandung /wp-json/ */
            if($source_code != $active_code && str_contains($uri, '/wp-json/')) {
                continue;
            }

            $translated_url = self::translate_url($uri, $active_code);
            header('Link: ' . '<' . $translated_url . '>; ' . implode('; ', array_slice($parts, 1)), false);
        }
                
        return true;        
    }
}