<?php
namespace WPJT\Modules;

class JT_Table {
    const TRANSLATION_TABLE = 'wpjt_translations'; 
    const STRING_TABLE      = 'wpjt_strings';     
    const SETTING_TABLE     = 'wpjt_settings';
    const SLUG_SOURCE_TABLE = 'wpjt_slug_source';
    const SLUG_TARGET_TABLE = 'wpjt_slug_target';
    const HTML_CACHE_TABLE  = 'wpjt_html_cache';

    private static function table_schemas(){
        return [
            self::TRANSLATION_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'id'                => 'BIGINT UNSIGNED AUTO_INCREMENT',
                    'string_key'        => 'CHAR(40) NOT NULL', //pakai sha1
                    'source_lang_code'  => 'VARCHAR(10) NOT NULL',
                    'target_lang_code'  => 'VARCHAR(10) NOT NULL',
                    'translation'       => 'TEXT NOT NULL',
                    'created_at'        => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'        => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',                    
                ],
                'indexes' => [
                    'primary' => [
                        'type'      => 'primary',
                        'columns'   => ['id'],
                    ],
                    'unique_translation' => [
                        'type'      => 'unique',
                        'columns'   => ['string_key', 'source_lang_code', 'target_lang_code'],
                    ],
                ],
            ],
            self::STRING_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'id'            => 'BIGINT UNSIGNED AUTO_INCREMENT',
                    'string_key'    => 'CHAR(40) NOT NULL', //pakai sha1
                    'lang_code'     => 'VARCHAR(10) NOT NULL',
                    'string'        => 'TEXT NOT NULL',
                    'created_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',                    
                ],
                'indexes' => [
                    'primary' => [
                        'type'      => 'primary',
                        'columns'   => ['id'],
                    ],
                    'unique_string' => [
                        'type'      => 'unique',
                        'columns'   => ['string_key', 'lang_code'],
                    ],
                ],
            ],
            self::SETTING_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'key_1'         => 'VARCHAR(25) NOT NULL',
                    'key_2'         => 'VARCHAR(25) NOT NULL',
                    'is_array'      => 'TINYINT NOT NULL',
                    's_value'       => 'TEXT NOT NULL',
                    'created_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',                    
                ],
                'indexes' => [
                    'primary' => [
                        'type'      => 'primary',
                        'columns'   => ['key_1', 'key_2'],
                    ],
                ],
            ],
            self::SLUG_SOURCE_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'id'            => 'BIGINT UNSIGNED AUTO_INCREMENT',
                    'slug'          => 'VARCHAR(200) NOT NULL',
                    'lang_code'     => 'VARCHAR(10) NOT NULL',
                    'created_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',                    
                ],
               'indexes' => [
                    'primary' => [
                        'type'      => 'primary',
                        'columns'   => ['id'],
                    ],
                    'unique_slug' => [
                        'type'      => 'unique',
                        'columns'   => ['slug', 'lang_code'],
                    ],
                ],
            ],
            self::SLUG_TARGET_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'id'            => 'BIGINT UNSIGNED AUTO_INCREMENT',
                    'slug'          => 'VARCHAR(200) NOT NULL', // ikut standar WordPress
                    'lang_code'     => 'VARCHAR(10) NOT NULL',
                    'source_id'     => 'BIGINT NOT NULL',
                    'created_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'    => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',
                ],
                'indexes' => [
                    'primary' => [
                        'type' => 'primary',
                        'columns' => ['id'],
                    ],
                    'unique_slug_translation' => [
                        'type'      => 'unique',
                        'columns'   => ['slug', 'lang_code'],
                    ],
                    'unique_slug_lang' => [
                        'type'      => 'unique',
                        'columns'   => ['lang_code', 'source_id'],
                    ],
                ],
            ],
            self::HTML_CACHE_TABLE => [
                'version' => '0.1',
                'fields' => [
                    'uri_key'           => 'CHAR(32) NOT NULL', // pakai md5
                    'source_lang_code'  => 'VARCHAR(10) NOT NULL',
                    'target_lang_code'  => 'VARCHAR(10) NOT NULL',                    
                    'html_key'          => 'CHAR(40) NOT NULL', //pakai sha1
                    'translated_html'   => 'LONGTEXT NOT NULL',
                    'created_at'        => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP',
                    'updated_at'        => 'DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',
                ],
                'indexes' => [                 
                    'primary' => [
                        'type'      => 'primary',
                        'columns'   => ['uri_key', 'source_lang_code', 'target_lang_code'],
                    ],
                    'lookup_key' => [
                        'type'    => 'key',
                        'columns' => ['uri_key', 'source_lang_code', 'target_lang_code', 'html_key'],
                    ],
                ],
            ],            
        ];
    }    

    public static function do_setup(){
        self::setup_tables();        

        if(WPJT_VERSION > JT_Settings::get_plugin_version()){
            JT_Query::set_setting('plugin', 'version', WPJT_VERSION);
        }
    }

    public static function remove_tables(){        
        if (!defined('WPJT_CLEAN_UNINSTALL')) define('WPJT_CLEAN_UNINSTALL', false);

        if( !(JT_Settings::get_clean_uninstall() || WPJT_CLEAN_UNINSTALL) ) return;

        global $wpdb;
        foreach(self::table_schemas() as $table_name=>$schema){
            self::remove_table($wpdb->prefix . $table_name);
        }
    }

    public static function setup_tables(){
        global $wpdb;
        foreach(self::table_schemas() as $table_name=>$schema){
            $installed_version = JT_Query::get_setting('table_version', $table_name) ?? '0.0';

            /** check table version */
            if($schema['version'] > $installed_version) {
                self::sync_table_schema($wpdb->prefix . $table_name, $schema);
                
                /** save table version */
                JT_Query::set_setting('table_version', $table_name, $schema['version'], true);
            }            
        }     
    }    

    private static function remove_table($table_name){
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange --  remove custom table during uninstall
        global $wpdb;
        $wpdb->query($wpdb->prepare("DROP TABLE IF EXISTS %i", $table_name)); 
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
    }    

    private static function sync_table_schema($full_table_name, $schema) {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared --  sync custom table during update/activate the plugin
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$full_table_name'") === $full_table_name;

        if(!$table_exists) {
            /**  Build CREATE TABLE SQL */
            $columns_sql = [];
            foreach ($schema['fields'] as $field => $definition) {
                $columns_sql[] = "`$field` $definition";
            }

            $indexes_sql = [];
            foreach ($schema['indexes'] as $key_name => $index) {
                $cols = implode(', ', array_map(fn($col) => "`$col`", $index['columns']));
                switch (strtolower($index['type'])) {
                    case 'primary':
                        $indexes_sql[] = "PRIMARY KEY ($cols)";
                        break;
                    case 'unique':
                        $indexes_sql[] = "UNIQUE KEY `$key_name` ($cols)";
                        break;
                    case 'key':
                        $indexes_sql[] = "KEY `$key_name` ($cols)";
                        break;
                    case 'fulltext':
                        $indexes_sql[] = "FULLTEXT KEY `$key_name` ($cols)";
                        break;
                }
            }

            $create_sql = "CREATE TABLE $full_table_name (\n  " .
                        implode(",\n  ", array_merge($columns_sql, $indexes_sql)) .
                        "\n) $charset_collate;";

            /** Create table */
            $wpdb->query($wpdb->prepare($create_sql));

            return;
        } 

        /** Get indexes */
        $index_data = $wpdb->get_results("SHOW INDEX FROM $full_table_name");
        $existing_indexes = [];
        foreach ($index_data as $idx) {
            if ($idx->Key_name === 'PRIMARY') {
                $existing_indexes['primary'] = [];
            } else {
                $existing_indexes[$idx->Key_name][] = $idx->Column_name;
            }
        }

        /** Drop indexes that no longer exist in schema 
         * ini harus duluan supaya aman saat ada remove column
        */
        foreach ($existing_indexes as $key_name => $cols) {
            if (!isset($schema['indexes'][$key_name])) {
                $key_clause = ($key_name === 'primary') ? "DROP PRIMARY KEY" : "DROP INDEX `$key_name`";
                $wpdb->query("ALTER TABLE $full_table_name $key_clause");
            }
        }

        /** Get columns */
        $existing_columns = $wpdb->get_col("SHOW COLUMNS FROM $full_table_name", 0);
        
        /** Remove obsolete columns */
        foreach ($existing_columns as $existing_column) {
            if (!isset($schema['fields'][$existing_column])) {
                $wpdb->query("ALTER TABLE $full_table_name DROP COLUMN `$existing_column`");
            }
        }

        /** Add/modify colums */
        $position = "FIRST";
        foreach ($schema['fields'] as $name => $definition) {            
            if (!in_array($name, $existing_columns)) {
                /** add columns */                
                $wpdb->query("ALTER TABLE `$full_table_name` ADD `$name` $definition $position");                
            } else {
                /** simply alter when exist */
                $wpdb->query("ALTER TABLE `$full_table_name` MODIFY `$name` $definition");
            }
            $position = "AFTER `$name`";
        }        

        /** Add indexes from schema if not exists */
        foreach ($schema['indexes'] as $key_name => $index) {
            $cols = implode(', ', array_map(fn($c) => "`$c`", $index['columns']));

            if (!isset($existing_indexes[$key_name])) {
                switch (strtolower(trim($index['type']))) {
                    case 'primary':
                        $wpdb->query("ALTER TABLE $full_table_name ADD PRIMARY KEY ($cols)");
                        break;
                    case 'unique':
                        $wpdb->query("ALTER TABLE $full_table_name ADD UNIQUE KEY `$key_name` ($cols)");
                        break;
                    case 'key':
                        $wpdb->query("ALTER TABLE $full_table_name ADD KEY `$key_name` ($cols)");
                        break;
                    case 'fulltext':
                        $wpdb->query("ALTER TABLE $full_table_name ADD FULLTEXT KEY `$key_name` ($cols)");
                        break;
                }
            }
        }
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
    }
}