<?php
namespace WPJT\Modules;

use WP_Exception;
use WPJT\Classes\JT_Locale;
use WPJT\Modules\JT_Table;

/** @package WPJT\Modules */
class JT_Query {    
    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct query only for custom tables. no direct query to WP tables
    public static function add_string(string $string_key, string $string, ?string $source_locale_code = null) {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching	 --  this is insert operation
        $source_locale = $source_locale_code ? wpjt_get_locale($source_locale_code) : wpjt_get_source_locale();

        if(!$source_locale) return;

        global $wpdb;

        return $wpdb->query($wpdb->prepare(
            "INSERT IGNORE INTO %i (string_key, lang_code, string)
            VALUES (%s, %s, %s)",
            [$wpdb->prefix . JT_Table::STRING_TABLE, $string_key, $source_locale->code, $string]
        ));
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    /**
     * @param string|array $string_key 
     * @param string $source_locale_code 
     * @return void 
     */
    public static function delete_string(string|array $string_key, string $source_locale_code): void {
        global $wpdb;
        
        if(is_array($string_key)) {
            $placeholders = implode(',', array_fill(0, count($string_key), '%s'));
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare --  it's safe, using placeholder for dynamic parameters
            $where_string_key = $wpdb->prepare("string_key IN ($placeholders)", $string_key);
        } else {
            $where_string_key = $wpdb->prepare("string_key = %s", $string_key);
        
        }
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM %i WHERE source_lang_code = %s", [$wpdb->prefix . JT_Table::TRANSLATION_TABLE, $source_locale_code]
        )  . ' AND ' . $where_string_key );
        
        $wpdb->query(
            $wpdb->prepare("DELETE FROM %i WHERE lang_code = %s", [$wpdb->prefix . JT_Table::STRING_TABLE, $source_locale_code]
        ) . ' AND ' . $where_string_key);        

        if(is_array($string_key)) {
            foreach($string_key as $key){
                $cache_key = $key . '|' . $source_locale_code;
                wp_cache_delete($cache_key, 'wpjt_str_registered');
            }
        } else {
            $cache_key = $string_key . '|' . $source_locale_code;
            wp_cache_delete($cache_key, 'wpjt_str_registered');
        }

    }

    /**
     * Get translation string from DB, please don't use this function directily. 
     * This is only for use by the string class (WPJT\Classes\JT_String)
     * @param string $string_key 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @return null|string 
     */
    public static function get_translation_string(string $string_key, string $source_locale_code, string $target_locale_code): ?string {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching	 --  cached at JT_String class (avoid double cache)
        global $wpdb;
        return $wpdb->get_var($wpdb->prepare(
            "SELECT translation FROM %i 
            WHERE string_key = %s AND source_lang_code = %s AND target_lang_code = %s"
            , [$wpdb->prefix . JT_Table::TRANSLATION_TABLE, $string_key, $source_locale_code, $target_locale_code]
        ));
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }     

    /**
     * @param string $string_key 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @param string $translation_string 
     * @return void 
     */
    public static function set_translation_string(string $string_key, string $source_locale_code, string $target_locale_code, string $translation_string ) {
        global $wpdb;

        $translation_string = trim($translation_string);
        
        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO %i (string_key, source_lang_code, target_lang_code, translation)
            VALUES (%s, %s, %s, %s)
            ON DUPLICATE KEY UPDATE translation=%s", 
            [$wpdb->prefix . JT_Table::TRANSLATION_TABLE, $string_key, $source_locale_code, $target_locale_code, $translation_string, $translation_string]
        ));

        $cache_key = $string_key . '|' . $source_locale_code . '|' . $target_locale_code;
        wp_cache_delete($cache_key, 'wpjt_str_translate');

        return $result;
    }



    /**
     * @param string $uri 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @param string $html_key 
     * @return null|string 
     */
    public static function get_html_cache(string $uri, string $source_locale_code, string $target_locale_code, string $html_key): ?string {        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- this is database caching
        global $wpdb;

        $uri_key = md5($uri);
        $html = $wpdb->get_var($wpdb->prepare(
            "SELECT translated_html FROM %i 
            WHERE uri_key = %s AND source_lang_code = %s AND target_lang_code = %s AND html_key = %s"
            , [$wpdb->prefix . JT_Table::HTML_CACHE_TABLE, $uri_key, $source_locale_code, $target_locale_code, $html_key]
        ));

        return $html ? $html: null;
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public static function get_html_cache_key(string $uri, string $source_locale_code, string $target_locale_code): ?string {        
        $uri_key = md5($uri);

        $cache_key = "{$uri_key}|{$source_locale_code}|{$target_locale_code}";
        $html_key = wp_cache_get($cache_key, 'wpjt_html_key');        
        if($html_key === false) {
            global $wpdb;
            $html_key = $wpdb->get_var($wpdb->prepare(
                "SELECT html_key FROM %i 
                WHERE uri_key = %s AND source_lang_code = %s AND target_lang_code = %s"
                , [$wpdb->prefix . JT_Table::HTML_CACHE_TABLE, $uri_key, $source_locale_code, $target_locale_code]
            ));      
            
            if($html_key){
                wp_cache_set($cache_key, $html_key, 'wpjt_html_key');
            }
        }

        return $html_key;
    }

    /**
     * @param string $uri 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @param string $translated_html 
     * @param string $html_key 
     * @return mixed 
     */
    public static function set_html_cache(string $uri, string $source_locale_code, string $target_locale_code, string $translated_html, string $html_key) {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- no caching needed
        global $wpdb;

        $uri_key = md5($uri);
        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO %i (uri_key, html_key, source_lang_code, target_lang_code, translated_html)
            VALUES (%s, %s, %s, %s, %s)
            ON DUPLICATE KEY UPDATE html_key=%s, translated_html=%s ", 
            [$wpdb->prefix . JT_Table::HTML_CACHE_TABLE, $uri_key, $html_key, $source_locale_code, $target_locale_code, $translated_html, $html_key, $translated_html]
        ));

        return $result;
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    public static function clear_html_cache(){
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- no caching needed        
        global $wpdb;
        
        // $result = $wpdb->query($wpdb->prepare(
        //     "TRUNCATE TABLE %i", 
        //     [$wpdb->prefix . JT_Table::HTML_CACHE_TABLE]
        // ));
        
        $result = $wpdb->query($wpdb->prepare(
            "DELETE FROM %i WHERE source_lang_code!=target_lang_code", 
            [$wpdb->prefix . JT_Table::HTML_CACHE_TABLE]
        ));        
        return $result;
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }

    /**
     * @param array $args 
     * @return array 
     */
    public static function get_translation_list(array $args):array{
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- No caching , this is search function, need realtime data
        $source_locale_code = JT_Locale::get_source_code();
        $target_locale_codes = array_filter(JT_Locale::get_codes(), fn($e) => $e !== $source_locale_code);

        $default_args = [
            'per_page' => 20,
            'page' => 1,
            'search' => '',
            'filter_langs' => $target_locale_codes,
            'show_translated' => false,
        ];

        $args = array_merge($default_args, $args);

        global $wpdb;

        if($args['show_translated']) {
            $search_condition = $args['search'] ? $wpdb->prepare(
                " AND (A.string LIKE %s OR B.translation LIKE %s) ", 
                ['%' . $wpdb->esc_like($args['search']) . '%', '%' . $wpdb->esc_like($args['search']) . '%']
            ) : '';
        } else {
            $search_condition = $args['search'] ? $wpdb->prepare(
                " AND A.string LIKE %s", 
                ['%' . $wpdb->esc_like($args['search']) . '%']
            ) : '';

            $search_condition .= " AND IFNULL(B.translation, '') = '' ";
        }

        $unions = [];
        $string_table = $wpdb->prefix . JT_Table::STRING_TABLE;
        $translation_table = $wpdb->prefix . JT_Table::TRANSLATION_TABLE;
        foreach( $args['filter_langs'] as $locale){
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared --  it's safe, using placeholder for dynamic parameters
            $unions[] = $wpdb->prepare(
                "SELECT A.string_key, A.lang_code, A.string , IFNULL(B.target_lang_code, %s) as target_lang_code, IFNULL(B.translation, '') as translation
                FROM %i A 
                LEFT OUTER JOIN %i B
                ON 
                    A.string_key=B.string_key 
                    AND A.lang_code=B.source_lang_code
                    AND B.target_lang_code = %s                    
                WHERE A.lang_code = %s {$search_condition}", 
                [$locale, $string_table, $translation_table, $locale, $source_locale_code]
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        }
        
        if(empty($unions)) return [];

        $temp_table = "{$wpdb->prefix}translation_query";
        $union_tables = "(" . implode(' UNION ', $unions) . ")";

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.SchemaChange -- This is temporary table, to handle complex query
        $wpdb->query($wpdb->prepare("DROP TEMPORARY TABLE IF EXISTS %i", $temp_table)); 

        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared --  false postive
        $wpdb->query($wpdb->prepare(
            "CREATE TEMPORARY TABLE %i ENGINE=MEMORY AS 
            SELECT ROW_NUMBER() OVER (ORDER BY T.string ASC) AS urut,  T.string_key, T.lang_code, T.target_lang_code FROM {$union_tables} T", 
            $temp_table
        )); 
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared --  false postive

        $wpdb->query($wpdb->prepare("ALTER TABLE %i ADD INDEX (string_key)", $temp_table)); 

        $total = $wpdb->get_var($wpdb->prepare("SELECT COUNT(DISTINCT string_key) FROM %i", $temp_table));     

        $offset = ($args['page'] - 1) * $args['per_page'];

        $distinct_results = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT string_key FROM %i             
            ORDER BY urut ASC LIMIT %d, %d",  
            [$temp_table, $offset, $args['per_page']]
        ));

        $keys = [];
        foreach($distinct_results as $row){
            $keys[] = $row->string_key;
        }

        if(empty($keys)) return [];

        $placeholder=implode(',', array_fill(0, count($keys), '%s'));
        // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, WordPress.DB.PreparedSQL.InterpolatedNotPrepared --  it's safe, using dynamic parameters
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT A.string_key, A.lang_code, B.string , A.target_lang_code, IFNULL(C.translation, '') as translation
            FROM %i A 
                INNER JOIN %i B ON A.string_key=B.string_key AND A.lang_code=B.lang_code
                LEFT JOIN %i C ON A.string_key=C.string_key AND A.lang_code=C.source_lang_code and A.target_lang_code=C.target_lang_code
            WHERE A.string_key IN ($placeholder)
            ORDER BY B.string ",  
            [$temp_table, $string_table, $translation_table , ...$keys]
        ));
        // phpcs:enable WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        $strings = [];
        foreach ($results as $row) {
            $strings[$row->string_key]['string'] = $row->string;
            $strings[$row->string_key]['translations'][$row->target_lang_code] = $row->translation;
        }        

        $wpdb->query($wpdb->prepare("DROP TEMPORARY TABLE IF EXISTS %i", $temp_table)); 

        return [
            'total' => $total,
            'per_page' => $args['per_page'],
            'page'  => $args['page'],
            'strings' => $strings,
        ];
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.SchemaChange
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    }    
    /**
     * @param string $key_1 
     * @param string $key_2 
     * @return mixed 
     */
    public static function get_setting(string $key_1, string $key_2){
        global $wpdb;

        $cache_key = "{$key_1}_{$key_2}";
        $setting = wp_cache_get($cache_key, 'wpjt_settings');
        if($setting === false) {
            global $wpdb;
            $row = $wpdb->get_row($wpdb->prepare(
                "SELECT is_array, s_value FROM %i
                WHERE key_1=%s AND key_2=%s", 
                [$wpdb->prefix . JT_Table::SETTING_TABLE, $key_1, $key_2]
            ));             
            
            if($row){
                $setting = $row->is_array ? unserialize($row->s_value) : $row->s_value;
                wp_cache_set($cache_key, $setting, 'wpjt_settings');
            } else {
                $setting = null;
            }
        }

        return $setting;
    }

    /**
     * To save WPJT Setting
     * @param string $key_1 
     * @param string $key_2 
     * @param mixed $value 
     * @param bool $force_update force update when exist, default: true
     * @return void 
     */
    public static function set_setting(string $key_1, string $key_2, $value, bool $force_update = true): void{
        global $wpdb;
        $cache_key = "{$key_1}_{$key_2}";
        $table = $wpdb->prefix . JT_Table::SETTING_TABLE;

        $is_array = is_array($value) ? 1 : 0;

        if($is_array){
            $value = serialize($value);            
        }

        if($force_update) {
            $wpdb->query($wpdb->prepare(
                "INSERT INTO %i (key_1, key_2, is_array, s_value) VALUES (%s, %s, %d, %s)
                ON DUPLICATE KEY UPDATE is_array=%d, s_value=%s", 
                [$table, $key_1, $key_2, $is_array, $value, $is_array, $value]
            )); 
        } else {
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- need realtime data
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM %i WHERE key_1 = %s AND key_2 = %s",
                [$table, $key_1, $key_2]
            ));
            // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching

            if(!$exists) {
                $wpdb->query($wpdb->prepare(
                    "INSERT INTO %i (key_1, key_2, is_array, s_value) VALUES (%s, %s, %d, %s)", 
                    [$table, $key_1, $key_2, $is_array, $value]
                )); 
            }
        }

        wp_cache_delete($cache_key, 'wpjt_settings');
    }

    /**
     * @param string $key_1 
     * @param string $key_2 
     * @return void 
     */
    public static function delete_setting(string $key_1, string $key_2): void {
        global $wpdb;
        $cache_key = "{$key_1}_{$key_2}";
        $table = $wpdb->prefix . JT_Table::SETTING_TABLE;

        $wpdb->query($wpdb->prepare(
            "DELETE %i WHERE key_1=%s AND key_2=%s",
            [$table, $key_1, $key_2]
        )); 

        wp_cache_delete($cache_key, 'wpjt_settings');
    }    

    /** @return void  */
    public static function set_initial_settings(){
        self::save_lang_default();        
    }

    /**
     * SAve default lang settings from json file
     * @return void 
     * @throws WP_Exception 
     */
    private static function save_lang_default(){
        $json_file = WPJT_PATH . 'asset/json/lang-default.json';
        
        if (!file_exists($json_file)) {
            wp_die("Cannot find " . esc_html( $json_file ));
        };

        // Membaca isi file JSON
        $json_data = file_get_contents($json_file);

        // Memeriksa apakah data JSON berhasil dibaca
        if ($json_data === false) {
            wp_die("Error: Unable to read JSON file.");
        }

        $data_array = json_decode($json_data, true);

        // Memeriksa apakah decoding JSON berhasil
        if (json_last_error() !== JSON_ERROR_NONE) {            
            wp_die("Error decoding JSON: " . esc_html(json_last_error_msg()));
        }

        foreach($data_array as $key => $value) {
            // default country
            self::set_setting('def_country', $key, $value['country'], false);

            // rtl
            if($value['rtl']){
                self::set_setting('def_rtl', $key, $value['rtl'], false);
            }
        }
    }

    /**
     * @param string $target_slug 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @return string 
     */
    public static function get_source_slug(string $target_slug, string $source_locale_code, string $target_locale_code): ?string{
        global $wpdb;
        $target_slug = sanitize_title($target_slug);
        $cache_key = "{$target_slug}|{$source_locale_code}|{$target_locale_code}";

        $source_slug = wp_cache_get($cache_key, 'wpjt_source_slug');
        if($source_slug === false) {
            $source_slug = $wpdb->get_var($wpdb->prepare(
                "SELECT source.slug 
                FROM %i target 
                    INNER JOIN %i source ON target.source_id=source.id
                WHERE 
                    target.slug = %s 
                    AND target.lang_code = %s 
                    AND source.lang_code = %s", 
                [
                    $wpdb->prefix . JT_Table::SLUG_TARGET_TABLE,
                    $wpdb->prefix . JT_Table::SLUG_SOURCE_TABLE,                    
                    $target_slug,
                    $target_locale_code,
                    $source_locale_code,            
                ]
            ));

            if(!$source_slug) return null;

            wp_cache_set($cache_key, $source_slug, 'wpjt_source_slug');
        }

        return $source_slug;        
    }

    /**
     * @param string $slug 
     * @param mixed $source_locale_code 
     * @return mixed 
     */
    public static function get_source_slug_id(string $slug, $source_locale_code = null) {
        global $wpdb;
                
        $source_locale_code = $source_locale_code ?? JT_Locale::get_source_code();

        $cache_key = $source_locale_code . '|' . $slug;

        $slug_id = wp_cache_get($cache_key, 'wpjt_slug_id');

        if($slug_id === false) {
            $slug_id = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM %i
                WHERE slug = %s AND lang_code = %s", 
                [$wpdb->prefix . JT_Table::SLUG_SOURCE_TABLE, $slug, $source_locale_code]
            ));

            if(is_null($slug_id)){
                $wpdb->insert($wpdb->prefix . JT_Table::SLUG_SOURCE_TABLE, [
                    'slug' => $slug,
                    'lang_code' => $source_locale_code,
                ]);
            }

            wp_cache_set($cache_key, $slug_id, 'wpjt_slug_id');
        }

        return $slug_id;
    }

    /**
     * @param string $source_slug 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @return string 
     */
    public static function get_target_slug(string $source_slug, string $source_locale_code, string $target_locale_code): ?string{
        global $wpdb;
        // pakai ini supaya auto register slug
        $source_id = self::get_source_slug_id($source_slug, $source_locale_code);
        $cache_key = "{$source_slug}|{$source_locale_code}|{$target_locale_code}";
        $target_slug = wp_cache_get($cache_key, 'wpjt_target_slug');
        if($target_slug === false) {
            $target_slug = $wpdb->get_var($wpdb->prepare(
                "SELECT slug FROM %i WHERE lang_code = %s and source_id=%d",
                [$wpdb->prefix . JT_Table::SLUG_TARGET_TABLE, $target_locale_code, $source_id]
            ));

            if($target_slug){
                wp_cache_set($cache_key, $target_slug, 'wpjt_target_slug');
            } else {
                $target_slug = null;
            }
        }
        return $target_slug;        
    } 

    /**
     * @param string $source_slug 
     * @param string $source_locale_code 
     * @param string $target_locale_code 
     * @param string $target_slug 
     * @return void 
     */
    public static function set_target_slug(string $source_slug, string $source_locale_code, string $target_locale_code, string $target_slug ): void{
        global $wpdb;
        $source_id = self::get_source_slug_id($source_slug, $source_locale_code);
        $target_slug = sanitize_title($target_slug);
         // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- false postive -> insert query
        $wpdb->query($wpdb->prepare(
            "INSERT INTO %i (slug, lang_code, source_id)
            VALUES (%s, %s, %s)
            ON DUPLICATE KEY UPDATE slug=%s", 
            [$wpdb->prefix . JT_Table::SLUG_TARGET_TABLE, $target_slug, $target_locale_code, $source_id, $target_slug]
        ));
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching

        $cache_key = "{$target_slug}|{$source_locale_code}|{$target_locale_code}";
        wp_cache_set($cache_key, $source_slug, 'wpjt_source_slug');
    }    
    // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
}