<?php
namespace WPJT\Modules;

use WPJT\Classes\JT_Locale;

/** @package WPJT\Modules */
class JT_Cache {    
    protected static string $device_type;
    protected static function get_cache_dir_html_key(string $uri_key){
        $dir = WPJT_CACHE_DIR . "html-keys/{$uri_key}/";
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
        return $dir;        
    }

    protected static function get_cache_dir_translated_html(string $uri_key, string $locale_code){
        $dir = WPJT_CACHE_DIR . "translated/{$locale_code}/{$uri_key}/";
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);            
        }
        return $dir;        
    }    

    protected static function set_meta(string $uri, $html_key){
        $uri_key = self::get_uri_key($uri);
        $device = self::get_device_type();
        
        $key_file = self::get_cache_dir_html_key($uri_key) . "{$device}.json";
        $data = [
            'html_key' => $html_key,
            'headers' => headers_list(),
        ];
        file_put_contents($key_file, json_encode($data));
    }
    
    public static function get_meta(string $uri){
        $uri_key = self::get_uri_key($uri);
        $device = self::get_device_type();

        $key_file = self::get_cache_dir_html_key($uri_key) . "{$device}.json";

        if (file_exists($key_file)) {    
            return json_decode(file_get_contents($key_file), true) ;
        }

        return null;
    }

    public static function get_html_cache(string $uri, string $locale_code, string $html_key){
        $uri_key = self::get_uri_key($uri);        
        $device = self::get_device_type();

        $cache_file = self::get_cache_dir_translated_html($uri_key, $locale_code) . "{$html_key}-{$device}.html";
        
        if (file_exists($cache_file)) { 
            $ftime = filemtime($cache_file);
            $etag = '"' . filesize($cache_file) . '-' . $ftime . '"';

            if (isset($_SERVER['HTTP_IF_NONE_MATCH']) && (sanitize_text_field(wp_unslash($_SERVER['HTTP_IF_NONE_MATCH'])) === $etag)) {
                http_response_code(304); // Not Modified
                exit;
            }             

            header('Content-Language: ' . $locale_code);       
            header('ETag: ' . $etag);
            header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $ftime) . ' GMT');

            return file_get_contents($cache_file);
        }

        return false;
    }

    public static function set_html_cache(string $uri, string $locale_code, string $html, string $html_key){
        $uri_key = self::get_uri_key($uri);
        $device = self::get_device_type();

        self::set_meta($uri, $html_key);

        $cache_dir = self::get_cache_dir_translated_html($uri_key, $locale_code);         
        self::clear_glob($cache_dir . "*-{$device}.html");
        
        $cache_file = $cache_dir . "{$html_key}-{$device}.html";
        file_put_contents($cache_file, $html);

        if($locale_code !== JT_Locale::get_source_code()) {
            header('X-Translated-Cache: MISS');
        }
    }

    protected static function get_uri_key($uri){
        return md5($uri . '|' . JT_Locale::get_source_code());
        // return JT_Locale::get_source()->slug . $uri ;
    }

    protected static function get_device_type() {
        $device_type = function() {
            $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? strtolower( sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) ) : '';
            if (preg_match('/ipad|tablet|(android(?!.*mobile))|kindle|silk|playbook|nexus 7|nexus 10|xoom/', $ua)) {
                return 'tablet';
            }
            if (preg_match('/mobile|iphone|ipod|android.*mobile|blackberry|iemobile|opera mini|windows phone/', $ua)) {
                return 'mobile';
            }
            return 'desktop';
        };

        if(!isset(self::$device_type)) {
            self::$device_type = $device_type();
        }

        return self::$device_type;
    }       

    public static function clear_translated(){
        global $wp_filesystem;
        
        if ( ! function_exists( '\WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        \WP_Filesystem();

        $dir = WPJT_CACHE_DIR . "translated/";
        $source_code = JT_Locale::get_source_code();

        if (!$wp_filesystem->is_dir($dir)) {
            return false;
        }

        $items = $wp_filesystem->dirlist( $dir );
        foreach ($items as $name => $info) {
            if ($info['type'] ==='dir' && $name === $source_code) {
                continue;
            }

            $wp_filesystem->delete( $dir . DIRECTORY_SEPARATOR . $name, true );
        }
    }

    public static function clear_all(){
        global $wp_filesystem;

        if ( ! function_exists( '\WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        \WP_Filesystem();

        $wp_filesystem->delete(WPJT_CACHE_DIR, true);
    }

    protected static function clear_glob($glob_dir){        
        global $wp_filesystem;
        
        if ( ! function_exists( '\WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        \WP_Filesystem();

        foreach (glob($glob_dir) as $file) {
            $wp_filesystem->delete($file);
        }
    }
}
