<?php
namespace WPJT\Helper;

use WPJT\Classes\JT_Locale;
use WPJT\Classes\JT_Path;
use WPJT\Modules\JT_Translator;

/** @package WPJT\Helper */
class JT_Util {

    public static function get_locale_time_stamp(string $date_str, ?string $locale_code=null, ?string $pattern=null) {        
        $locale_code = $locale_code ?? JT_Locale::get_source_code();
        $pattern = $pattern ?? self::wp_date_format_to_icu(get_option('date_format'));

        $fmt = new \IntlDateFormatter($locale_code, \IntlDateFormatter::NONE, \IntlDateFormatter::NONE);
        $fmt->setPattern($pattern);        
        
        return $fmt->parse($date_str);
    }    

    public static function wp_date_format_to_icu($php_format) {
        $map = [
            // Hari
            'd' => 'dd',
            'j' => 'd',
            'D' => 'EEE',
            'l' => 'EEEE',
            'N' => 'e',      // ISO day (1-7) — bukan padanan penuh
            'w' => 'c',      // 0 (Minggu) - 6 (Sabtu)
            'z' => 'D',      // Hari ke-berapa dalam tahun — mendekati

            // Minggu
            'W' => 'w',      // Nomor minggu — tidak identik

            // Bulan
            'F' => 'MMMM',
            'm' => 'MM',
            'M' => 'MMM',
            'n' => 'M',
            't' => '',       // Jumlah hari dalam bulan — tidak tersedia

            // Tahun
            'L' => '',       // Tahun kabisat — tidak tersedia
            'o' => 'Y',      // ISO year number — mendekati
            'Y' => 'yyyy',
            'y' => 'yy',

            // Waktu
            'a' => 'a',
            'A' => 'a',
            'g' => 'h',
            'G' => 'H',
            'h' => 'hh',
            'H' => 'HH',
            'i' => 'mm',
            's' => 'ss',
            'u' => 'SSS',    // Microseconds (tidak penuh)

            // Zona waktu
            'e' => 'VV',     // Timezone ID
            'T' => 'z',      // Abbrev
            'P' => 'xxx',
            'O' => 'xx',
            'Z' => '',       // Offset in seconds — tidak ada

            // Full format
            'c' => 'yyyy-MM-dd\'T\'HH:mm:ssxxx',
            'r' => 'EEE, dd MMM yyyy HH:mm:ss Z',
            'U' => 'X',
        ];

        $icu = '';
        $escaping = false;

        $len = strlen($php_format);
        for ($i = 0; $i < $len; $i++) {
            $char = $php_format[$i];

            // Escape literal
            if (!ctype_alpha($char)) {
                if (!$escaping) {
                    $icu .= "'";
                    $escaping = true;
                }
                if ($char === "'") {
                    $icu .= "''";
                } else {
                    $icu .= $char;
                }
                continue;
            }

            if ($escaping) {
                $icu .= "'";
                $escaping = false;
            }

            // Translate format
            $icu .= isset($map[$char]) ? $map[$char] : $char;
        }

        if ($escaping) {
            $icu .= "'";
        }

        return $icu;
    }    

    /**
     * @param array $args 
     * @return string|false 
     * @throws WP_Exception 
     */
    public static function get_language_switcher(array $args = []){
        $default_args = [
            'hide_flag' => false,
            'show_name' => false,
            'display_name' => 'lang_native_name', // name or locale
        ];
        
        $args = array_merge($default_args, $args);

        $allowed_name_key = ['locale', 'lang', 'name', 'native_name', 'english_name', 'lang_name', 'lang_native_name', 'lang_english_name'];
        $name_key = in_array($args['display_name'], $allowed_name_key) ? $args['display_name'] : 'lang_native_name';
        
        $path = isset($_SERVER['REQUEST_URI']) ? sanitize_url( wp_unslash($_SERVER['REQUEST_URI']) ) : '';
        
        wp_enqueue_style('wpjt_language_switcher');

        
        /** parse current URL */
        $parsed = wp_parse_url( JT_Util::get_current_url() );
        $query = isset($parsed['query']) ? $parsed['query'] : '';
        $o_path = JT_Path::get_instance( isset($parsed['path']) ? $parsed['path'] : '' );

        ob_start();
        ?>
        <div class="wpjt-language-switcher">
            <?php 
            foreach ( JT_Locale::get_codes() as $locale_code) {                 
                $locale_detail = wpjt_get_locale($locale_code);
                $url = home_url($o_path->value($locale_detail->code) . $query);

                // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage --  dynamic image based on locale
                $flag = $args['hide_flag'] ? '' : "<img class='wpjt-lang-flag' src='{$locale_detail->flag}' alt='{$locale_detail->name}'>";
                $name = $args['show_name'] ? $locale_detail->$name_key : '';
                $title = $args['show_name'] ? '' : "title='{$locale_detail->$name_key}'";

                if($name){
                    $flag = $flag . '<span>' . $name . '</span>';
                }
                
                if ($locale_detail->is_active) {
                    ?>
                    <div class="wpjt-language wpjt-current" <?php echo wp_kses_post($title); ?>><div class="wpjt-lang-block" > <?php echo wp_kses_post($flag); ?> </div></div>
                    <?php 
                } else {
                    ?>
                    <div class="wpjt-language" <?php echo wp_kses_post($title); ?>><a class="wpjt-lang-block" href="<?php echo esc_url($url); ?>"> <?php echo wp_kses_post($flag); ?> </a></div>
                    <?php 
                }

            } ?>
        </div>
        <?php
        return ob_get_clean();
    }      

    public static function get_language_switcher_x(array $args = []){
        $default_args = [
            'hide_flag' => false,
            'show_name' => false,
            'display_name' => 'lang_native_name', // name or locale
        ];
        
        $args = array_merge($default_args, $args);

        $allowed_name_key = ['locale', 'lang', 'name', 'native_name', 'english_name', 'lang_name', 'lang_native_name', 'lang_english_name'];
        $name_key = in_array($args['display_name'], $allowed_name_key) ? $args['display_name'] : 'lang_native_name';
        
        $path = isset($_SERVER['REQUEST_URI']) ? sanitize_url( wp_unslash($_SERVER['REQUEST_URI']) ) : '';
        
        wp_enqueue_style('wpjt_language_switcher');
        
        ob_start();
        ?>
        <div class="wpjt-language-switcher">
            <?php 
            foreach ( JT_Locale::get_codes() as $locale_code) {                 
                $locale_detail = wpjt_get_locale($locale_code);

                if($locale_detail->is_source) {
                    $url = home_url($path);
                    // $url = JT_Translator::translate_url();
                } else {
                    $url = home_url('/' . $locale_detail->slug .  $path);
                }
                // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage --  dynamic image based on locale
                $flag = $args['hide_flag'] ? '' : "<img class='wpjt-lang-flag' src='{$locale_detail->flag}' alt='{$locale_detail->name}'>";
                $name = $args['show_name'] ? $locale_detail->$name_key : '';
                $title = $args['show_name'] ? '' : "title='{$locale_detail->$name_key}'";

                if($name){
                    $flag = $flag . '<span>' . $name . '</span>';
                }
                
                if ($locale_detail->is_active) {
                    ?>
                    <div class="wpjt-language wpjt-current" <?php echo wp_kses_post($title); ?>><div class="wpjt-lang-block" > <?php echo wp_kses_post($flag); ?> </div></div>
                    <?php 
                } else {
                    ?>
                    <div class="wpjt-language" <?php echo wp_kses_post($title); ?>><a class="wpjt-lang-block" href="<?php echo esc_url($url); ?>"> <?php echo wp_kses_post($flag); ?> </a></div>
                    <?php 
                }

            } ?>
        </div>
        <?php
        return ob_get_clean();
    }    

    public static function get_current_url(){
        $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http");
        $url .= '://' . isset($_SERVER['HTTP_HOST']) ? sanitize_url( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
        $url .= isset($_SERVER['REQUEST_URI']) ? sanitize_url( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        return $url;
    }

    public static function is_internal_link( $link ) {        
        $parsed = wp_parse_url(strtolower( $link ) );
        if(array_key_exists('scheme', $parsed) && array_key_exists('host', $parsed)){
            /** Ikut logic WP
             * ref: https://developer.wordpress.org/reference/functions/wp_is_internal_link/
             */
            return in_array( $parsed['host'], wp_internal_hosts(), true );
        } else {
            /** jika scheme dan host kosong anggap internal link */
            return true;
        }
    }   

    public static function debug($var, bool $exit=false){
        // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_print_r  --  used only on development mode
        if(!defined('WPJT_DEV_MODE')) define('WPJT_DEV_MODE', false);
        if(!WPJT_DEV_MODE) return;        
        echo '<pre>';        
        print_r($var);
        echo '</pre>';
        if($exit) die();
        // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_print_r
    }

    public static function get_device_type() {
        $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        if (preg_match('/ipad|tablet|(android(?!.*mobile))|kindle|silk|playbook|nexus 7|nexus 10|xoom/', $ua)) {
            return 'tablet';
        }
        if (preg_match('/mobile|iphone|ipod|android.*mobile|blackberry|iemobile|opera mini|windows phone/', $ua)) {
            return 'mobile';
        }
        return 'desktop';
    }    
}