<?php
namespace WPJT\Classes;

use WPJT\Helper\JT_Util;
use WPJT\Modules\JT_Query;

/** @package WPJT\Classes */
class JT_String {
    public readonly bool $is_date;

    protected readonly string $key, $string, $locale_code;
    protected readonly int $lspc, $rspc;    
    protected readonly int|bool $timestamp;    
    protected readonly array $args;

    protected array $translated_strings;
    protected array $translateds;

    public function __construct(string $string, array $args=[]) {        
        if(trim($string) === '') return;

        $this->args = $args;        
        $args = array_merge([
            'locale_code' => JT_Locale::get_source_code(),
            'post_id'     => null,
            'path'        => null,
            'group'       => null,
        ], $this->args);        

        $this->string = $string;

        $locale_code = $args['locale_code'] ?? JT_Locale::get_source_code();

        $locale = wpjt_get_locale($locale_code);        
        if($locale){
            $this->locale_code = $locale->code;
            $this->key = sha1(trim($string));

            // Simpan jumlah spasi depan-belakang (kita tidak mentranslate space depan dan belakang)
            $this->lspc = strlen($string) - strlen(ltrim($string, ' '));
            $this->rspc = strlen($string) - strlen(rtrim($string, ' '));
            
            if(class_exists('\IntlDateFormatter')){
                $this->timestamp = JT_Util::get_locale_time_stamp(trim($string));
                $this->is_date = ($this->timestamp !== false); 
            } else {
                $this->is_date = false;
            }

            /** Tanggal tidak perlu diregister
             * saat sedang debug display juga tidak perlu register, agar error message tidak masuk ke strings
             */                         
            $debug = defined('WP_DEBUG') && WP_DEBUG;
            $debug_display = defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY;
            if ( !$this->is_date && !($debug && $debug_display) ) {
                $this->register();
            }                        
        }
    }

    /**
     * Get instance from cache
     */
    public static function get_instance(string $string, array $args=[]): ?self {
        $str_args = empty($args) ? '' : json_encode($args);
        $cache_key = sha1(trim($string) . $str_args);
        
        $me = wp_cache_get($cache_key, 'wpjt_string_instance');
        if(false === $me) {
            try {
                $me = new self($string, $args);
                wp_cache_set($cache_key, $me, 'wpjt_string_instance');
            } catch (\InvalidArgumentException $e) {
                $me = null; // Set ke null jika gagal
            } 
        }
        return $me;
    }

    public function register(){
        if(!isset($this->key)) return;

        $cache_key = $this->key . '|' . $this->locale_code;
        $registered = wp_cache_get($cache_key, 'wpjt_str_registered');
        if($registered === false) {
            $registered = JT_Query::add_string($this->key, trim($this->string), $this->locale_code);
            wp_cache_set($cache_key, $registered, 'wpjt_str_registered');
        }

        return $registered;
    }

    public function is_translated(string $target_locale_code): bool{
        if(!isset($this->key)) return false;

        $target_locale = wpjt_get_locale($target_locale_code ?? $this->locale_code);
        if(!$target_locale) return false;

        if(!isset($this->translateds[$target_locale->code])) {
            $cache_key = $this->key . '|' . $this->locale_code . '|' . $target_locale->code;
            $translated = wp_cache_get($cache_key, 'wpjt_str_translate');

            if($translated === false) {
                $translated = JT_Query::get_translation_string($this->key, $this->locale_code, $target_locale->code);            
            }

            if($translated) {
                /** Sekalian set variable ini supaya nggak 2x kerja, mumpung udah dapat nilainya */
                $this->translated_strings[$target_locale->code] = $translated;            
            }

            $this->translateds[$target_locale->code] = (bool) $translated;
        }

        return $this->translateds[$target_locale->code];
    }

    public function value(?string $target_locale_code=null): ?string{
        if(!isset($this->key)) return null;

        $target_locale = wpjt_get_locale($target_locale_code ?? $this->locale_code);
        if(!$target_locale) return null;

        if($target_locale->code == $this->locale_code) return $this->string;

        if(!isset($this->translated_strings[$target_locale->code])){
            $cache_key = $this->key . '|' . $this->locale_code . '|' . $target_locale->code;
            $translated = wp_cache_get($cache_key, 'wpjt_str_translate');
        
            if($translated === false) {
                if($this->is_date){
                    $formatter = new \IntlDateFormatter($target_locale->code, \IntlDateFormatter::LONG, \IntlDateFormatter::NONE);
                    $translated = $formatter->format($this->timestamp);
                } else {
                    $translated = JT_Query::get_translation_string($this->key, $this->locale_code, $target_locale->code);
                }
                
                if($translated) {
                    wp_cache_set($cache_key, $translated, 'wpjt_str_translate', 86400); // cache 24 jam
                    $this->translateds[$target_locale->code] = true;
                } else {
                    /** saat belum diterjemahkan by default kita akan menggunakan source string */
                    $translated = trim($this->string);
                }
            }

            $translated = str_repeat(' ', $this->lspc) . $translated . str_repeat(' ', $this->rspc);
            $this->translated_strings[$target_locale->code] = $translated;
        }

        return $this->translated_strings[$target_locale->code];
    }

    public function set_translation(string $target_locale_code, string $translation_string) {
        if(!isset($this->key)) return false;

        $target_locale = wpjt_get_locale($target_locale_code);
        $translation_string = trim($translation_string);

        if( !$target_locale || ($this->locale_code == $target_locale->code) || ($translation_string == '') ) return;

        $result = JT_Query::set_translation_string($this->key, $this->locale_code, $target_locale->code, $translation_string);        

        if($result){
            $this->translated_strings[$target_locale->code] = $translation_string;
            $this->translateds[$target_locale->code] = true;
        }                
        return $result;
    }
}