<?php
namespace WPJT\Classes;

use WPJT\Modules\JT_Query;

/** @package WPJT\Classes */
class JT_Path {
    public readonly string $locale_code;
    public readonly bool $is_source;

    protected readonly string $path, $rslash;
    protected readonly array $slugs;

    protected array $paths;
    protected array $translateds;

    public function __construct(string $path) {
        $this->path = $path?:'/';
        
        $slugs = array_filter(explode('/', trim($this->path, '/')));
        $first_slug = $slugs[0] ?? '';   

        if($first_slug) {
            /** Periksa apakah slug pertama merupakan lang slug */
            $locale_code = JT_Locale::get_code_by_slug($first_slug);
            /** Jika tidak dikenal maka locale codenya adalah source locale code */
            $locale_code = $locale_code ?: JT_Locale::get_source_code();
            /** periksa apakah locale code valid */
            $locale = wpjt_get_locale($locale_code);
            if($locale) {
                $this->locale_code = $locale->code;
                $this->is_source = ($locale->code == JT_Locale::get_source_code());
            }
        }

        /** pastikan nilainya terisi, defaultnya dianggap source */
        if(!isset($this->locale_code)) {
            $this->locale_code = JT_Locale::get_source_code();
        }

        /** pastikan nilainya terisi*/
        if(!isset($this->is_source)) {
            $this->is_source = ($this->locale_code == JT_Locale::get_source_code());
        }

        /** jika bukan source maka buang lang slug dari array slugs  */
        if(!$this->is_source){
            array_shift($slugs);
        }
        $this->slugs = $slugs;
        
        /** untuk trailing slash */
        $this->rslash = (empty($this->slugs) || strlen(rtrim($this->path, '/')) === strlen($this->path)) ? '' : '/';
    }
    /**
     * Get instance from cache
     */
    public static function get_instance(string $path): ?self {
        $cache_key = $path?:'/';        
        $me = wp_cache_get($cache_key, 'wpjt_path_instance');
        if(false === $me) {
            try {
                $me = new self($path);
                wp_cache_set($cache_key, $me, 'wpjt_path_instance');
            } catch (\InvalidArgumentException $e) {
                $me = null; // Set ke null jika gagal
            } 
        }
        return $me;
    }    

    public function value(?string $target_locale_code=null): string {
        $target_locale = wpjt_get_locale($target_locale_code ?? $this->locale_code);
        
        /** Kondisi target locale nggak valid atau cari $this->path */
        if(!$target_locale || $target_locale->code == $this->locale_code) {
            /** tidak langsung return $this->path 
             * diproses ulang supaya lang slug standard */
            if($this->is_source) {
                $translated_path = trim(implode('/', $this->slugs), '/');        
            } else {
                $translated_path = $target_locale->slug . '/' . trim(implode('/', $this->slugs), '/');    
            }
            return '/' . $translated_path . $this->rslash;
            // return $this->path;
        }

        if(!isset($this->paths[$target_locale->code])){
            $cache_key = $this->path . '|' . $target_locale->code;
            $translated_path = wp_cache_get($cache_key, 'wpjt_path_translate');
            
            if($translated_path === false) {
                /** Kondisi mencari original slug
                 * Untuk mendapatkan original slug maka sourcenya menggunakan target code 
                 * dan targetnya menggunakan locale code instance ini
                 */
                if($target_locale->code == JT_Locale::get_source_code()) {
                    $slugs=[];
                    foreach($this->slugs as $i=>$slug){
                        $tranlated_slug = JT_Query::get_source_slug($slug, $target_locale->code, $this->locale_code);
                        if(!$tranlated_slug){
                            $this->translateds[$target_locale->code] = false;
                            $slugs[$i] = $slug;
                        } else {
                            $slugs[$i] = $tranlated_slug;
                        }
                    }
                    $translated_path = trim(implode('/', $slugs), '/');
                }
                
                /** Kondisi mencari translated slug
                 * Untuk mencari translated source maka instance ini harus sebagai source
                 * Kita tidak support mencari translated slug berdasarkan hasil translasi, 
                 * karena kebutuhannya hampir tidak akan pernah ada dan malah akan bikin bingung
                 */
                if($this->is_source) {
                    $slugs=[];
                    foreach($this->slugs as $i=>$slug){
                        $tranlated_slug = JT_Query::get_target_slug($slug, JT_Locale::get_source_code(), $target_locale->code);
                        if(!$tranlated_slug){
                            $this->translateds[$target_locale->code] = false;
                            $slugs[$i] = $slug;
                        } else {
                            $slugs[$i] = $tranlated_slug;
                        }
                    }
                    $translated_path = $target_locale->slug . '/' . trim(implode('/', $slugs), '/');
                }

                wp_cache_set($cache_key, $translated_path, 'wpjt_path_translate');
            }
            
            $this->paths[$target_locale->code] = '/' . $translated_path . $this->rslash;
        }
        return $this->paths[$target_locale->code];
    }
    
    public function is_translated(string $target_locale_code): bool{
        $target_locale = wpjt_get_locale($target_locale_code ?? $this->locale_code);
        if(!$target_locale) return false;

        if(!isset($this->translateds[$target_locale->code])) {
            /** logic cek translated atau tidak ada di dalam value function 
             * value function hanya mendeteksi jika not translated
             * jika function sudah dipanggil namun variable belum diset artinya translated
            */
            $this->value($target_locale->code);
            if(!isset($this->translateds[$target_locale->code])) {
                $this->translateds[$target_locale->code] = true;
            }
        }

        return $this->translateds[$target_locale->code];
    }    
}