<?php
namespace WPJT\Classes;

use InvalidArgumentException;
use WPJT\Modules\JT_Query;
use WPJT\Modules\JT_Settings;

/** @package WPJT\Classes */
class JT_Locale {
    public readonly string $code;
    public readonly string $display;

    public readonly string $lang;
    public readonly string $script;
    public readonly string $region;
    
    public readonly string $slug;
    public readonly string $flag;
    public readonly string $is_source;
    public readonly string $is_active;

    public readonly string $name;
    public readonly string $native_name;
    public readonly string $english_name;
    
    public readonly string $lang_name;
    public readonly string $lang_native_name;
    public readonly string $lang_english_name;
    public readonly string $rtl;

    
    /** static */
    protected static string $active_locale_code;
    protected static string $browser_locale_code;
    protected static string $http_accept_language;    
    protected static string $source_locale_code;

    /**
     * @param string $str_locale 
     * @param string $str_display 
     * @return void 
     * @throws InvalidArgumentException 
     */
    public function __construct(string $str_locale, string $str_display) {
        $code = self::get_code($str_locale);
        if ( null === $code) {
            throw new \InvalidArgumentException("str_locale argument is invalid!");
        }

        $this->code = $code;
        $this->display = $str_display;
        $this->generate_detail();
    }

    /**
     * Get instance from cache
     * @param null|string $str_locale 
     * @param null|string $str_display_locale 
     * @return null|JT_Locale 
     */
    public static function get_instance( ?string $str_locale = null, ?string $str_display_locale = null ): ?self {
        $str_locale = $str_locale ?? wpjt_get_active_locale_code();
        $str_display_locale = $str_display_locale ?? wpjt_get_active_locale_code();

        $cache_key = $str_locale . '|' . $str_display_locale;
        $me = wp_cache_get($cache_key, 'wpjt_locale_instance');
        if(false === $me) {
            try {
                $me = new self($str_locale, $str_display_locale);
                wp_cache_set($cache_key, $me, 'wpjt_locale_instance');
            } catch (\InvalidArgumentException $e) {
                $me = null; // Set ke null jika gagal
            } 
        }

        return $me;
    }

    protected function generate_detail(){
        // urutan lang pertama, region ke-2 (jangan diubah!)
        $this->lang = \Locale::getPrimaryLanguage($this->code);
        $this->script = \Locale::getScript($this->code);
        $this->region = \Locale::getRegion($this->code);

        $this->is_source = $this->code === self::get_code( get_locale() );
        $this->is_active = ( wpjt_get_active_locale_code() === $this->code );

        $this->slug = apply_filters('wpjt_locale_slug', strtolower(str_replace('-','_',$this->code)), $this->code);
        $this->flag = apply_filters('wpjt_locale_flag_url', $this->get_flag(), $this->code, $this->region);

        $this->name = apply_filters('wpjt_locale_name', \Locale::getDisplayName($this->code, $this->display), $this->code);
        $this->native_name = apply_filters('wpjt_locale_native_name', \Locale::getDisplayName($this->code, $this->code), $this->code);
        $this->english_name = apply_filters('wpjt_locale_english_name', \Locale::getDisplayName($this->code, 'en'), $this->code);
        
        $this->lang_name = apply_filters('wpjt_lang_name', \Locale::getDisplayName($this->lang, $this->display), $this->code, $this->lang);
        $this->lang_native_name = apply_filters('wpjt_lang_native_name', \Locale::getDisplayName($this->lang, $this->lang), $this->code, $this->lang);
        $this->lang_english_name = apply_filters('wpjt_lang_english_name', \Locale::getDisplayName($this->lang, 'en'), $this->code, $this->lang);
        $this->rtl = apply_filters('wpjt_lang_rtl', JT_Settings::get_default_rtl($this->lang), $this->code, $this->lang);
    }

    protected function get_flag(){        
        $region = $this->region ? $this->region : JT_Settings::get_default_country($this->lang);
        $region = $region ? $region : 'zz';

        $flag_url = wp_cache_get($region, 'wpjt_locale_flag_url');
        if(false === $flag_url) {                  
            $url_format = WPJT_URL . 'asset/flags/%s.webp';
            $url_format = apply_filters('wpjt_locale_flag_url_format', $url_format, $this->code);
            $flag_url = sprintf($url_format, strtolower($region));            
            wp_cache_set($region, $flag_url, 'wpjt_locale_flag_url');
        }

        return $flag_url;
    }

    /** @return null|string  */
    public static function get_active_code(): ?string{
        if(!isset(self::$active_locale_code)) {
            self::$active_locale_code = self::get_code(\Locale::getDefault());
        }
        return self::$active_locale_code;
    }

    /**
     * @param string $str_locale 
     * @return void 
     */
    public static function set_active_code(string $str_locale): void{
        $locale_code = self::get_code($str_locale);
        if($locale_code){
            self::$active_locale_code = $locale_code;
            \Locale::setDefault(self::$active_locale_code);
        }                        
    }
    
    public static function get_browser_code(): ?string{
        if( !isset(self::$http_accept_language) ) {
            if(!isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) return null;
            
            self::$http_accept_language =  sanitize_text_field(wp_unslash($_SERVER['HTTP_ACCEPT_LANGUAGE']));
            $languages = explode(',', self::$http_accept_language);
            $preferred_lang = explode(';', $languages[0])[0]; 

            self::$browser_locale_code =  self::get_code($preferred_lang) ; 
        }
        return self::$browser_locale_code;
    }       

    /**
     * @param string $str_locale 
     * @return null|string 
     */
    public static function get_code(string $str_locale): ?string{
        $lang = \Locale::getPrimaryLanguage($str_locale);
        $lang_name = \Locale::getDisplayName($lang);
        $script = \Locale::getScript($str_locale);

        if($lang===$lang_name) return null;

        $region = \Locale::getRegion($str_locale);
        $full_name = \Locale::getDisplayName($str_locale);

        $code = $lang;

        if( $script ) {
            $code .= '-' . $script;
        }        
        
        if( $region && !($full_name === "$lang_name ($region)") ) {
            $code .= '-' . $region;
        }

        return $code;
    }      

    /**
     * Get supported locales
     * @return array
     */
    public static function get_codes(): array{
        return JT_Query::get_setting('site', 'locales') ?? [];
    }        

    /**
     * Save supported locales
     * @param array $locale_codes 
     * @return void 
     */
    public static function set_codes(array $locale_codes): void{
        // gabung dengan default locale
        $locale_codes = [self::get_active_code(), ...$locale_codes];

        /** agar farmat locale seragam */
        $locale_codes = array_map(function($locale_code) {
            return wpjt_get_locale($locale_code)->code;
        }, $locale_codes);

        // pastikan unique
        $locale_codes = array_unique($locale_codes);

        /** pastikan locale valid */
        $locale_codes = array_filter($locale_codes, function($locale_code) {        
            return wpjt_get_locale($locale_code) !== null;
        });         

        JT_Query::set_setting('site', 'locales', $locale_codes);
    }    


    /**
     * Get locale code by slug, if slug empty will return source locale
     * @param null|string $slug 
     * @return null|string 
     */
    public static function get_code_by_slug(?string $slug = null): ?string{
        if(null === $slug) {
            return self::get_active_code();
        }
        
        foreach(self::get_codes() as $locale_code) {
            $the_locale = wpjt_get_locale($locale_code);
            // pengecekan slug tidak case sensitive dan bisa pakai locale code, ini berfungsi agar diredirect saat typo
            if( strtolower($slug) == strtolower($locale_code) || strtolower($slug) == strtolower($the_locale->slug)){
                return $locale_code;
            }
        }
        return null;
    }

    
    /** @return null|JT_Locale  */
    public static function get_source(): ?JT_Locale {
        return self::get_instance(get_locale());
    }

    public static function get_source_code(){
        if( !isset(self::$source_locale_code) ) {
            self::$source_locale_code = self::get_source()->code;
        }
        return self::$source_locale_code;        
    }
}