<?php
/**
 * Plugin Name: JSON-LD Generator for AEO SEO
 * Plugin URI: https://answerseo.com/json-ld-generator
 * Description: Simple lightweight plugin to add FAQs and auto-insert FAQ JSON-LD schema into homepage header. To use Settings -> JSON-LD Generator. 
 * Version: 1.0
 * Author: answerSEO
 * Author URI: https://answerseo.com
 * License: GPL2
 */

if (!defined('ABSPATH')) exit;

class FAQ_JSONLD_Generator {

    private $option_name = 'faq_jsonld_data';

    public function __construct() {
        add_action('admin_menu', [$this, 'jsonldfaq_add_admin_page']);
        add_action('admin_init', [$this, 'jsonldfaq_handle_form']);
        add_action('wp_head', [$this, 'jsonldfaq_insert_jsonld_in_header']);
    }

    public function jsonldfaq_add_admin_page() {
        add_options_page(
            'JSON-LD Generator for FAQs',
            'JSON-LD Generator',
            'manage_options',
            'faq-jsonld',
            [$this, 'jsonldfaq_render_admin_page']
        );
    }

    public function jsonldfaq_handle_form() {
        if ( ! isset( $_POST['faq_jsonld_nonce'] ) || 
            ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['faq_jsonld_nonce'] ) ), 'faq_jsonld_action' ) ) {
             return;
        }

        $faqs = get_option($this->option_name, []);

        // Add FAQ
        if (isset($_POST['new_question'], $_POST['new_answer'])) {
            $faqs[] = [
                'question' => sanitize_text_field(wp_unslash($_POST['new_question'])),
                'answer'   => sanitize_textarea_field(wp_unslash($_POST['new_answer'])),
            ];
        }

        // Update FAQ
        if ( isset( $_POST['update_faq'] ) && isset( $_POST['faq_index'] ) ) {
            $index = intval( $_POST['faq_index'] );

            if ( isset( $faqs[ $index ] ) ) {
                $question = isset( $_POST['question'] ) ? sanitize_text_field( wp_unslash( $_POST['question'] ) ) : '';
                $answer   = isset( $_POST['answer'] )   ? sanitize_textarea_field( wp_unslash( $_POST['answer'] ) ) : '';

                // Only update if we have non-empty values
                if ( $question !== '' && $answer !== '' ) {
                    $faqs[ $index ]['question'] = $question;
                    $faqs[ $index ]['answer']   = $answer;
                }
            }
        }

        // Delete FAQ
        if (isset($_POST['delete_faq'])) {
            $index = isset($_POST['faq_index']) ? intval($_POST['faq_index']) : 0;
            if (isset($faqs[$index])) {
                unset($faqs[$index]);
                $faqs = array_values($faqs); // reindex
            }
        }

        update_option($this->option_name, $faqs);
    }

 public function jsonldfaq_render_admin_page() {
        $faqs = get_option($this->option_name, []);
        ?>
        <div class="wrap" style="background-color:#fff; padding:25px;border-radius: 8px;border:1px solid brown;">
            <h1 style="text-align:center;color:#FF7C0D">JSON-LD Generator using FAQs</h1>

            <h2>Add New FAQ</h2>
            <form method="post">
                <?php wp_nonce_field('faq_jsonld_action', 'faq_jsonld_nonce'); ?>
                <p><input type="text" name="new_question" placeholder="Question" required style="width: 50%;"></p>
                <p><textarea name="new_answer" placeholder="Answer" required style="width: 50%; height: 80px;"></textarea></p>
                <p><button class="button button-primary" type="submit">Add FAQ</button></p>
            </form>

            <h2>Existing FAQs</h2>
            <?php if (!empty($faqs)): ?>
                <?php foreach ($faqs as $index => $faq): ?>
                    <form method="post" style="margin-bottom: 20px; background: #fff; padding: 10px; border: 1px solid #ddd;">
                        <?php wp_nonce_field('faq_jsonld_action', 'faq_jsonld_nonce'); ?>
                        <input type="hidden" name="faq_index" value="<?php echo esc_attr($index); ?>">
                        <p>
                            <strong>Question:</strong> 
                            <input type="text" name="question" value="<?php echo esc_attr($faq['question']); ?>" style="width: 80%;">
                        </p>
                        <p>
                            <strong>Answer:</strong> 
                            <textarea name="answer" style="width: 80%; height: 60px;"><?php echo esc_textarea($faq['answer']); ?></textarea>
                        </p>
                        <p>
                            <button type="submit" name="update_faq" class="button button-secondary">Update</button>
                            <button type="submit" name="delete_faq" class="button button-link-delete" onclick="return confirm('Delete this FAQ?');">Delete</button>
                        </p>
                    </form>
                <?php endforeach; ?>
            <?php else: ?>
                <p>No FAQs yet. Add one above.</p>
            <?php endif; ?>

            <?php if (!empty($faqs)): ?>
                <h2>Generated JSON-LD Preview</h2>
                <pre style="background:#000; padding:10px; border:1px solid #ddd; max-width:70%; overflow:auto; white-space:pre-wrap;color:#fff;text-align: left;"><?php
                    $data = [
                        "@context" => "https://schema.org",
                        "@type" => "FAQPage",
                        "mainEntity" => []
                    ];
                    foreach ($faqs as $faq) {
                        $data["mainEntity"][] = [
                            "@type" => "Question",
                            "name" => $faq['question'],
                            "acceptedAnswer" => [
                                "@type" => "Answer",
                                "text" => $faq['answer']
                            ]
                        ];
                    }
                    echo esc_html( wp_json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) );
                ?></pre>
            <?php endif; ?>
            <p style="text-align: center;font-size: 18px;color:#FF7C0D;">Plugin by <strong><a href="https://answerseo.com" target="_blank">AnswerSEO.com</a></strong> - we work for Position Zero...</p>
        </div>
        <?php
    }

    public function jsonldfaq_insert_jsonld_in_header() {
    if ( ! is_front_page() ) {
        return;
    }

    $faqs = get_option( $this->option_name, [] );

    if ( empty( $faqs ) || ! is_array( $faqs ) ) {
        return;
    }

    $main_entities = [];

    foreach ( $faqs as $faq ) {
        $question = isset( $faq['question'] ) ? sanitize_text_field( $faq['question'] ) : '';
        $answer   = isset( $faq['answer'] ) ? sanitize_textarea_field( $faq['answer'] ) : '';

        if ( $question && $answer ) {
            $main_entities[] = [
                '@type' => 'Question',
                'name'  => $question,
                'acceptedAnswer' => [
                    '@type' => 'Answer',
                    'text'  => $answer,
                ],
            ];
        }
    }

    if ( empty( $main_entities ) ) {
        return;
    }

    $data = [
        '@context'   => 'https://schema.org',
        '@type'      => 'FAQPage',
        'mainEntity' => $main_entities,
    ];


    if ( $data ) {
        printf(
            '<script type="application/ld+json">%s</script>' . "\n",
            wp_json_encode( $data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE )
        );
    }
}
}

new FAQ_JSONLD_Generator();
