<?php
/**
 * Pixengine Uninstall
 *
 * Removes all plugin data when uninstalled.
 *
 * @package Pixengine
 */

// If uninstall not called from WordPress, exit
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

/**
 * Clean up plugin data
 */
function pixengine_uninstall_cleanup() {
    // Remove plugin options
    delete_option('pixengine_quality');
    delete_option('pixengine_max_width');
    delete_option('pixengine_usage_data');

    // v1.1.0 options
    delete_option('pixengine_output_format');
    delete_option('pixengine_avif_quality');
    delete_option('pixengine_lazy_loading_enabled');
    delete_option('pixengine_lazy_loading_exclude_first');
    delete_option('pixengine_cache_enabled');
    delete_option('pixengine_cache_max_age');
    delete_option('pixengine_responsive_enabled');
    delete_option('pixengine_picture_element');
    delete_option('pixengine_newsletter_dismissed');

    // Clear scheduled cron events
    $timestamp = wp_next_scheduled('pixengine_monthly_reset');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'pixengine_monthly_reset');
    }

    // Optional: Remove WebP files (commented out by default for safety)
    // Uncomment if you want to delete all WebP files on uninstall
    /*
    $upload_dir = wp_upload_dir();
    $base_dir = $upload_dir['basedir'];

    // Recursively find and delete .webp files
    pixengine_delete_webp_files($base_dir);
    */

    // Optional: Clean .htaccess rules (commented out by default)
    /*
    pixengine_remove_htaccess_rules();
    */
}

/**
 * Recursively delete WebP files from directory
 *
 * @param string $dir Directory path
 */
function pixengine_delete_webp_files($dir) {
    if (!is_dir($dir)) {
        return;
    }

    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );

    foreach ($files as $file) {
        if ($file->isFile() && $file->getExtension() === 'webp') {
            @wp_delete_file($file->getRealPath());
        }
    }
}

/**
 * Remove WebP rules from .htaccess
 */
function pixengine_remove_htaccess_rules() {
    $htaccess_file = content_url('/.htaccess');

    if (!file_exists($htaccess_file)) {
        return;
    }

    $existing_content = file_get_contents($htaccess_file);

    // Remove Pixengine WebP rules
    $new_content = preg_replace(
        '/# BEGIN WebP Image Converter.*?# END WebP Image Converter\s*/s',
        '',
        $existing_content
    );

    if ($new_content !== $existing_content) {
        @file_put_contents($htaccess_file, $new_content);
    }
}

// Run cleanup
pixengine_uninstall_cleanup();
