<?php
/**
 * Plugin Name: JPrompt's Pixengine - Image Converter & Optimizer
 * Plugin URI: https://j-prompt.com/plugins/pixengine
 * Description: Automatically converts uploaded images to WebP with resizing. Bulk converter for existing images with .htaccess redirect.
 * Version: 1.1.0
 * Author: Jani Poklar
 * Author URI: https://j-prompt.com
 * License: GPL v2 or later
 * Text Domain: jprompts-pixengine
 */

if (!defined('ABSPATH')) {
    exit;
}

class Pixengine_Converter {

    private $quality = 80;
    private $max_width = 1920;

    public function __construct() {
        // Check WebP support
        if (!$this->pixengine_check_webp_support()) {
            add_action('admin_notices', [$this, 'pixengine_webp_support_notice']);
            return;
        }

        // Hook into thumbnail generation to convert everything AFTER WordPress generates thumbnails
        add_filter('wp_generate_attachment_metadata', [$this, 'pixengine_convert_all_sizes'], 10, 2);

        // Add settings
        add_action('admin_menu', [$this, 'pixengine_add_admin_menu']);
        add_action('admin_init', [$this, 'pixengine_register_settings']);

        // Add bulk converter
        add_action('admin_enqueue_scripts', [$this, 'pixengine_enqueue_admin_scripts']);
        add_action('wp_ajax_pixengine_convert_existing', [$this, 'pixengine_ajax_convert_existing_images']);
        add_action('wp_ajax_pixengine_update_htaccess', [$this, 'pixengine_ajax_update_htaccess']);
        add_action('wp_ajax_pixengine_regenerate_thumbnails', [$this, 'pixengine_ajax_regenerate_thumbnails']);

        // Allow WebP uploads
        add_filter('mime_types', [$this, 'pixengine_add_webp_mime_type']);
        add_filter('file_is_displayable_image', [$this, 'pixengine_webp_is_displayable'], 10, 2);

        // AVIF support
        add_filter('mime_types', [$this, 'pixengine_add_avif_mime_type']);
        add_filter('file_is_displayable_image', [$this, 'pixengine_avif_is_displayable'], 10, 2);

        // Lazy loading
        if (get_option('pixengine_lazy_loading_enabled', true)) {
            add_filter('the_content', [$this, 'pixengine_add_lazy_loading_to_content'], 99);
            add_filter('wp_get_attachment_image_attributes', [$this, 'pixengine_add_lazy_loading_to_attachment'], 10, 3);
            add_filter('post_thumbnail_html', [$this, 'pixengine_add_lazy_loading_to_thumbnail'], 10, 5);
        }

        // Responsive images
        if (get_option('pixengine_responsive_enabled', true)) {
            add_filter('wp_calculate_image_srcset', [$this, 'pixengine_filter_srcset'], 10, 5);
            if (get_option('pixengine_picture_element', false)) {
                add_filter('wp_get_attachment_image', [$this, 'pixengine_generate_picture_element'], 10, 5);
            }
        }

        // Newsletter notice
        add_action('admin_notices', [$this, 'pixengine_display_newsletter_notice']);
        add_action('wp_ajax_pixengine_dismiss_newsletter', [$this, 'pixengine_ajax_dismiss_newsletter_notice']);
    }

    /**
     * Get WP_Filesystem instance
     */
    private function pixengine_get_filesystem() {
        global $wp_filesystem;

        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        if ( empty( $wp_filesystem ) ) {
            WP_Filesystem();
        }

        return $wp_filesystem;
    }

    /**
     * Check if server supports WebP
     */
    private function pixengine_check_webp_support() {
        if (function_exists('imagewebp')) {
            return true;
        }

        if (extension_loaded('imagick')) {
            $imagick = new Imagick();
            return in_array('WEBP', $imagick->queryFormats());
        }

        return false;
    }

    /**
     * Display admin notice if WebP not supported
     */
    public function pixengine_webp_support_notice() {
        ?>
        <div class="notice notice-error">
            <p><?php esc_html_e('WebP Image Converter: Your server does not support WebP format. Please enable GD with WebP support or install Imagick.', 'jprompts-pixengine'); ?></p>
        </div>
        <?php
    }

    /**
     * Convert main image and all thumbnails to optimized format after WordPress generates them
     */
    public function pixengine_convert_all_sizes($metadata, $attachment_id) {
        $file = get_attached_file($attachment_id);

        if (!$file || !file_exists($file)) {
            return $metadata;
        }

        // Only process images (JPG, PNG, or WebP)
        $mime_type = get_post_mime_type($attachment_id);
        if (strpos($mime_type, 'image/') !== 0) {
            return $metadata;
        }

        if (!in_array($mime_type, ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'])) {
            return $metadata;
        }

        // Get quality settings
        $this->quality = get_option('pixengine_quality', 80);
        $this->max_width = get_option('pixengine_max_width', 1920);

        // Get output format preference
        $output_format = get_option('pixengine_output_format', 'webp');

        // Determine which format to use for primary conversion
        $use_avif = ($output_format === 'avif' || $output_format === 'both') && $this->pixengine_check_avif_support();
        $use_webp = ($output_format === 'webp' || $output_format === 'both');

        // For new uploads, use AVIF as primary if available, otherwise WebP
        $primary_format = $use_avif ? 'avif' : 'webp';

        // Convert all thumbnails first (WordPress has generated them from the original)
        if (!empty($metadata['sizes'])) {
            $base_dir = dirname($file);

            foreach ($metadata['sizes'] as $size_name => $size_data) {
                $thumb_file = $base_dir . '/' . $size_data['file'];
                $ext = pathinfo($thumb_file, PATHINFO_EXTENSION);

                if (file_exists($thumb_file) && !in_array($ext, ['webp', 'avif'])) {
                    // For "both" mode, create both versions but keep original for fallback
                    if ($output_format === 'both') {
                        // Create both formats, keep original
                        $this->pixengine_convert_image_to_webp($thumb_file, true);
                        $this->pixengine_convert_image_to_avif($thumb_file, true);
                        // Primary file remains original, served via .htaccess
                    } elseif ($primary_format === 'avif' && $use_avif) {
                        $thumb_avif = $this->pixengine_convert_image_to_avif($thumb_file, false);
                        if ($thumb_avif && file_exists($thumb_avif)) {
                            $metadata['sizes'][$size_name]['file'] = basename($thumb_avif);
                            $metadata['sizes'][$size_name]['mime-type'] = 'image/avif';
                        }
                    } else {
                        $thumb_webp = $this->pixengine_convert_image_to_webp($thumb_file, false);
                        if ($thumb_webp && file_exists($thumb_webp)) {
                            $metadata['sizes'][$size_name]['file'] = basename($thumb_webp);
                            $metadata['sizes'][$size_name]['mime-type'] = 'image/webp';
                        }
                    }
                }
            }
        }

        // Now convert the main image (after thumbnails are done)
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        if (!in_array($ext, ['webp', 'avif'])) {
            if ($output_format === 'both') {
                // Create both formats, keep original for fallback
                $this->pixengine_convert_image_to_webp($file, true);
                $this->pixengine_convert_image_to_avif($file, true);
                // Original remains as primary, optimized versions served via .htaccess
            } elseif ($primary_format === 'avif' && $use_avif) {
                $avif_file = $this->pixengine_convert_image_to_avif($file, false);
                if ($avif_file && file_exists($avif_file)) {
                    update_attached_file($attachment_id, $avif_file);
                    if (isset($metadata['file'])) {
                        $metadata['file'] = str_replace(
                            basename($file),
                            basename($avif_file),
                            $metadata['file']
                        );
                    }
                    wp_update_post([
                        'ID' => $attachment_id,
                        'post_mime_type' => 'image/avif'
                    ]);
                }
            } else {
                $webp_file = $this->pixengine_convert_image_to_webp($file, false);
                if ($webp_file && file_exists($webp_file)) {
                    update_attached_file($attachment_id, $webp_file);
                    if (isset($metadata['file'])) {
                        $metadata['file'] = str_replace(
                            basename($file),
                            basename($webp_file),
                            $metadata['file']
                        );
                    }
                    wp_update_post([
                        'ID' => $attachment_id,
                        'post_mime_type' => 'image/webp'
                    ]);
                }
            }
        }

        return $metadata;
    }

    /**
     * Convert image using Imagick (better compression)
     */
    private function pixengine_convert_with_imagick($source_file, $webp_file, $quality, $max_width, $keep_original) {
        try {
            $imagick = new Imagick($source_file);

            // Get current dimensions
            $current_width = $imagick->getImageWidth();
            $current_height = $imagick->getImageHeight();

            // Resize if needed
            if ($current_width > $max_width) {
                $new_height = intval(($current_height / $current_width) * $max_width);
                $imagick->resizeImage($max_width, $new_height, Imagick::FILTER_LANCZOS, 1);
            }

            // Strip metadata to reduce file size
            $imagick->stripImage();

            // Set WebP format
            $imagick->setImageFormat('webp');

            // Set compression quality
            $imagick->setImageCompressionQuality($quality);

            // Use lossy compression for better file size reduction
            $imagick->setOption('webp:method', '6'); // 0-6, higher = better compression but slower
            $imagick->setOption('webp:lossless', 'false');
            $imagick->setOption('webp:alpha-quality', '100'); // Preserve alpha channel quality
            $imagick->setOption('webp:auto-filter', 'true'); // Auto-adjust filter strength

            // Write the WebP file
            $imagick->writeImage($webp_file);
            $imagick->clear();

            if (file_exists($webp_file)) {
                // Set same permissions as original using WP_Filesystem
                $filesystem = $this->pixengine_get_filesystem();
                if ( $filesystem ) {
                    $perms = @fileperms($source_file);
                    if ($perms) {
                        $filesystem->chmod($webp_file, $perms);
                    }
                }

                // Delete original if not keeping it
                if (!$keep_original && file_exists($source_file)) {
                    @wp_delete_file($source_file);
                }

                return $webp_file;
            }
        } catch (Exception $e) {
            return false;
        }

        return false;
    }

    /**
     * Convert single image to WebP with optional resizing
     */
    private function pixengine_convert_image_to_webp($source_file, $keep_original = false) {

        if (!file_exists($source_file)) {
            return false;
        }

        $pathinfo = pathinfo($source_file);
        $webp_file = $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '.webp';

        // Get quality and max width settings
        $quality = get_option('pixengine_quality', 75); // Lowered default for better WebP compression
        $max_width = get_option('pixengine_max_width', 1920);

        // Try Imagick first (usually better compression)
        if (extension_loaded('imagick')) {
            $webp_imagick = $this->pixengine_convert_with_imagick($source_file, $webp_file, $quality, $max_width, $keep_original);
            if ($webp_imagick) {
                return $webp_imagick;
            }
        }

        // Fallback to GD
        // Try to detect image type
        $image_type = @exif_imagetype($source_file);

        if (!$image_type) {
            // Fallback to extension-based detection
            $ext = strtolower($pathinfo['extension']);
            if (in_array($ext, ['jpg', 'jpeg'])) {
                $image_type = IMAGETYPE_JPEG;
            } elseif ($ext === 'png') {
                $image_type = IMAGETYPE_PNG;
            } elseif ($ext === 'webp') {
                $image_type = IMAGETYPE_WEBP;
            } else {
                return false;
            }
        }

        // Create image resource
        $image = null;

        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $image = @imagecreatefromjpeg($source_file);
                break;
            case IMAGETYPE_PNG:
                $image = @imagecreatefrompng($source_file);
                if ($image) {
                    // Preserve transparency
                    imagealphablending($image, false);
                    imagesavealpha($image, true);
                }
                break;
            case IMAGETYPE_WEBP:
                $image = @imagecreatefromwebp($source_file);
                break;
        }

        if (!$image) {
            return false;
        }

        // Get current dimensions
        $current_width = imagesx($image);
        $current_height = imagesy($image);

        // Resize if image is wider than max_width
        if ($current_width > $max_width) {
            // Calculate new dimensions maintaining aspect ratio
            $new_width = $max_width;
            $new_height = intval(($current_height / $current_width) * $new_width);

            // Create new resized image
            $resized_image = imagecreatetruecolor($new_width, $new_height);

            // Preserve transparency for PNG
            if ($image_type === IMAGETYPE_PNG) {
                imagealphablending($resized_image, false);
                imagesavealpha($resized_image, true);
                $transparent = imagecolorallocatealpha($resized_image, 0, 0, 0, 127);
                imagefill($resized_image, 0, 0, $transparent);
            }

            // Perform resize with high quality
            imagecopyresampled(
                $resized_image,
                $image,
                0, 0, 0, 0,
                $new_width, $new_height,
                $current_width, $current_height
            );

            // Destroy original and use resized
            imagedestroy($image);
            $image = $resized_image;
        }

        // Preserve alpha channel for transparency
        imagealphablending($image, false);
        imagesavealpha($image, true);

        // Convert to WebP with optimized settings
        $result = @imagewebp($image, $webp_file, $quality);
        imagedestroy($image);

        if ($result && file_exists($webp_file)) {
            // Set same permissions as original using WP_Filesystem
            $filesystem = $this->pixengine_get_filesystem();
            if ( $filesystem ) {
                $perms = @fileperms($source_file);
                if ($perms) {
                    $filesystem->chmod($webp_file, $perms);
                }
            }

            // Delete original only if not keeping it (new uploads)
            if (!$keep_original && file_exists($source_file)) {
                @wp_delete_file($source_file);
            }

            return $webp_file;
        }

        return false;
    }

    /**
     * Add WebP MIME type
     */
    public function pixengine_add_webp_mime_type($mimes) {
        $mimes['webp'] = 'image/webp';
        return $mimes;
    }

    /**
     * Allow WebP to be displayed
     */
    public function pixengine_webp_is_displayable($result, $path) {
        if ($result === false) {
            $info = @getimagesize($path);
            if ($info !== false && isset($info['mime']) && $info['mime'] === 'image/webp') {
                $result = true;
            }
        }
        return $result;
    }

    /**
     * Check if server supports AVIF
     */
    private function pixengine_check_avif_support() {
        // Check GD for AVIF support (PHP 8.1+)
        if (function_exists('imageavif')) {
            return true;
        }

        // Check Imagick for AVIF support
        if (extension_loaded('imagick')) {
            $imagick = new Imagick();
            return in_array('AVIF', $imagick->queryFormats());
        }

        return false;
    }

    /**
     * Add AVIF MIME type
     */
    public function pixengine_add_avif_mime_type($mimes) {
        $mimes['avif'] = 'image/avif';
        return $mimes;
    }

    /**
     * Allow AVIF to be displayed
     */
    public function pixengine_avif_is_displayable($result, $path) {
        if ($result === false) {
            $info = @getimagesize($path);
            if ($info !== false && isset($info['mime']) && $info['mime'] === 'image/avif') {
                $result = true;
            }
        }
        return $result;
    }

    /**
     * Convert image using Imagick to AVIF
     */
    private function pixengine_convert_with_imagick_avif($source_file, $avif_file, $quality, $max_width, $keep_original) {
        try {
            $imagick = new Imagick($source_file);

            // Get current dimensions
            $current_width = $imagick->getImageWidth();
            $current_height = $imagick->getImageHeight();

            // Resize if needed
            if ($current_width > $max_width) {
                $new_height = intval(($current_height / $current_width) * $max_width);
                $imagick->resizeImage($max_width, $new_height, Imagick::FILTER_LANCZOS, 1);
            }

            // Strip metadata to reduce file size
            $imagick->stripImage();

            // Set AVIF format
            $imagick->setImageFormat('avif');

            // Set compression quality
            $imagick->setImageCompressionQuality($quality);

            // Write the AVIF file
            $imagick->writeImage($avif_file);
            $imagick->clear();

            if (file_exists($avif_file)) {
                // Set same permissions as original using WP_Filesystem
                $filesystem = $this->pixengine_get_filesystem();
                if ($filesystem) {
                    $perms = @fileperms($source_file);
                    if ($perms) {
                        $filesystem->chmod($avif_file, $perms);
                    }
                }

                // Delete original if not keeping it
                if (!$keep_original && file_exists($source_file)) {
                    @wp_delete_file($source_file);
                }

                return $avif_file;
            }
        } catch (Exception $e) {
            return false;
        }

        return false;
    }

    /**
     * Convert single image to AVIF with optional resizing
     */
    private function pixengine_convert_image_to_avif($source_file, $keep_original = false) {
        if (!file_exists($source_file)) {
            return false;
        }

        // Check if AVIF is supported
        if (!$this->pixengine_check_avif_support()) {
            return false;
        }

        $pathinfo = pathinfo($source_file);
        $avif_file = $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '.avif';

        // Get quality and max width settings
        $quality = get_option('pixengine_avif_quality', 70);
        $max_width = get_option('pixengine_max_width', 1920);

        // Try Imagick first (usually better compression)
        if (extension_loaded('imagick')) {
            $imagick = new Imagick();
            if (in_array('AVIF', $imagick->queryFormats())) {
                $avif_imagick = $this->pixengine_convert_with_imagick_avif($source_file, $avif_file, $quality, $max_width, $keep_original);
                if ($avif_imagick) {
                    return $avif_imagick;
                }
            }
        }

        // Fallback to GD (PHP 8.1+)
        if (!function_exists('imageavif')) {
            return false;
        }

        // Try to detect image type
        $image_type = @exif_imagetype($source_file);

        if (!$image_type) {
            // Fallback to extension-based detection
            $ext = strtolower($pathinfo['extension']);
            if (in_array($ext, ['jpg', 'jpeg'])) {
                $image_type = IMAGETYPE_JPEG;
            } elseif ($ext === 'png') {
                $image_type = IMAGETYPE_PNG;
            } elseif ($ext === 'webp') {
                $image_type = IMAGETYPE_WEBP;
            } else {
                return false;
            }
        }

        // Create image resource
        $image = null;

        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $image = @imagecreatefromjpeg($source_file);
                break;
            case IMAGETYPE_PNG:
                $image = @imagecreatefrompng($source_file);
                if ($image) {
                    imagealphablending($image, false);
                    imagesavealpha($image, true);
                }
                break;
            case IMAGETYPE_WEBP:
                $image = @imagecreatefromwebp($source_file);
                break;
        }

        if (!$image) {
            return false;
        }

        // Get current dimensions
        $current_width = imagesx($image);
        $current_height = imagesy($image);

        // Resize if image is wider than max_width
        if ($current_width > $max_width) {
            $new_width = $max_width;
            $new_height = intval(($current_height / $current_width) * $new_width);

            $resized_image = imagecreatetruecolor($new_width, $new_height);

            if ($image_type === IMAGETYPE_PNG) {
                imagealphablending($resized_image, false);
                imagesavealpha($resized_image, true);
                $transparent = imagecolorallocatealpha($resized_image, 0, 0, 0, 127);
                imagefill($resized_image, 0, 0, $transparent);
            }

            imagecopyresampled(
                $resized_image,
                $image,
                0, 0, 0, 0,
                $new_width, $new_height,
                $current_width, $current_height
            );

            imagedestroy($image);
            $image = $resized_image;
        }

        // Preserve alpha channel
        imagealphablending($image, false);
        imagesavealpha($image, true);

        // Convert to AVIF
        $result = @imageavif($image, $avif_file, $quality);
        imagedestroy($image);

        if ($result && file_exists($avif_file)) {
            // Set same permissions as original
            $filesystem = $this->pixengine_get_filesystem();
            if ($filesystem) {
                $perms = @fileperms($source_file);
                if ($perms) {
                    $filesystem->chmod($avif_file, $perms);
                }
            }

            // Delete original only if not keeping it
            if (!$keep_original && file_exists($source_file)) {
                @wp_delete_file($source_file);
            }

            return $avif_file;
        }

        return false;
    }

    /**
     * Add admin menu
     */
    public function pixengine_add_admin_menu() {
        // Add top-level menu
        add_menu_page(
            __('Pixengine Converter', 'jprompts-pixengine'),
            __('Pixengine', 'jprompts-pixengine'),
            'manage_options',
            'pixengine-converter',
            [$this, 'pixengine_admin_page'],
            'dashicons-images-alt2',
            65
        );

        // Add submenu for settings (same as main page)
        add_submenu_page(
            'pixengine-converter',
            __('Settings', 'jprompts-pixengine'),
            __('Settings', 'jprompts-pixengine'),
            'manage_options',
            'pixengine-converter',
            [$this, 'pixengine_admin_page']
        );
    }

    /**
     * Register settings
     */
    public function pixengine_register_settings() {
        // WebP quality
        register_setting('pixengine_settings', 'pixengine_quality', [
            'type' => 'integer',
            'default' => 75,
            'sanitize_callback' => function($value) {
                return max(1, min(100, intval($value)));
            }
        ]);

        // Max width
        register_setting('pixengine_settings', 'pixengine_max_width', [
            'type' => 'integer',
            'default' => 1920,
            'sanitize_callback' => function($value) {
                $val = intval($value);
                return $val > 0 ? $val : 1920;
            }
        ]);

        // Output format (webp, avif, both)
        register_setting('pixengine_settings', 'pixengine_output_format', [
            'type' => 'string',
            'default' => 'webp',
            'sanitize_callback' => function($value) {
                return in_array($value, ['webp', 'avif', 'both']) ? $value : 'webp';
            }
        ]);

        // AVIF quality
        register_setting('pixengine_settings', 'pixengine_avif_quality', [
            'type' => 'integer',
            'default' => 70,
            'sanitize_callback' => function($value) {
                return max(1, min(100, intval($value)));
            }
        ]);

        // Lazy loading enabled
        register_setting('pixengine_settings', 'pixengine_lazy_loading_enabled', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => function($value) {
                return (bool) $value;
            }
        ]);

        // Lazy loading exclude first N images
        register_setting('pixengine_settings', 'pixengine_lazy_loading_exclude_first', [
            'type' => 'integer',
            'default' => 3,
            'sanitize_callback' => function($value) {
                return max(0, intval($value));
            }
        ]);

        // Cache enabled
        register_setting('pixengine_settings', 'pixengine_cache_enabled', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => function($value) {
                return (bool) $value;
            }
        ]);

        // Cache max age (seconds)
        register_setting('pixengine_settings', 'pixengine_cache_max_age', [
            'type' => 'integer',
            'default' => 31536000,
            'sanitize_callback' => function($value) {
                return max(0, intval($value));
            }
        ]);

        // Responsive images enabled
        register_setting('pixengine_settings', 'pixengine_responsive_enabled', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => function($value) {
                return (bool) $value;
            }
        ]);

        // Picture element enabled
        register_setting('pixengine_settings', 'pixengine_picture_element', [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => function($value) {
                return (bool) $value;
            }
        ]);
    }

    /**
     * Admin page
     */
    public function pixengine_admin_page() {
        // Get some stats
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image/webp',
            'posts_per_page' => -1,
            'post_status' => 'any',
            'fields' => 'ids'
        ];
        $webp_query = new WP_Query($args);
        $webp_count = $webp_query->found_posts;

        $args['post_mime_type'] = 'image/avif';
        $avif_query = new WP_Query($args);
        $avif_count = $avif_query->found_posts;

        $args['post_mime_type'] = ['image/jpeg', 'image/jpg', 'image/png'];
        $other_query = new WP_Query($args);
        $other_count = $other_query->found_posts;

        // Check AVIF support
        $avif_supported = $this->pixengine_check_avif_support();

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Pixengine - Image Converter & Optimizer', 'jprompts-pixengine'); ?></h1>

            <!-- Go Pro -->
            <div style="background: #fff; border: 1px solid #ccc; padding: 20px; margin: 20px 0; border-radius: 4px;">
                <h2><?php esc_html_e('Go Pro', 'jprompts-pixengine'); ?></h2>
                <p><?php esc_html_e('Get priority support and additional features with the Pro version.', 'jprompts-pixengine'); ?></p>
                <p style="margin-top: 15px;">
                    <a href="https://j-prompt.com/plugins/pixengine/" target="_blank" class="button button-primary">
                        <?php esc_html_e('Go Pro', 'jprompts-pixengine'); ?>
                    </a>
                </p>
            </div>

            <div style="background: #fff; border: 1px solid #ccc; padding: 20px; margin: 20px 0; border-radius: 4px;">
                <h2><?php esc_html_e('Media Library Stats', 'jprompts-pixengine'); ?></h2>
                <table class="widefat">
                    <tr>
                        <td><strong><?php esc_html_e('WebP Images:', 'jprompts-pixengine'); ?></strong></td>
                        <td><?php echo number_format($webp_count); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php esc_html_e('AVIF Images:', 'jprompts-pixengine'); ?></strong></td>
                        <td><?php echo number_format($avif_count); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php esc_html_e('Other Images (JPG/PNG):', 'jprompts-pixengine'); ?></strong></td>
                        <td><?php echo number_format($other_count); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php esc_html_e('AVIF Support:', 'jprompts-pixengine'); ?></strong></td>
                        <td>
                            <?php if ($avif_supported): ?>
                                <span style="color: green;"><?php esc_html_e('Available', 'jprompts-pixengine'); ?></span>
                            <?php else: ?>
                                <span style="color: #999;"><?php esc_html_e('Not available (requires PHP 8.1+ or Imagick with AVIF)', 'jprompts-pixengine'); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
                <p class="description" style="margin-top: 10px;">
                    <?php esc_html_e('All new uploads will be automatically converted based on your output format settings.', 'jprompts-pixengine'); ?>
                    <?php if ($other_count > 0): ?>
                        <br><strong><?php esc_html_e('Note:', 'jprompts-pixengine'); ?></strong>
                        <?php printf(esc_html__('You have %d existing JPG/PNG images. These will remain as-is unless you manually convert them.', 'jprompts-pixengine'), intval($other_count)); ?>
                    <?php endif; ?>
                </p>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields('pixengine_settings'); ?>

                <!-- Output Format Section -->
                <h2><?php esc_html_e('Output Format', 'jprompts-pixengine'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="pixengine_output_format"><?php esc_html_e('Image Format', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <?php $output_format = get_option('pixengine_output_format', 'webp'); ?>
                            <select id="pixengine_output_format" name="pixengine_output_format">
                                <option value="webp" <?php selected($output_format, 'webp'); ?>><?php esc_html_e('WebP Only', 'jprompts-pixengine'); ?></option>
                                <option value="avif" <?php selected($output_format, 'avif'); ?> <?php disabled(!$avif_supported); ?>><?php esc_html_e('AVIF Only', 'jprompts-pixengine'); ?></option>
                                <option value="both" <?php selected($output_format, 'both'); ?> <?php disabled(!$avif_supported); ?>><?php esc_html_e('Both (WebP + AVIF)', 'jprompts-pixengine'); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('AVIF offers better compression than WebP but has less browser support. "Both" creates both formats for maximum compatibility.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_quality"><?php esc_html_e('WebP Quality', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="pixengine_quality" name="pixengine_quality"
                                   value="<?php echo esc_attr(get_option('pixengine_quality', 75)); ?>"
                                   min="1" max="100" style="width: 80px;" />
                            <p class="description">
                                <?php esc_html_e('Quality for WebP conversion (1-100). WebP 75 is roughly equal to JPEG 85-90. Default: 75', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_avif_quality"><?php esc_html_e('AVIF Quality', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="pixengine_avif_quality" name="pixengine_avif_quality"
                                   value="<?php echo esc_attr(get_option('pixengine_avif_quality', 70)); ?>"
                                   min="1" max="100" style="width: 80px;" <?php disabled(!$avif_supported); ?> />
                            <p class="description">
                                <?php esc_html_e('Quality for AVIF conversion (1-100). AVIF achieves better compression, so lower values work well. Default: 70', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_max_width"><?php esc_html_e('Maximum Width', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="pixengine_max_width" name="pixengine_max_width"
                                   value="<?php echo esc_attr(get_option('pixengine_max_width', 1920)); ?>"
                                   min="100" step="1" style="width: 100px;" /> px
                            <p class="description">
                                <?php esc_html_e('Maximum image width in pixels. Images wider than this will be resized. Default: 1920px', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <!-- Lazy Loading Section -->
                <h2><?php esc_html_e('Lazy Loading', 'jprompts-pixengine'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="pixengine_lazy_loading_enabled"><?php esc_html_e('Enable Lazy Loading', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="pixengine_lazy_loading_enabled" name="pixengine_lazy_loading_enabled"
                                   value="1" <?php checked(get_option('pixengine_lazy_loading_enabled', true)); ?> />
                            <label for="pixengine_lazy_loading_enabled">
                                <?php esc_html_e('Add loading="lazy" attribute to images', 'jprompts-pixengine'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Defers loading of off-screen images to improve page load speed.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_lazy_loading_exclude_first"><?php esc_html_e('Exclude First N Images', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="pixengine_lazy_loading_exclude_first" name="pixengine_lazy_loading_exclude_first"
                                   value="<?php echo esc_attr(get_option('pixengine_lazy_loading_exclude_first', 3)); ?>"
                                   min="0" style="width: 80px;" />
                            <p class="description">
                                <?php esc_html_e('Number of images to exclude from lazy loading (above-the-fold images). Default: 3', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <!-- Responsive Images Section -->
                <h2><?php esc_html_e('Responsive Images', 'jprompts-pixengine'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="pixengine_responsive_enabled"><?php esc_html_e('Enable Responsive Optimization', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="pixengine_responsive_enabled" name="pixengine_responsive_enabled"
                                   value="1" <?php checked(get_option('pixengine_responsive_enabled', true)); ?> />
                            <label for="pixengine_responsive_enabled">
                                <?php esc_html_e('Update srcset to use optimized format URLs', 'jprompts-pixengine'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Automatically updates image srcset attributes to point to optimized WebP/AVIF versions.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_picture_element"><?php esc_html_e('Use Picture Element', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="pixengine_picture_element" name="pixengine_picture_element"
                                   value="1" <?php checked(get_option('pixengine_picture_element', false)); ?> />
                            <label for="pixengine_picture_element">
                                <?php esc_html_e('Wrap images in &lt;picture&gt; elements for format fallback', 'jprompts-pixengine'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Provides explicit format fallback: AVIF > WebP > Original. May not be needed if using .htaccess rules.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <!-- Caching Section -->
                <h2><?php esc_html_e('Caching', 'jprompts-pixengine'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="pixengine_cache_enabled"><?php esc_html_e('Enable Cache Headers', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="pixengine_cache_enabled" name="pixengine_cache_enabled"
                                   value="1" <?php checked(get_option('pixengine_cache_enabled', true)); ?> />
                            <label for="pixengine_cache_enabled">
                                <?php esc_html_e('Add cache headers to .htaccess for optimized images', 'jprompts-pixengine'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Configures browser caching for WebP and AVIF images to improve repeat visit performance.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="pixengine_cache_max_age"><?php esc_html_e('Cache Duration', 'jprompts-pixengine'); ?></label>
                        </th>
                        <td>
                            <?php $cache_max_age = get_option('pixengine_cache_max_age', 31536000); ?>
                            <select id="pixengine_cache_max_age" name="pixengine_cache_max_age">
                                <option value="86400" <?php selected($cache_max_age, 86400); ?>><?php esc_html_e('1 Day', 'jprompts-pixengine'); ?></option>
                                <option value="604800" <?php selected($cache_max_age, 604800); ?>><?php esc_html_e('1 Week', 'jprompts-pixengine'); ?></option>
                                <option value="2592000" <?php selected($cache_max_age, 2592000); ?>><?php esc_html_e('1 Month', 'jprompts-pixengine'); ?></option>
                                <option value="31536000" <?php selected($cache_max_age, 31536000); ?>><?php esc_html_e('1 Year (Recommended)', 'jprompts-pixengine'); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('How long browsers should cache optimized images. 1 year is recommended for static images.', 'jprompts-pixengine'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>

            <hr>

            <h2><?php esc_html_e('Regenerate Thumbnails', 'jprompts-pixengine'); ?></h2>
            <div class="notice notice-info inline">
                <p>
                    <strong><?php esc_html_e('Fix Missing Thumbnails:', 'jprompts-pixengine'); ?></strong><br>
                    <?php esc_html_e('If you have images with missing thumbnails, use this tool to regenerate them. WordPress will create all thumbnail sizes and convert them to WebP.', 'jprompts-pixengine'); ?>
                </p>
            </div>

            <button type="button" id="regenerate-thumbnails-btn" class="button button-secondary">
                <?php esc_html_e('Regenerate All Thumbnails', 'jprompts-pixengine'); ?>
            </button>

            <div id="regenerate-progress" style="margin-top: 20px; display: none;">
                <div style="background: #fff; border: 1px solid #ccc; padding: 20px; border-radius: 4px;">
                    <h3><?php esc_html_e('Regeneration Progress', 'jprompts-pixengine'); ?></h3>
                    <div id="regenerate-progress-bar" style="width: 100%; background: #f0f0f0; height: 30px; border-radius: 4px; overflow: hidden;">
                        <div id="regenerate-progress-fill" style="width: 0%; background: #46b450; height: 100%; transition: width 0.3s;"></div>
                    </div>
                    <p id="regenerate-progress-text" style="margin-top: 10px;">0 / 0</p>
                    <div id="regenerate-log" style="max-height: 300px; overflow-y: auto; margin-top: 15px; font-family: monospace; font-size: 12px;"></div>
                </div>
            </div>

            <hr>

            <h2><?php esc_html_e('Convert Existing Images', 'jprompts-pixengine'); ?></h2>
            <div class="notice notice-info inline">
                <p>
                    <strong><?php esc_html_e('Bulk Conversion:', 'jprompts-pixengine'); ?></strong><br>
                    <?php esc_html_e('Convert all existing JPG/PNG images in your media library to WebP format. Original files will be kept and .htaccess will be configured to serve WebP versions automatically.', 'jprompts-pixengine'); ?>
                </p>
            </div>

            <button type="button" id="convert-existing-images" class="button button-primary">
                <?php esc_html_e('Start Bulk Conversion', 'jprompts-pixengine'); ?>
            </button>

            <div id="conversion-progress" style="margin-top: 20px; display: none;">
                <div style="background: #fff; border: 1px solid #ccc; padding: 20px; border-radius: 4px;">
                    <h3><?php esc_html_e('Conversion Progress', 'jprompts-pixengine'); ?></h3>
                    <div id="progress-bar" style="width: 100%; background: #f0f0f0; height: 30px; border-radius: 4px; overflow: hidden;">
                        <div id="progress-fill" style="width: 0%; background: #2271b1; height: 100%; transition: width 0.3s;"></div>
                    </div>
                    <p id="progress-text" style="margin-top: 10px;">0 / 0</p>
                    <div id="conversion-log" style="max-height: 300px; overflow-y: auto; margin-top: 15px; font-family: monospace; font-size: 12px;"></div>
                </div>
            </div>

            <hr>

            <div style="background: #fffbcc; border: 1px solid #e6db55; padding: 15px; border-radius: 4px;">
                <h3><?php esc_html_e('⚠️ Important Notes', 'jprompts-pixengine'); ?></h3>
                <h4><?php esc_html_e('New Uploads:', 'jprompts-pixengine'); ?></h4>
                <ul style="margin-left: 20px;">
                    <li><?php esc_html_e('Original JPG/PNG files are permanently deleted', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Only WebP versions are stored (saves storage)', 'jprompts-pixengine'); ?></li>
                </ul>

                <h4 style="margin-top: 10px;"><?php esc_html_e('Bulk Conversion (Existing Images):', 'jprompts-pixengine'); ?></h4>
                <ul style="margin-left: 20px;">
                    <li><?php esc_html_e('Original files are kept alongside WebP versions', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('.htaccess automatically serves WebP to supported browsers', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Fallback to original images for older browsers', 'jprompts-pixengine'); ?></li>
                </ul>

                <h4 style="margin-top: 10px;"><?php esc_html_e('General:', 'jprompts-pixengine'); ?></h4>
                <ul style="margin-left: 20px;">
                    <li><?php esc_html_e('Make sure your server and browser support WebP', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Test with a few images before bulk converting', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Backup your site before bulk conversion', 'jprompts-pixengine'); ?></li>
                </ul>
            </div>

            <div style="margin-top: 20px; background: #f0f0f1; padding: 15px; border-radius: 4px;">
                <h3><?php esc_html_e('How It Works', 'jprompts-pixengine'); ?></h3>

                <div style="background: #d4edda; border: 1px solid #c3e6cb; padding: 10px; margin-bottom: 15px; border-radius: 4px;">
                    <strong><?php esc_html_e('✓ Optimized for Maximum Compression:', 'jprompts-pixengine'); ?></strong>
                    <ul style="margin: 5px 0 0 20px;">
                        <li><?php esc_html_e('Uses Imagick (if available) for superior compression', 'jprompts-pixengine'); ?></li>
                        <li><?php esc_html_e('Strips metadata to reduce file size', 'jprompts-pixengine'); ?></li>
                        <li><?php esc_html_e('Optimized WebP settings for 40-70% file size reduction', 'jprompts-pixengine'); ?></li>
                        <li><?php esc_html_e('Preserves transparency for PNG images', 'jprompts-pixengine'); ?></li>
                    </ul>
                </div>

                <h4><?php esc_html_e('New Uploads (Automatic):', 'jprompts-pixengine'); ?></h4>
                <ol style="margin-left: 20px;">
                    <li><?php esc_html_e('Upload an image (JPG or PNG)', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Plugin resizes if width > max width (maintains aspect ratio)', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Converts to WebP with optimized compression', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Original file is deleted (saves storage)', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('All thumbnails are resized and converted to WebP', 'jprompts-pixengine'); ?></li>
                </ol>

                <h4 style="margin-top: 15px;"><?php esc_html_e('Existing Images (Bulk Conversion):', 'jprompts-pixengine'); ?></h4>
                <ol style="margin-left: 20px;">
                    <li><?php esc_html_e('Click "Start Bulk Conversion" button above', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Plugin scans all JPG/PNG images in media library', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Creates WebP versions alongside originals', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Original files are kept (safer for existing images)', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('.htaccess is updated to serve WebP when browser supports it', 'jprompts-pixengine'); ?></li>
                    <li><?php esc_html_e('Site automatically serves WebP to supported browsers', 'jprompts-pixengine'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }

    /**
     * Enqueue admin scripts
     */
    public function pixengine_enqueue_admin_scripts($hook) {
        if ($hook !== 'toplevel_page_pixengine-converter') {
            return;
        }

        wp_enqueue_script('jquery');

        // Inline script for bulk conversion and thumbnail regeneration
        $script = "
        jQuery(document).ready(function($) {
            let isConverting = false;
            let isRegenerating = false;
            let totalImages = 0;
            let processedImages = 0;
            
            $('#convert-existing-images').on('click', function() {
                if (isConverting) {
                    return;
                }
                
                if (!confirm('This will convert all existing JPG and PNG images to WebP format. Original files will be kept. Continue?')) {
                    return;
                }
                
                isConverting = true;
                processedImages = 0;
                
                $(this).prop('disabled', true).text('Converting...');
                $('#conversion-progress').show();
                $('#conversion-log').html('');
                
                convertBatch(0);
            });
            
            function convertBatch(offset) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'pixengine_convert_existing',
                        nonce: '" . wp_create_nonce('pixengine_bulk_nonce') . "',
                        offset: offset
                    },
                    success: function(response) {
                        if (response.success) {
                            const data = response.data;
                            totalImages = data.total;
                            processedImages = data.processed;
                            
                            // Update progress
                            const percentage = totalImages > 0 ? (processedImages / totalImages * 100) : 0;
                            $('#progress-fill').css('width', percentage + '%');
                            $('#progress-text').text(processedImages + ' / ' + totalImages + ' images processed');
                            
                            // Log converted images
                            if (data.converted && data.converted.length > 0) {
                                data.converted.forEach(function(item) {
                                    $('#conversion-log').append(
                                        '<div style=\"color: green;\">✓ Converted: ' + item.title + ' → ' + item.webp + '</div>'
                                    );
                                });
                                
                                // Auto-scroll to bottom
                                $('#conversion-log').scrollTop($('#conversion-log')[0].scrollHeight);
                            }
                            
                            // Continue with next batch
                            if (data.has_more) {
                                convertBatch(data.processed);
                            } else {
                                // Finished - update htaccess
                                updateHtaccess();
                            }
                        } else {
                            handleError(response.data.message || 'Unknown error');
                        }
                    },
                    error: function(xhr, status, error) {
                        handleError('AJAX error: ' + error);
                    }
                });
            }
            
            function updateHtaccess() {
                $('#conversion-log').append('<div style=\"color: blue;\">⚙ Updating .htaccess file...</div>');
                $('#conversion-log').scrollTop($('#conversion-log')[0].scrollHeight);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'pixengine_update_htaccess',
                        nonce: '" . wp_create_nonce('pixengine_htaccess_nonce') . "'
                    },
                    success: function(response) {
                        isConverting = false;
                        $('#convert-existing-images').prop('disabled', false).text('Start Bulk Conversion');
                        
                        if (response.success) {
                            $('#conversion-log').append('<div style=\"color: green; font-weight: bold;\">✓ .htaccess updated successfully!</div>');
                            $('#conversion-log').append('<div style=\"color: blue; font-weight: bold; margin-top: 10px;\">✓ Conversion complete! WebP images will be served automatically.</div>');
                        } else {
                            $('#conversion-log').append('<div style=\"color: orange;\">⚠ Conversion complete but .htaccess update failed. You may need to update it manually.</div>');
                        }
                        $('#conversion-log').scrollTop($('#conversion-log')[0].scrollHeight);
                    },
                    error: function() {
                        isConverting = false;
                        $('#convert-existing-images').prop('disabled', false).text('Start Bulk Conversion');
                        $('#conversion-log').append('<div style=\"color: orange;\">⚠ Conversion complete but .htaccess update failed. You may need to update it manually.</div>');
                        $('#conversion-log').scrollTop($('#conversion-log')[0].scrollHeight);
                    }
                });
            }
            
            function handleError(message) {
                isConverting = false;
                $('#convert-existing-images').prop('disabled', false).text('Start Bulk Conversion');
                $('#conversion-log').append('<div style=\"color: red;\">✗ Error: ' + message + '</div>');
                $('#conversion-log').scrollTop($('#conversion-log')[0].scrollHeight);
                alert('An error occurred during conversion. Check the log for details.');
            }

            // Regenerate Thumbnails Handler
            $('#regenerate-thumbnails-btn').on('click', function() {
                if (isRegenerating) {
                    return;
                }

                if (!confirm('This will regenerate thumbnails for all WebP images. This may take a while. Continue?')) {
                    return;
                }

                isRegenerating = true;
                processedImages = 0;

                $(this).prop('disabled', true).text('Regenerating...');
                $('#regenerate-progress').show();
                $('#regenerate-log').html('');

                regenerateBatch(0);
            });

            function regenerateBatch(offset) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'pixengine_regenerate_thumbnails',
                        nonce: '" . wp_create_nonce('pixengine_regenerate_nonce') . "',
                        offset: offset
                    },
                    success: function(response) {
                        if (response.success) {
                            const data = response.data;
                            totalImages = data.total;
                            processedImages = data.processed;

                            // Update progress
                            const percentage = totalImages > 0 ? (processedImages / totalImages * 100) : 0;
                            $('#regenerate-progress-fill').css('width', percentage + '%');
                            $('#regenerate-progress-text').text(processedImages + ' / ' + totalImages + ' images processed');

                            // Log regenerated images
                            if (data.regenerated && data.regenerated.length > 0) {
                                data.regenerated.forEach(function(item) {
                                    if (item.success) {
                                        $('#regenerate-log').append(
                                            '<div style=\"color: green;\">✓ Regenerated: ' + item.title + ' (' + item.count + ' thumbnails)</div>'
                                        );
                                    } else {
                                        $('#regenerate-log').append(
                                            '<div style=\"color: orange;\">⚠ Skipped: ' + item.title + ' - ' + item.message + '</div>'
                                        );
                                    }
                                });

                                // Auto-scroll to bottom
                                $('#regenerate-log').scrollTop($('#regenerate-log')[0].scrollHeight);
                            }

                            // Continue with next batch
                            if (data.has_more) {
                                regenerateBatch(data.processed);
                            } else {
                                // Finished
                                isRegenerating = false;
                                $('#regenerate-thumbnails-btn').prop('disabled', false).text('Regenerate All Thumbnails');
                                $('#regenerate-log').append('<div style=\"color: blue; font-weight: bold; margin-top: 10px;\">✓ Thumbnail regeneration complete!</div>');
                                $('#regenerate-log').scrollTop($('#regenerate-log')[0].scrollHeight);
                            }
                        } else {
                            handleRegenerateError(response.data.message || 'Unknown error');
                        }
                    },
                    error: function(xhr, status, error) {
                        handleRegenerateError('AJAX error: ' + error);
                    }
                });
            }

            function handleRegenerateError(message) {
                isRegenerating = false;
                $('#regenerate-thumbnails-btn').prop('disabled', false).text('Regenerate All Thumbnails');
                $('#regenerate-log').append('<div style=\"color: red;\">✗ Error: ' + message + '</div>');
                $('#regenerate-log').scrollTop($('#regenerate-log')[0].scrollHeight);
                alert('An error occurred during regeneration. Check the log for details.');
            }
        });
        ";

        wp_add_inline_script('jquery', $script);
    }

    /**
     * AJAX handler for bulk conversion
     */
    public function pixengine_ajax_convert_existing_images() {
        check_ajax_referer('pixengine_bulk_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = 10;

        // Get output format preference
        $output_format = get_option('pixengine_output_format', 'webp');
        $create_webp = ($output_format === 'webp' || $output_format === 'both');
        $create_avif = ($output_format === 'avif' || $output_format === 'both') && $this->pixengine_check_avif_support();

        // Get images
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => ['image/jpeg', 'image/jpg', 'image/png'],
            'posts_per_page' => $batch_size,
            'offset' => $offset,
            'post_status' => 'any'
        ];

        $query = new WP_Query($args);

        // Get total count
        $total_args = [
            'post_type' => 'attachment',
            'post_mime_type' => ['image/jpeg', 'image/jpg', 'image/png'],
            'posts_per_page' => -1,
            'post_status' => 'any',
            'fields' => 'ids'
        ];
        $total_query = new WP_Query($total_args);
        $total = $total_query->found_posts;

        if ($query->have_posts()) {
            $converted = [];

            foreach ($query->posts as $post) {
                $file = get_attached_file($post->ID);

                if ($file && file_exists($file)) {
                    $formats_created = [];

                    // Convert to WebP keeping original
                    if ($create_webp) {
                        $webp_file = $this->pixengine_convert_image_to_webp($file, true);
                        if ($webp_file) {
                            $formats_created[] = 'webp';
                        }
                    }

                    // Convert to AVIF keeping original
                    if ($create_avif) {
                        $avif_file = $this->pixengine_convert_image_to_avif($file, true);
                        if ($avif_file) {
                            $formats_created[] = 'avif';
                        }
                    }

                    if (!empty($formats_created)) {
                        $converted[] = [
                            'id' => $post->ID,
                            'title' => get_the_title($post->ID),
                            'webp' => implode(', ', $formats_created)
                        ];

                        // Convert thumbnails keeping originals
                        $metadata = wp_get_attachment_metadata($post->ID);
                        if (!empty($metadata['sizes'])) {
                            $base_dir = dirname($file);
                            foreach ($metadata['sizes'] as $size => $size_data) {
                                $thumb_file = $base_dir . '/' . $size_data['file'];
                                if (file_exists($thumb_file)) {
                                    if ($create_webp) {
                                        $this->pixengine_convert_image_to_webp($thumb_file, true);
                                    }
                                    if ($create_avif) {
                                        $this->pixengine_convert_image_to_avif($thumb_file, true);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            wp_send_json_success([
                'converted' => $converted,
                'total' => $total,
                'processed' => min($offset + $batch_size, $total),
                'has_more' => ($offset + $batch_size) < $total
            ]);
        } else {
            wp_send_json_success([
                'converted' => [],
                'total' => $total,
                'processed' => $offset,
                'has_more' => false
            ]);
        }
    }

    /**
     * AJAX handler for thumbnail regeneration
     */
    public function pixengine_ajax_regenerate_thumbnails() {
        check_ajax_referer('pixengine_regenerate_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = 5; // Smaller batch size since regeneration is more intensive

        // Get all WebP images (these are the ones that might have missing thumbnails)
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image/webp',
            'posts_per_page' => $batch_size,
            'offset' => $offset,
            'post_status' => 'any'
        ];

        $query = new WP_Query($args);

        // Get total count
        $total_args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image/webp',
            'posts_per_page' => -1,
            'post_status' => 'any',
            'fields' => 'ids'
        ];
        $total_query = new WP_Query($total_args);
        $total = $total_query->found_posts;

        if ($query->have_posts()) {
            $regenerated = [];

            foreach ($query->posts as $post) {
                $file = get_attached_file($post->ID);

                if (!$file || !file_exists($file)) {
                    $regenerated[] = [
                        'id' => $post->ID,
                        'title' => get_the_title($post->ID),
                        'success' => false,
                        'message' => 'File not found'
                    ];
                    continue;
                }

                // Get current metadata
                $old_metadata = wp_get_attachment_metadata($post->ID);
                $old_thumb_count = !empty($old_metadata['sizes']) ? count($old_metadata['sizes']) : 0;

                // Require WordPress image functions
                require_once(ABSPATH . 'wp-admin/includes/image.php');

                // Regenerate thumbnails - this will trigger our convert_all_sizes hook
                $metadata = wp_generate_attachment_metadata($post->ID, $file);

                if ($metadata && !is_wp_error($metadata)) {
                    wp_update_attachment_metadata($post->ID, $metadata);

                    $new_thumb_count = !empty($metadata['sizes']) ? count($metadata['sizes']) : 0;

                    $regenerated[] = [
                        'id' => $post->ID,
                        'title' => get_the_title($post->ID),
                        'success' => true,
                        'count' => $new_thumb_count,
                        'message' => "Regenerated {$new_thumb_count} thumbnails"
                    ];
                } else {
                    $regenerated[] = [
                        'id' => $post->ID,
                        'title' => get_the_title($post->ID),
                        'success' => false,
                        'message' => 'Failed to regenerate'
                    ];
                }
            }

            wp_send_json_success([
                'regenerated' => $regenerated,
                'total' => $total,
                'processed' => min($offset + $batch_size, $total),
                'has_more' => ($offset + $batch_size) < $total
            ]);
        } else {
            wp_send_json_success([
                'regenerated' => [],
                'total' => $total,
                'processed' => $offset,
                'has_more' => false
            ]);
        }
    }

    /**
     * AJAX handler for updating .htaccess with optimized image redirect rules
     */
    public function pixengine_ajax_update_htaccess() {
        check_ajax_referer('pixengine_htaccess_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $htaccess_file = content_url('/.htaccess');

        // Get output format preference
        $output_format = get_option('pixengine_output_format', 'webp');
        $cache_enabled = get_option('pixengine_cache_enabled', true);
        $cache_max_age = get_option('pixengine_cache_max_age', 31536000);

        // Build redirect rules
        $rules = '# BEGIN Pixengine Image Converter' . "\n";
        $rules .= '<IfModule mod_rewrite.c>' . "\n";
        $rules .= '  RewriteEngine On' . "\n";
        $rules .= '  ' . "\n";

        // AVIF rules first (better compression, check first)
        if ($output_format === 'avif' || $output_format === 'both') {
            $rules .= '  # AVIF first (better compression)' . "\n";
            $rules .= '  RewriteCond %{HTTP_ACCEPT} image/avif' . "\n";
            $rules .= '  RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$' . "\n";
            $rules .= '  RewriteCond %1.avif -f' . "\n";
            $rules .= '  RewriteRule ^(.+)\.(jpe?g|png)$ $1.avif [T=image/avif,E=accept:1,L]' . "\n";
            $rules .= '  ' . "\n";
        }

        // WebP rules (fallback if AVIF not supported or only WebP selected)
        if ($output_format === 'webp' || $output_format === 'both') {
            $rules .= '  # WebP' . ($output_format === 'both' ? ' fallback' : '') . "\n";
            $rules .= '  RewriteCond %{HTTP_ACCEPT} image/webp' . "\n";
            $rules .= '  RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$' . "\n";
            $rules .= '  RewriteCond %1.webp -f' . "\n";
            $rules .= '  RewriteRule ^(.+)\.(jpe?g|png)$ $1.webp [T=image/webp,E=accept:1,L]' . "\n";
        }

        $rules .= '</IfModule>' . "\n";
        $rules .= '' . "\n";

        // Headers for Vary
        $rules .= '<IfModule mod_headers.c>' . "\n";
        $rules .= '  Header append Vary Accept env=REDIRECT_accept' . "\n";
        $rules .= '</IfModule>' . "\n";
        $rules .= '' . "\n";

        // MIME types
        $rules .= '# Serve optimized images with correct MIME type' . "\n";
        $rules .= '<IfModule mod_mime.c>' . "\n";
        $rules .= '  AddType image/webp .webp' . "\n";
        $rules .= '  AddType image/avif .avif' . "\n";
        $rules .= '</IfModule>' . "\n";

        // Caching rules
        if ($cache_enabled) {
            $rules .= '' . "\n";
            $rules .= '# Cache optimized images' . "\n";
            $rules .= '<IfModule mod_expires.c>' . "\n";
            $rules .= '  ExpiresActive On' . "\n";
            $rules .= '  ExpiresByType image/webp "access plus 1 year"' . "\n";
            $rules .= '  ExpiresByType image/avif "access plus 1 year"' . "\n";
            $rules .= '</IfModule>' . "\n";
            $rules .= '<IfModule mod_headers.c>' . "\n";
            $rules .= '  <FilesMatch "\.(webp|avif)$">' . "\n";
            $rules .= '    Header set Cache-Control "public, max-age=' . intval($cache_max_age) . ', immutable"' . "\n";
            $rules .= '  </FilesMatch>' . "\n";
            $rules .= '</IfModule>' . "\n";
        }

        $rules .= '# END Pixengine Image Converter' . "\n";

        $existing_content = '';
        if (file_exists($htaccess_file)) {
            $existing_content = file_get_contents($htaccess_file);

            // Remove old rules if they exist (both old and new marker names)
            $existing_content = preg_replace(
                '/# BEGIN (WebP|Pixengine) Image Converter.*?# END (WebP|Pixengine) Image Converter\s*/s',
                '',
                $existing_content
            );
        }

        // Add new rules at the beginning
        $new_content = $rules . $existing_content;

        // Try to write the file
        $result = @file_put_contents($htaccess_file, $new_content);

        if ($result !== false) {
            // Set proper permissions using WP_Filesystem
            $filesystem = $this->pixengine_get_filesystem();
            if ($filesystem) {
                $filesystem->chmod($htaccess_file, 0644);
            }
            wp_send_json_success(['message' => '.htaccess updated successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to write .htaccess file. Please check permissions.']);
        }
    }

    /**
     * Detect installed caching plugins
     */
    private function pixengine_detect_caching_plugins() {
        $plugins = [];

        if (defined('WP_CACHE') && WP_CACHE) {
            $plugins[] = 'wp-cache';
        }

        if (defined('W3TC')) {
            $plugins[] = 'w3-total-cache';
        }

        if (defined('LSCWP_V')) {
            $plugins[] = 'litespeed-cache';
        }

        if (class_exists('WP_Super_Cache')) {
            $plugins[] = 'wp-super-cache';
        }

        return $plugins;
    }

    /**
     * Add lazy loading to content images
     */
    public function pixengine_add_lazy_loading_to_content($content) {
        if (empty($content)) {
            return $content;
        }

        $exclude_first = get_option('pixengine_lazy_loading_exclude_first', 3);
        $image_count = 0;

        // Find all img tags and add loading="lazy" to those after the first N
        $content = preg_replace_callback(
            '/<img([^>]*)>/i',
            function ($matches) use (&$image_count, $exclude_first) {
                $image_count++;

                // Skip first N images (above the fold)
                if ($image_count <= $exclude_first) {
                    return $matches[0];
                }

                $attributes = $matches[1];

                // Skip if already has loading attribute
                if (preg_match('/\bloading\s*=/i', $attributes)) {
                    return $matches[0];
                }

                // Add loading="lazy"
                return '<img loading="lazy"' . $attributes . '>';
            },
            $content
        );

        return $content;
    }

    /**
     * Add lazy loading to attachment images
     */
    public function pixengine_add_lazy_loading_to_attachment($attr, $attachment, $size) {
        static $attachment_count = 0;
        $attachment_count++;

        $exclude_first = get_option('pixengine_lazy_loading_exclude_first', 3);

        // Skip first N images
        if ($attachment_count <= $exclude_first) {
            return $attr;
        }

        // Skip if already has loading attribute
        if (isset($attr['loading'])) {
            return $attr;
        }

        $attr['loading'] = 'lazy';
        return $attr;
    }

    /**
     * Add lazy loading to post thumbnails
     */
    public function pixengine_add_lazy_loading_to_thumbnail($html, $post_id, $post_thumbnail_id, $size, $attr) {
        static $thumbnail_count = 0;
        $thumbnail_count++;

        $exclude_first = get_option('pixengine_lazy_loading_exclude_first', 3);

        // Skip first N images
        if ($thumbnail_count <= $exclude_first) {
            return $html;
        }

        // Skip if already has loading attribute
        if (strpos($html, 'loading=') !== false) {
            return $html;
        }

        // Add loading="lazy" to img tag
        return preg_replace('/<img/i', '<img loading="lazy"', $html, 1);
    }

    /**
     * Check if image should have lazy loading
     */
    private function pixengine_should_lazy_load($image_index) {
        $exclude_first = get_option('pixengine_lazy_loading_exclude_first', 3);
        return $image_index > $exclude_first;
    }

    /**
     * Filter srcset to use optimized format URLs
     */
    public function pixengine_filter_srcset($sources, $size_array, $image_src, $image_meta, $attachment_id) {
        if (empty($sources)) {
            return $sources;
        }

        $output_format = get_option('pixengine_output_format', 'webp');

        foreach ($sources as $width => $source) {
            $url = $source['url'];

            // Check for optimized versions
            $formats_to_check = [];
            if ($output_format === 'avif' || $output_format === 'both') {
                $formats_to_check[] = 'avif';
            }
            if ($output_format === 'webp' || $output_format === 'both') {
                $formats_to_check[] = 'webp';
            }

            foreach ($formats_to_check as $format) {
                $optimized_url = preg_replace('/\.(jpe?g|png)$/i', '.' . $format, $url);
                if ($optimized_url !== $url) {
                    // Check if the optimized file exists
                    $upload_dir = wp_upload_dir();
                    $relative_path = str_replace($upload_dir['baseurl'], '', $optimized_url);
                    $file_path = $upload_dir['basedir'] . $relative_path;

                    if (file_exists($file_path)) {
                        $sources[$width]['url'] = $optimized_url;
                        break; // Use first available format
                    }
                }
            }
        }

        return $sources;
    }

    /**
     * Get available formats for an image
     */
    private function pixengine_get_available_formats($image_path) {
        $formats = [];
        $pathinfo = pathinfo($image_path);
        $base_path = $pathinfo['dirname'] . '/' . $pathinfo['filename'];

        if (file_exists($base_path . '.avif')) {
            $formats[] = 'avif';
        }
        if (file_exists($base_path . '.webp')) {
            $formats[] = 'webp';
        }
        if (file_exists($image_path)) {
            $formats[] = $pathinfo['extension'];
        }

        return $formats;
    }

    /**
     * Generate picture element for format fallback
     */
    public function pixengine_generate_picture_element($html, $attachment_id, $size, $icon, $attr) {
        if (empty($html) || strpos($html, '<img') === false) {
            return $html;
        }

        $file = get_attached_file($attachment_id);
        if (!$file) {
            return $html;
        }

        $formats = $this->pixengine_get_available_formats($file);

        // If no optimized formats, return original
        if (count($formats) <= 1) {
            return $html;
        }

        $upload_dir = wp_upload_dir();
        $pathinfo = pathinfo($file);
        $base_url = str_replace($upload_dir['basedir'], $upload_dir['baseurl'], $pathinfo['dirname'] . '/' . $pathinfo['filename']);

        // Build picture element
        $picture = '<picture>';

        // Add AVIF source first (best compression)
        if (in_array('avif', $formats)) {
            $picture .= '<source type="image/avif" srcset="' . esc_url($base_url . '.avif') . '">';
        }

        // Add WebP source
        if (in_array('webp', $formats)) {
            $picture .= '<source type="image/webp" srcset="' . esc_url($base_url . '.webp') . '">';
        }

        // Add original img tag
        $picture .= $html;
        $picture .= '</picture>';

        return $picture;
    }

    /**
     * Display newsletter admin notice
     */
    public function pixengine_display_newsletter_notice() {
        // Only show on Pixengine admin page
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'toplevel_page_pixengine-converter') {
            return;
        }

        // Check if dismissed
        if (get_option('pixengine_newsletter_dismissed', false)) {
            return;
        }

        ?>
        <div class="notice notice-info is-dismissible pixengine-newsletter-notice">
            <p>
                <strong><?php esc_html_e('Stay Updated with JPrompt!', 'jprompts-pixengine'); ?></strong><br>
                <?php esc_html_e('Get tips, tutorials, and updates about Pixengine and other WordPress optimization tools.', 'jprompts-pixengine'); ?>
                <?php echo __('We will appreciate it if you leave a review on our plugin on <a href="https://wordpress.org/support/plugin/jprompts-pixengine/reviews/#new-post">here</a>.', 'jprompts-pixengine'); ?>
            </p>
            <p>
                <a href="https://j-prompt.com/#newsletter" target="_blank" class="button button-primary">
                    <?php esc_html_e('Subscribe to our Newsletter', 'jprompts-pixengine'); ?>
                </a>
                <button type="button" class="button pixengine-dismiss-newsletter" style="margin-left: 10px;">
                    <?php esc_html_e('Dismiss', 'jprompts-pixengine'); ?>
                </button>
            </p>
        </div>
        <script>
            jQuery(document).ready(function($) {
                $('.pixengine-dismiss-newsletter, .pixengine-newsletter-notice .notice-dismiss').on('click', function(e) {
                    e.preventDefault();
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'pixengine_dismiss_newsletter',
                            nonce: '<?php echo esc_js(wp_create_nonce('pixengine_newsletter_nonce')); ?>'
                        },
                        success: function() {
                            $('.pixengine-newsletter-notice').fadeOut();
                        }
                    });
                });
            });
        </script>
        <?php
    }

    /**
     * AJAX handler for dismissing newsletter notice
     */
    public function pixengine_ajax_dismiss_newsletter_notice() {
        check_ajax_referer('pixengine_newsletter_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        update_option('pixengine_newsletter_dismissed', true);
        wp_send_json_success();
    }
}

// Initialize the plugin
new Pixengine_Converter();
