<?php
/**
 * Plugin Name: Auto Alt Images
 * Description: Automatically fills in missing alt attributes for images using post title or filename.
 * Version: 1.0.0
 * Author: Neil Williams
 * License: GPL2+
 */

defined( 'ABSPATH' ) || exit;

/**
 * Fallback for images inserted via wp_get_attachment_image()
 *    → uses filename if no alt is set
 */
add_filter( 'wp_get_attachment_image_attributes', function( $attr, $attachment ) {

    if ( empty( $attr['alt'] ) ) {
        // Fallback: use the filename (without extension)
        $filename = pathinfo( get_attached_file( $attachment->ID ), PATHINFO_FILENAME );
        $attr['alt'] = ucwords( str_replace( array('-', '_'), ' ', $filename ) );
    }

    return $attr;

}, 10, 2 );


/**
 * Filter post content <img> tags
 *    → injects current post title as alt if missing
 */
add_filter( 'the_content', function( $content ) {
    global $post;

    if ( ! $post ) {
        return $content;
    }

    $content = preg_replace_callback(
        '/<img([^>]+)>/i',
        function( $matches ) use ( $post ) {
            $img_tag = $matches[0];

            // Skip if alt already present
            if ( strpos( $img_tag, 'alt=' ) !== false ) {
                return $img_tag;
            }

            // Insert alt with current post title
            return preg_replace(
                '/<img/i',
                '<img alt="' . esc_attr( $post->post_title ) . '"',
                $img_tag,
                1
            );
        },
        $content
    );

    return $content;
});
