<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * این تابع یک URL را برای کوئری دیتابیس نرمال‌سازی می‌کند.
 * ورودی کاربر (چه فارسی و چه کدگذاری شده) را به فرمت استاندارد دیتابیس تبدیل می‌کند.
 */
function jaycoto_normalize_url_for_db_query($url) {
	if (empty(trim($url))) {
		return '';
	}
	$decoded_url = urldecode(trim($url));
    $parts = wp_parse_url($decoded_url);

	if (false === $parts || !isset($parts['scheme']) || !isset($parts['host'])) {
		return esc_url_raw($decoded_url);
	}

	$path_segments = explode('/', $parts['path'] ?? '');
	$encoded_path_segments = array_map('rawurlencode', $path_segments);
	$encoded_path = implode('/', $encoded_path_segments);

	$scheme   = $parts['scheme'] . '://';
	$host     = $parts['host'];
	$port     = isset($parts['port']) ? ':' . $parts['port'] : '';
	$query    = isset($parts['query']) ? '?' . $parts['query'] : '';
	$fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

	return $scheme . $host . $port . $encoded_path . $query . $fragment;
}

if (!current_user_can('edit_published_posts')) {
	wp_die('شما اجازه دسترسی به این بخش را ندارید.');
}

global $wpdb;
$table = $wpdb->prefix . 'jay_jaycoto_clicks';
require_once plugin_dir_path( __DIR__ ) . 'includes/jaycoto-jdf.php';

// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
$anchor_text_raw = isset($_GET['anchor_text']) ? wp_unslash($_GET['anchor_text']) : '';
$date_filter_raw = isset($_GET['click_date']) ? wp_unslash($_GET['click_date']) : '';
$range_days_raw  = isset($_GET['range_days']) ? wp_unslash($_GET['range_days']) : '';
$target_url_raw  = isset($_GET['target_url']) ? wp_unslash($_GET['target_url']) : '';
$source_page_raw = isset($_GET['source_page']) ? wp_unslash($_GET['source_page']) : '';
$paged_raw       = isset($_GET['paged']) ? wp_unslash($_GET['paged']) : '1';
$orderby_raw     = isset($_GET['orderby']) ? wp_unslash($_GET['orderby']) : 'click_date';
$order_raw       = isset($_GET['order']) ? wp_unslash($_GET['order']) : 'DESC';
// phpcs:enable

$anchor_text = sanitize_text_field($anchor_text_raw);
$date_filter = sanitize_text_field($date_filter_raw);
$range_days  = intval($range_days_raw);
$paged       = max(1, intval($paged_raw));
$orderby_param = sanitize_key($orderby_raw);
$order_param   = sanitize_key($order_raw);

// نسخه نرمال‌شده برای کوئری
$target_url_for_db  = jaycoto_normalize_url_for_db_query($target_url_raw);
$source_page_for_db = jaycoto_normalize_url_for_db_query($source_page_raw);
// نسخه خوانا برای نمایش در فیلدها
$target_url_display  = sanitize_text_field(urldecode($target_url_raw));
$source_page_display = sanitize_text_field(urldecode($source_page_raw));

$where_clauses   = [];
$params          = [];
$where_clauses[] = 'anchor_text = %s';
$params[]        = $anchor_text;

if (!empty($date_filter)) {
	list($jy, $jm, $jd) = explode('/', $date_filter);
	$gregorian         = jaycoto_jalali_to_gregorian((int) $jy, (int) $jm, (int) $jd);
	$date_g            = sprintf('%04d-%02d-%02d', $gregorian[0], $gregorian[1], $gregorian[2]);
	$where_clauses[]   = 'click_date = %s';
	$params[]          = $date_g;
} elseif ($range_days > 0) {
	$date_from       = gmdate('Y-m-d', strtotime("-$range_days days"));
	$where_clauses[] = 'click_date >= %s';
	$params[]        = $date_from;
}
if (!empty($target_url_for_db)) {
	$where_clauses[] = "TRIM(TRAILING '/' FROM target_url) = TRIM(TRAILING '/' FROM %s)";
	$params[]        = rtrim($target_url_for_db, '/');
}
if (!empty($source_page_for_db)) {
	$where_clauses[] = "TRIM(TRAILING '/' FROM source_page) = TRIM(TRAILING '/' FROM %s)";
	$params[]        = rtrim($source_page_for_db, '/');
}

$where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
$order_by  = 'clicks' === $orderby_param ? 'click_count' : 'click_date';
$order     = 'asc' === strtolower($order_param) ? 'ASC' : 'DESC';
$per_page  = 10;
$offset    = ($paged - 1) * $per_page;

$cache_params = array_merge($params, [$paged, $order_by, $order]);
$cache_key    = 'jaycoto_clicks_detail_' . md5(wp_json_encode($cache_params));
$cache_group  = 'jaycoto_clicks_details';
$cached_data  = wp_cache_get($cache_key, $cache_group);

if (false === $cached_data) { 
    
$total_sql = "SELECT COUNT(*) FROM {$table} {$where_sql}";
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
$total = $wpdb->get_var($wpdb->prepare($total_sql, $params));

	$sql          = "SELECT * FROM {$table} {$where_sql} ORDER BY {$order_by} {$order} LIMIT %d OFFSET %d";
	$query_params = array_merge($params, [$per_page, $offset]);
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results($wpdb->prepare($sql, $query_params));

	$cached_data = ['total' => $total, 'results' => $results];
	wp_cache_set($cache_key, $cached_data, $cache_group, HOUR_IN_SECONDS);
}

$total       = $cached_data['total'];
$results     = $cached_data['results'];
$total_pages = ceil($total / $per_page);

$grouped_details = [];
foreach ($results as $row) {
	$key = $row->target_url . '|' . $row->source_page . '|' . $row->click_date;
	if (!isset($grouped_details[$key])) {
		$grouped_details[$key] = [
			'target_url'  => urldecode($row->target_url),
			'source_page' => urldecode($row->source_page),
			'click_date'  => jaycoto_jdate('Y/m/d', strtotime($row->click_date)),
			'click_count' => $row->click_count,
		];
	} else {
		$grouped_details[$key]['click_count'] += $row->click_count;
	}
}
// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
$current_url = esc_url_raw(remove_query_arg(array('orderby', 'order', 'paged'), wp_unslash($_SERVER['REQUEST_URI'] ?? '')));
$toggle_order = ('ASC' === $order) ? 'desc' : 'asc';
?>
<div class="wrap">
	<h1>📊 جزئیات کلیک‌ها برای: <?php echo esc_html($anchor_text); ?></h1>
	
    <div style="position: relative;display:flex; align-items:center; gap:10px; margin-bottom:10px;">
		<a href="<?php echo esc_url(admin_url('admin.php?page=jaycoto_click_stats')); ?>" class="button">⬅️ بازگشت</a>
		<button id="filter-help-toggle" class="button button-secondary" title="راهنمای فیلترها" style="padding: 4px 8px; line-height:1;">❔</button>
		<div id="filter-help-box" style="display:none; position:absolute; top:10px; right:0; background:#fefefe; border:1px solid #ccc; padding:15px; max-width:400px; box-shadow:0 0 10px rgba(0,0,0,0.1); z-index:1000;">
			<p style="margin:0;">
				🔹 فیلترهای تاریخ و آدرس هرکدام به صورت مستقل قابل استفاده‌اند.<br>
				🔹 فیلدهای آدرس مقصد و صفحه کلیک‌شده می‌توانند همزمان یا جداگانه پر شوند.<br>
				🔹 آدرس‌ها نسبت به / در انتهای URL حساس نیستند.<br>
				🔹 نتایج دقیق با توجه به فیلترها نشان داده می‌شوند.<br>
			</p>
		</div>
	</div>
	
	<form method="get" style="margin-bottom:20px; border: 1px solid #c3c4c7; padding: 15px; background: #fff; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
		<input type="hidden" name="page" value="jaycoto-click-details">
		<input type="hidden" name="anchor_text" value="<?php echo esc_attr($anchor_text); ?>">
		<div style="display: flex; align-items: flex-end; gap: 15px; flex-wrap: wrap;">
			<div>
				<label for="jaycoto-click-date" style="display: block; margin-bottom: 5px;">تاریخ (شمسی):</label>
				<input id="jaycoto-click-date" type="text" name="click_date" class="persian-date" value="<?php echo esc_attr($date_filter); ?>" placeholder="مثلاً 1403/03/01" autocomplete="off">
			</div>
			<div>
				<label for="jaycoto-target-url" style="display: block; margin-bottom: 5px;">فیلتر آدرس مقصد:</label>
				<input id="jaycoto-target-url" type="text" name="target_url" value="<?php echo esc_attr($target_url_display); ?>" placeholder="بخشی از URL مقصد" style="direction: ltr; text-align: left; width: 250px;">
			</div>
			<div>
				<label for="jaycoto-source-page" style="display: block; margin-bottom: 5px;">فیلتر صفحه کلیک‌شده:</label>
				<input id="jaycoto-source-page" type="text" name="source_page" value="<?php echo esc_attr($source_page_display); ?>" placeholder="بخشی از URL مبدا" style="direction: ltr; text-align: left; width: 250px;">
			</div>
			<div>
				<button class="button button-primary" type="submit">اعمال فیلترها</button>
			</div>
		</div>
		<div style="margin-top: 15px; padding-top: 15px; border-top: 1px dashed #ddd; display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
			<strong>بازه زمانی سریع:</strong>
			<?php $active_range = $range_days; ?>
			<a href="<?php echo esc_url(add_query_arg(['range_days' => 1, 'click_date' => ''])); ?>" class="button button-secondary <?php echo ($active_range === 1) ? 'active' : ''; ?>">🕐 ۲۴ ساعت</a>
			<a href="<?php echo esc_url(add_query_arg(['range_days' => 7, 'click_date' => ''])); ?>" class="button button-secondary <?php echo ($active_range === 7) ? 'active' : ''; ?>">🗓 ۷ روز</a>
			<a href="<?php echo esc_url(add_query_arg(['range_days' => 28, 'click_date' => ''])); ?>" class="button button-secondary <?php echo ($active_range === 28) ? 'active' : ''; ?>">📅 ۲۸ روز</a>
			<a href="<?php echo esc_url(add_query_arg(['range_days' => 90, 'click_date' => ''])); ?>" class="button button-secondary <?php echo ($active_range === 90) ? 'active' : ''; ?>">📅 ۹۰ روز</a>
			
			<?php if ( ! empty( $date_filter ) || $range_days > 0 || ! empty( $target_url_display ) || ! empty( $source_page_display ) ) : ?>
				<?php $reset_url = remove_query_arg( array( 'click_date', 'range_days', 'target_url', 'source_page' ) ); ?>
				<a href="<?php echo esc_url( $reset_url ); ?>" class="button">ریست تمام فیلترها</a>
			<?php endif; ?>
		</div>
	</form>

	<?php if (!empty($grouped_details)): ?>
		<p>🔢 تعداد رکوردها: <?php echo esc_html($total); ?></p>
		<table class="widefat striped">
			<thead>
				<tr>
					<th>#</th>
					<th>🎯 آدرس مقصد</th>
					<th>📄 صفحه کلیک‌شده</th>
					<th>📅 تاریخ</th>
					<th>
						<a href="<?php echo esc_url(add_query_arg(['orderby' => 'clicks', 'order' => $toggle_order], $current_url)); ?>">
							🔢 تعداد کلیک <?php echo ($order_by === 'click_count') ? ('ASC' === $order ? '⬆️' : '⬇️') : ''; ?>
						</a>
					</th>
				</tr>
			</thead>
			<tbody>
				<?php
				$row_num = $offset + 1;
				foreach ($grouped_details as $detail):
				?>
				<tr>
					<td><?php echo esc_html($row_num++); ?></td>
					<td><a href="<?php echo esc_url($detail['target_url']); ?>" target="_blank"><?php echo esc_html($detail['target_url']); ?></a></td>
					<td><a href="<?php echo esc_url($detail['source_page']); ?>" target="_blank"><?php echo esc_html($detail['source_page']); ?></a></td>
					<td><?php echo esc_html($detail['click_date']); ?></td>
					<td><?php echo esc_html($detail['click_count']); ?></td>
				</tr>
				<?php endforeach; ?>
			</tbody>
		</table>

		<?php if ($total_pages > 1): ?>
			<div style="margin-top: 20px; text-align:center;display: flex;justify-content: center;align-items: center;gap: 5px;">
				<?php
				$base_url = remove_query_arg('paged');
				$range = 2;
				if ($paged > 1) {
					echo '<a class="button" href="' . esc_url(add_query_arg('paged', $paged - 1, $base_url)) . '">◀ قبلی</a>';
				}
				for ($i = max(1, $paged - $range); $i <= min($total_pages, $paged + $range); $i++) {
					$class = ($i === $paged) ? 'button button-primary' : 'button';
					echo '<a class="' . esc_attr($class) . '" href="' . esc_url(add_query_arg('paged', $i, $base_url)) . '">' . esc_html($i) . '</a>';
				}
				if ($paged < $total_pages) {
					echo '<a class="button" href="' . esc_url(add_query_arg('paged', $paged + 1, $base_url)) . '">بعدی ▶</a>';
				}
				?>
			</div>
		<?php endif; ?>
	<?php else: ?>
		<p>
			هیچ رکوردی یافت نشد.
			<?php if (!empty($target_url_display)): ?>
				برای آدرس:
				<strong style="direction: ltr; display: inline-block;"><?php echo esc_html($target_url_display); ?></strong>
			<?php endif; ?>
			<?php if (!empty($source_page_display)): ?>
				در صفحه:
				<strong style="direction: ltr; display: inline-block;"><?php echo esc_html($source_page_display); ?></strong>
			<?php endif; ?>
		</p>
	<?php endif; ?>
</div>