<?php

namespace Itmar\WpsettingClassPackage;

if (!defined('ABSPATH')) exit;

class ItmarRedirectControl
{
    private static $instance = null;
    private $redirect_option = 'itmar_redirect_to_subdir';
    private $index_php_path;

    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->index_php_path = $_SERVER['DOCUMENT_ROOT'] . '/index.php';
        // 設定保存処理
        add_action('admin_init', [$this, 'maybe_apply_redirect']);
        //admin_notices でエラー通知
        add_action('admin_notices', [$this, 'admin_notice_permission_error']);
    }

    public function maybe_apply_redirect()
    {
        //ログインユーザーが管理者であること
        if (!current_user_can('manage_options')) return;

        // nonceで想定外のフォームのsubmitを処理しない
        if (
            !isset($_POST['_wpnonce']) ||
            !wp_verify_nonce($_POST['_wpnonce'], 'itmar_setting_nonce')
        ) {
            return;
        }

        // 設定の保存リクエスト時のみ動作
        if (isset($_POST['submit'])) {

            $enabled = isset($_POST[$this->redirect_option]) ? 1 : 0;
            update_option($this->redirect_option, $enabled);

            $siteurl = get_option('siteurl');
            $parsed_url = parse_url($siteurl);
            $domain_root = $parsed_url['scheme'] . '://' . $parsed_url['host'] . '/';

            $parsed_path = isset($parsed_url['path']) ? trim($parsed_url['path'], '/') : '';

            if ($enabled) {
                $success = $this->generate_index_php($parsed_path);
                if ($success) {
                    // サイトアドレス(URL) をルートに変更
                    update_option('home', $domain_root);
                } else {
                    // 失敗時は設定戻す
                    update_option($this->redirect_option, 0);
                    update_option('home', trailingslashit($siteurl)); // 元に戻す
                    update_option('itmar_permission_error', 1);
                }
            } else {
                $this->remove_index_php();
                // サイトアドレス(URL) を元に戻す
                update_option('home', trailingslashit($siteurl));
            }
        }
    }

    /** index.php を生成 */
    private function generate_index_php($subdir)
    {
        if (empty($subdir)) return false; // サブディレクトリがないなら不要

        // すでに index.php が存在する場合はエラー
        if (file_exists($this->index_php_path)) {
            //$needle = "require __DIR__ . '/{$subdir}/wp-blog-header.php';";
            $contents = file_get_contents($this->index_php_path);

            $pattern = '/require\s+__DIR__\s*\.\s*[\'"]\/\s*' . preg_quote($subdir, '/') . '\s*\/wp-blog-header\.php[\'"];/';
            if (preg_match($pattern, $contents)) {

                // ファイルが存在し、require 文も含まれている
                return true;
            } else {
                // ファイルはあるが、指定の require 文は含まれていない
                return false;
            }
        }

        // index.php 内容
        $content = <<<PHP
            <?php
            /**
            * Auto-generated by ItmarRedirectControl
            */
            define('WP_USE_THEMES', true);
            require __DIR__ . '/{$subdir}/wp-blog-header.php';
            PHP;

        // 書き込み権限確認
        if (!is_writable(dirname($this->index_php_path))) {
            return false;
        }

        // index.php 作成
        return file_put_contents($this->index_php_path, $content) !== false;
    }

    /** ItmarRedirectControl が生成した index.php のみ削除 */
    private function remove_index_php()
    {
        if (
            file_exists($this->index_php_path) &&
            is_writable($this->index_php_path)
        ) {
            $content = file_get_contents($this->index_php_path);
            if (strpos($content, 'Auto-generated by ItmarRedirectControl') !== false) {
                unlink($this->index_php_path);
            }
        }
    }



    /** 設定画面をレンダリング */
    public function render_settings_section()
    {
        $enabled = get_option($this->redirect_option, 0);

?>
        <h2><?php
            esc_html_e('Redirect Settings', 'wpsetting-class-package');
            ?></h2>
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Redirecting from the domain root URL', 'wpsetting-class-package'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr($this->redirect_option); ?>" value="1" <?php checked($enabled, 1); ?>>
                        <?php esc_html_e('Redirect the domain root URL to this site', 'wpsetting-class-package'); ?>
                    </label>
                    <p class="description"><?php esc_html_e('If this site is installed on a subdomain, it will be accessible at the root domain URL.', 'wpsetting-class-package'); ?></p>
                </td>
            </tr>
        </table>

        <?php
    }

    public function admin_notice_permission_error()
    {
        if (get_option('itmar_permission_error')) {
        ?>
            <div class="notice notice-error is-dismissible">
                <p><?php esc_html_e('Failed to create index.php. The file already exists or insufficient permissions. Please remove index.php manually if you want to enable redirect.', 'wpsetting-class-package'); ?></p>
            </div>
<?php
            delete_option('itmar_permission_error'); // 表示後に削除
        }
    }
}
