<?php

namespace Itmar\SNS\Twitter;

use WP_REST_Request;
use WP_Error;

class TokenReceiver
{
    public function __construct()
    {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    public function register_routes()
    {
        register_rest_route('itmar-sns/v1', '/receive_twitter_token', [
            'methods'  => 'POST',
            'callback' => [$this, 'handle_token'],
            'permission_callback' => [$this, 'permission_check'],
        ]);
    }

    /**
     * 権限チェック
     */
    public function permission_check(WP_REST_Request $request)
    {
        $secret   = "itmaroon_sns_rest_key";
        $received = $request->get_header('x-itmar-sns-secret');

        if (empty($secret) || ! is_string($received) || ! hash_equals($secret, $received)) {
            return new WP_Error(
                'rest_forbidden',
                __('Invalid signature.', 'itmaroon-social-post-sync'),
                ['status' => 403]
            );
        }

        return true;
    }

    public function handle_token($request)
    {
        $access_token  = sanitize_text_field($request->get_param('access_token'));
        $refresh_token = sanitize_text_field($request->get_param('refresh_token'));
        $expires_in    = intval($request->get_param('expires_in'));

        if (!$access_token || !$refresh_token) {
            return new \WP_Error('missing_data', 'Access token or refresh token missing', ['status' => 400]);
        }

        update_option('itmar_twitter_access_token', $access_token);
        update_option('itmar_twitter_refresh_token', $refresh_token);
        update_option('itmar_twitter_expires_at', time() + $expires_in);

        $warnings = [];
        $user = $this->fetch_twitter_user($access_token);

        if (is_array($user)) {
            update_option('itmar_twitter_user', $user);
        } elseif (is_string($user)) {
            $warnings[] = $user;
        }

        return rest_ensure_response([
            'status' => 'success',
            'message' => 'Tokens received and saved',
            'warnings' => $warnings,
        ]);
    }

    private function fetch_twitter_user($access_token)
    {
        $response = wp_remote_get('https://api.twitter.com/2/users/me?user.fields=name,profile_image_url,username', [
            'headers' => ['Authorization' => 'Bearer ' . $access_token]
        ]);

        if (is_wp_error($response)) return "Request error: " . $response->get_error_message();

        $status = wp_remote_retrieve_response_code($response);
        if ($status == 429) {
            $headers = wp_remote_retrieve_headers($response);
            $reset_timestamp = isset($headers['x-rate-limit-reset']) ? intval($headers['x-rate-limit-reset']) : null;
            if ($reset_timestamp) {
                $time_left = $reset_timestamp - time();
                $formatted = wp_date('Y-m-d H:i:s', (int) $reset_timestamp);
                return "Rate limit exceeded. Will reset at: {$formatted} (" . gmdate("H:i:s", $time_left) . " remaining)";
            } else {
                return "Rate limit exceeded. No reset time provided.";
            }
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($body['data']['id'])) return "Invalid response or user not found.";

        return [
            'id' => $body['data']['id'],
            'name' => $body['data']['name'] ?? '',
            'username' => $body['data']['username'] ?? '',
            'avatar_url' => $body['data']['profile_image_url'] ?? '',
        ];
    }
}
