<?php
defined( 'ABSPATH' ) || exit( 'No direct script access allowed' );
/**
 * Plugin Name: Iran Phone Format For GForms
 * Description: Adds Iranian mobile phone number format and validation to Gravity Forms. Supports both Persian and English digits.
 * Version: 1.0.4
 * Author: Hafez Bahari
 * Author URI: https://hafezbahari.ir
 * License: GPLv2 or later
 * Text Domain: iran-phone-format-for-gforms
 * 
 * 
 * Iran Phone Format for Gravity Forms plugin adds a new "Iran" phone format option to Gravity Forms Phone field.
 * It supports both Persian and English digits, automatically normalizing the input before validation.
 *
 * Features:
 * - Adds Iran phone format option in Gravity Forms
 * - Regex-based validation for mobile numbers starting with 09
 * - Converts Persian digits to English before validation
 * - Works both with Persian and English digits
 * - Customizable error message (translatable)
 *
 * Example valid number: 09123456789 or ۰۹۱۲۳۴۵۶۷۸۹
 * 
*/

add_action('plugins_loaded', 'ipffg_gf_iran_phone_format_init');

function ipffg_gf_iran_phone_format_init() {
    if (!class_exists('GFForms')) {
        return;
    }

    // Add the Iran phone format
    add_filter('gform_phone_formats', 'ipffg_add_iran_phone_format');
    
    // Validate the Iran phone format
    add_filter('gform_field_validation', 'ipffg_validate_iran_phone_format', 10, 4);
}

function ipffg_add_iran_phone_format($phone_formats) {
    $phone_formats['iran'] = array(
        'label'       => 'Iran',
        'mask'        => false,
        'regex'       => '/^[0۰][9۹][0-9۰-۹]{9}$/',
        'instruction' => '',
    );
    return $phone_formats;
}

function ipffg_validate_iran_phone_format($result, $value, $form, $field) {
    // Check if the field is a phone field and the format is 'Iran'.
    if ($field->type == 'phone' && $field->phoneFormat == 'iran') {
        // Convert Persian digits to English
        $persian_digits = array('۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹');
        $english_digits = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9');
        $value = str_replace($persian_digits, $english_digits, $value);

        // Remove any non-digit characters
        $value = preg_replace('/[^0-9]/', '', $value);

        if (!preg_match('/^09[0-9]{9}$/', $value)) {
            $result['is_valid'] = false;
            $result['message'] = __('شماره موبایل را در این فرمت وارد کنید: ۰۹۱۲۳۴۵۶۷۸۹', 'iran-phone-format-for-gforms');
        } else {
            $result['is_valid'] = true;
            $result['value'] = $value; // Store the normalized value
        }
    }
    return $result;
}