<?php
/*
Plugin Name: IP Logger for WP Job Manager
Description: Logs the IP address of job posters in WP Job Manager.
Version: 1.0
Author: Rohan Sadanandan
Author URI: https://rohansadanandan.com
License: GPLv2 or later
Text Domain: ip-logger-for-wp-job-manager
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class IP_Logger_For_WP_Job_Manager {

    /**
     * Constructor to initialize hooks and actions.
     */
    public function __construct() {
        add_action( 'job_manager_job_submitted', [ $this, 'save_job_submission_ip' ] );
        add_action( 'add_meta_boxes', [ $this, 'add_job_listing_ip_meta_box' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'ilfwpjm_enqueue_admin_styles' ] );
    }

    /**
     * Get the IP address of the user submitting the job.
     *
     * @return string User's IP address.
     */
    
    public function get_user_ip_address() {
        $ip_keys = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR'
        );

        foreach ( $ip_keys as $key ) {
            if ( isset( $_SERVER[ $key ] ) ) {
                $raw_ip_value = wp_unslash( $_SERVER[ $key ] );
                $ip_list = explode( ',', $raw_ip_value );
                foreach ( $ip_list as $ip ) {
                    $ip = trim( $ip );
                    if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                        return sanitize_text_field( $ip );
                    }
                }
            }
        }

        return esc_html__( 'IP not found', 'ip-logger-for-wp-job-manager' );
    }


    /**
     * Save the IP address of the user submitting the job.
     *
     * @param int $job_id The ID of the job listing.
     */
    public function save_job_submission_ip( $job_id ) {
        $ip_address = $this->get_user_ip_address();
        update_post_meta( $job_id, '_job_submission_ip', $ip_address );
    }

    /**
     * Add a meta box to the job listing to display the IP address.
     */
    public function add_job_listing_ip_meta_box() {
        add_meta_box(
            'job_listing_ip',
            __( 'Job Submission IP Address', 'ip-logger-for-wp-job-manager' ),
            [ $this, 'display_job_submission_ip' ],
            'job_listing',
            'side',
            'default'
        );
    }

    /**
     * Display the IP address in the custom meta box.
     *
     * @param WP_Post $post The post object.
     */
    public function display_job_submission_ip( $post ) {
        $ip_address = get_post_meta( $post->ID, '_job_submission_ip', true );
        if ( $ip_address ) {
            echo '<div class="ilfwpjm-ip-box">' . esc_html( $ip_address ) . '</div>';
        } else {
            echo '<em>' . esc_html__( 'No IP address recorded.', 'ip-logger-for-wp-job-manager' ) . '</em>';
        }
    }
    /**
     * Enqueue admin CSS for IP display.
     */ 
    public function ilfwpjm_enqueue_admin_styles( $hook ) {
        global $post;

    // Only load styles on job listing edit screen.
    if ( in_array( $hook, ['post.php', 'post-new.php'], true ) && isset( $post ) && $post->post_type === 'job_listing' ) {
        wp_enqueue_style(
            'ilfwpjm-admin-style',
            plugin_dir_url( __FILE__ ) . 'assets/admin.css',
            [],
            '1.0'
        );
    }
}

}

new IP_Logger_For_WP_Job_Manager();
