<?php
namespace IntentDeep\VirtualFiles\Bootstrap;

use IntentDeep\VirtualFiles\Controllers\SettingsController;
use IntentDeep\VirtualFiles\Controllers\VirtualFilesController;
use IntentDeep\VirtualFiles\Controllers\AdminBarController;
use IntentDeep\VirtualFiles\Controllers\AnalyticsController;
use IntentDeep\VirtualFiles\Controllers\AiController;
use IntentDeep\VirtualFiles\Core\Asset;
use IntentDeep\VirtualFiles\Traits\ErrorHandler;

class Settings {
    use ErrorHandler;

    private const PLUGIN_PAGE = 'virtual-files';
    private const ERROR_PREFIX = 'Virtual Files';

    private SettingsController $settingsController;
    private VirtualFilesController $virtualFilesController;
    private AdminBarController $adminBarController;
    private AnalyticsController $analyticsController;
    private AiController $aiController;

    /**
     * Initialize the plugin settings
     */
    public function __construct() {
        $this->initializeControllers();
        $this->registerHooks();
        $this->registerAjaxHandlers();
        $this->registerAdminPostHandlers();
    }

    /**
     * Initialize controller instances
     */
    private function initializeControllers(): void {
        $this->settingsController = new SettingsController();
        $this->virtualFilesController = new VirtualFilesController();
        $this->adminBarController = new AdminBarController();
        $this->analyticsController = new AnalyticsController();
        $this->aiController = new AiController();
    }

    /**
     * Register WordPress hooks
     */
    private function registerHooks(): void {
        // Menu and Assets
        add_action('admin_menu', [$this->analyticsController, 'add_page']);
        add_action('admin_menu', [$this->settingsController, 'add_page']);
        add_action('admin_enqueue_scripts', [$this, 'assets'], 999);

        // Plugin Settings
        add_filter("plugin_action_links_" . VIRTUAL_FILES_PLUGIN_BASE, [$this, 'setting_link']);
        add_filter('script_loader_tag', [$this, 'add_module_to_script'], 10, 3);

        // Admin notices for admin bar actions
        add_action('admin_notices', [AdminBarController::class, 'display_admin_notices']);
    }

    /**
     * Register AJAX handlers
     */
    private function registerAjaxHandlers(): void {
        // Settings Actions
        $this->addAjaxAction('intentdeep_vf_save_options', [$this->settingsController, 'save_options']);
        $this->addAjaxAction('intentdeep_vf_flush_rewrite', [$this->settingsController, 'flushRewriteRules'], true);
        $this->addAjaxAction('intentdeep_vf_save_preference', [$this->settingsController, 'savePreference']);

        // Data Management Actions
        $this->addAjaxAction('intentdeep_vf_reset_plugin_data', [$this->settingsController, 'resetPluginData']);

        // AI Generation Action
        $this->addAjaxAction('intentdeep_vf_generate_ai_content', [$this->aiController, 'generate_content']);
        $this->addAjaxAction('intentdeep_vf_search_posts', [$this->aiController, 'search_posts']);

        // Cache Actions (Pro only)
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            $this->addAjaxAction('intentdeep_vf_get_cache_stats', [$this->settingsController, 'getCacheStats']);
            $this->addAjaxAction('intentdeep_vf_warm_up_cache', [$this->settingsController, 'warmUpCache']);
            $this->addAjaxAction('intentdeep_vf_clear_cache', [$this->settingsController, 'clearCache']);
            $this->addAjaxAction('intentdeep_vf_export', [$this->settingsController, 'exportFiles']);
            $this->addAjaxAction('intentdeep_vf_import', [$this->settingsController, 'importFiles']);

            // Analytics Actions (Pro only)
            $this->addAjaxAction('intentdeep_vf_refresh_analytics', [$this->analyticsController, 'ajax_refresh_analytics']);
        }
    }

    /**
     * Register admin-post handlers for admin bar actions
     */
    private function registerAdminPostHandlers(): void {
        // Admin bar actions (Pro only for cache actions)
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            add_action('admin_post_intentdeep_vf_clear_cache_admin_bar', [AdminBarController::class, 'handle_clear_cache_action']);
            add_action('admin_post_intentdeep_vf_warm_cache_admin_bar', [AdminBarController::class, 'handle_warm_cache_action']);
        }

        add_action('admin_post_intentdeep_vf_flush_rewrite_admin_bar', [AdminBarController::class, 'handle_flush_rewrite_action']);
    }

    /**
     * Add an AJAX action with error handling
     *
     * @param string $action The AJAX action name
     * @param callable $callback The callback function
     * @param boolean $nopriv Whether to add nopriv action
     */
    private function addAjaxAction(string $action, callable $callback, bool $nopriv = false): void {
        $hook = "wp_ajax_$action";
        
        add_action($hook, function() use ($action, $callback) {
            try {
                call_user_func($callback);
            } catch (\Exception $e) {
                intentdeep_vf_log("Error in action wp_ajax_{$action}: " . $e->getMessage());
                wp_send_json_error($e->getMessage());
            }
        });

        if ($nopriv) {
            add_action("wp_ajax_nopriv_$action", $callback);
        }
    }

    /**
     * Add settings link to plugins page
     */
    public function setting_link(array $links): array {
        array_unshift(
            $links, 
            sprintf('<a href="admin.php?page=%s">Settings</a>', self::PLUGIN_PAGE)
        );
        return $links;
    }

    /**
     * Enqueue assets for the plugin
     */
    public function assets(): void {
        if (!$this->is_plugin_page()) {
            return;
        }

        $this->enqueue_production_assets();
    }

    /**
     * Check if current page is plugin page
     */
    private function is_plugin_page(): bool {
        return isset($_GET['page'])
            && !empty($_GET['page'])
            && $_GET['page'] === 'intentdeep_vf_settings';
    }

    /**
     * Enqueue production assets
     */
    private function enqueue_production_assets(): void {
        Asset::style('intentdeep_vf__styles', 'admin/assets/css/settings.css');
        Asset::script('intentdeep_vf__main', 'admin/assets/js/settings.js', ['jquery'], true);
    }

    /**
     * Add module type to script tags for ES6 modules
     */
    public function add_module_to_script(string $tag, string $handle): string {
        $module_scripts = ['intentdeep_vf__script', 'intentdeep_vf__main', 'intentdeep_vf__client'];

        if (in_array($handle, $module_scripts)) {
            $tag = preg_replace('/type=[\'"]text\/javascript[\'"]/', '', $tag);
            return str_replace(' src', ' type="module" src', $tag);
        }

        return $tag;
    }
}