<?php
namespace IntentDeep\VirtualFiles\Traits;

trait VirtualFilesRenderer {
    
    /**
     * Render file details meta box
     */
    public function renderFileDetailsMetaBox($post) {
        wp_nonce_field('idep_virtual_file_save_meta', 'idep_virtual_file_nonce');

        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $status = get_post_meta($post->ID, '_vf_status', true);
        $mime_type = get_post_meta($post->ID, '_vf_mime_type', true);
        $access_count = get_post_meta($post->ID, '_vf_access_count', true);

        // Get allowed extensions
        $settings = $this->getSettings();
        $allowed_extensions = $this->getAllowedExtensions();
        ?>

        <div class="virtual-files-meta">
            <div class="field-group">
                <div class="vf-filename-wrapper">
                    <div class="vf-name-wrapper">
                        <label for="vf_filename" class="vf-name-label">
                            <?php esc_html_e('Path / File Name', 'intentdeep-virtual-files'); ?>
                            <span class="required-asterisk">*</span>
                        </label>
                        <input type="text" id="vf_filename" name="vf_filename" value="<?php echo esc_attr($this->getFilenameWithoutExtension($filename)); ?>" required placeholder="<?php esc_attr_e('robots or .well-known/security or api/v1/data', 'intentdeep-virtual-files'); ?>" />
                    </div>
                    <div class="vf-file-type-wrapper">
                        <label for="vf_extension" class="vf-extension-label"><?php esc_html_e('File Type', 'intentdeep-virtual-files'); ?></label>
                        <select id="vf_extension" name="vf_extension" class="vf-extension-dropdown">
                        <?php foreach ($this->getFileTypeOptions() as $extension => $data): ?>
                            <option value="<?php echo esc_attr($extension); ?>"
                                    data-mime="<?php echo esc_attr($data['mime']); ?>"
                                    data-pro="<?php echo esc_attr($data['pro'] ? 'true' : 'false'); ?>"
                                    <?php
                                    // Select current extension
                                    selected($this->getFileExtension($filename), $extension);
                                    // Disable if pro feature and user is not pro
                                    if ($data['pro'] && (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only())) {
                                        echo 'disabled';
                                    }
                                    ?>>
                                <?php
                                echo esc_html($data['label']);
                                if ($data['pro'] && (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only())) {
                                    echo ' 🔒';
                                }
                                ?>
                            </option>
                        <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <p class="description has-icon">
                    <?php
                    if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
                        esc_html_e('Enter file path without extension (e.g., robots for root or .well-known/security for nested paths). Select extension from dropdown. WordPress core paths are restricted.', 'intentdeep-virtual-files');
                    } else {
                        esc_html_e('Enter file path without extension (e.g., robots or api/data). Select extension from dropdown. Free version supports basic text files. Upgrade to Pro for advanced formats.', 'intentdeep-virtual-files');
                    }
                    ?>
                </p>
            </div>
    </div>

    <?php
    }

    /**
     * Render enhanced file info sidebar widget
     */
    public function renderEnhancedInfoSidebar($post) {
        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $status = get_post_meta($post->ID, '_vf_status', true);
        $mime_type = get_post_meta($post->ID, '_vf_mime_type', true);
        $content_size = strlen(get_post_meta($post->ID, '_vf_content', true));
        $modified_date = get_post_meta($post->ID, '_vf_modified', true);
        $access_count = get_post_meta($post->ID, '_vf_access_count', true);

        ?>
        <div class="vf-enhanced-sidebar">
            <!-- File Status Controls -->
            <div class="vf-sidebar-section">
                <label class="vf-sidebar-label">⚙️ <?php esc_html_e('File Settings', 'intentdeep-virtual-files'); ?></label>

                <!-- File Status Selector -->
                <div style="margin-bottom: 12px;">
                    <label class="vf-control-label" for="vf_status_sidebar"><?php esc_html_e('Status', 'intentdeep-virtual-files'); ?></label>
                    <select id="vf_status_sidebar" name="vf_status" class="vf-sidebar-select">
                        <option value="active" <?php selected($status, 'active'); ?>><?php esc_html_e('🟢 Active - File accessible via URL', 'intentdeep-virtual-files'); ?></option>
                        <option value="inactive" <?php selected($status, 'inactive'); ?>><?php esc_html_e('🔴 Inactive - File disabled', 'intentdeep-virtual-files'); ?></option>
                    </select>
                    <p class="vf-control-description">
                        <?php esc_html_e('Active files are accessible via their URL. Inactive files return 404.', 'intentdeep-virtual-files'); ?>
                    </p>
                </div>

                <!-- File Type (MIME) Display -->
                <div>
                    <label class="vf-control-label"><?php esc_html_e('MIME Type', 'intentdeep-virtual-files'); ?> <span class="vf-mime-indicator" style="font-size: 10px; margin-left: 6px;"><?php esc_html_e('auto-detected', 'intentdeep-virtual-files'); ?></span></label>
                    <div class="vf-mime-sidebar">
                        <span class="vf-mime-value"><?php echo esc_html($mime_type); ?></span>
                    </div>
                    <p class="vf-control-description">
                        <?php esc_html_e('Content type automatically detected from file extension.', 'intentdeep-virtual-files'); ?>
                    </p>
                </div>

            </div>

            <!-- Public URL (for active files) -->
            <?php if ($status === 'active' && !empty($filename)): ?>
            <div class="vf-sidebar-section">
                <label class="vf-sidebar-label">
                    🔗 <?php esc_html_e('Public URL', 'intentdeep-virtual-files'); ?>
                    <span class="vf-url-copy-feedback">✓ <?php esc_html_e('Copied!', 'intentdeep-virtual-files'); ?></span>
                </label>
                <input type="text" readonly value="<?php echo esc_url($this->getFileUrl($post, $filename)); ?>" class="vf-url-input" id="vf-sidebar-public-url" />
                <div class="vf-sidebar-buttons">
                    <button type="button" class="vf-sidebar-btn vf-copy-btn" data-target="vf-sidebar-public-url">
                        <span class="dashicons dashicons-admin-page"></span>&nbsp;<?php esc_html_e('Copy', 'intentdeep-virtual-files'); ?>
                    </button>
                    <a href="<?php echo esc_url($this->getFileUrl($post, $filename)); ?>" target="_blank" class="vf-sidebar-btn vf-visit-btn">
                        <span class="dashicons dashicons-visibility"></span>&nbsp;<?php esc_html_e('Visit', 'intentdeep-virtual-files'); ?>
                    </a>
                </div>
            </div>
            <?php endif; ?>

            <!-- Modified Date Display -->
                <div style="margin-bottom: 15px;">
                    <div class="vf-mime-sidebar">
                        <span class="vf-modified-date"><?php esc_html_e('Modified:', 'intentdeep-virtual-files'); ?> <?php echo $modified_date ? esc_html(human_time_diff(strtotime($modified_date), current_time('timestamp')) . ' ago') : esc_html__('Never', 'intentdeep-virtual-files'); ?></span>
                    </div>
                </div>

                                  <?php if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only() && $access_count): ?>
            <!-- Pro Analytics -->
            <div class="vf-sidebar-section">
                <label class="vf-sidebar-label">📊 <?php esc_html_e('Pro Analytics', 'intentdeep-virtual-files'); ?></label>
                <div class="vf-analytics-grid">
                    <div class="vf-analytic-item">
                        <span class="vf-analytic-number"><?php echo esc_html(number_format_i18n($access_count)); ?></span>
                        <span class="vf-analytic-label"><?php esc_html_e('Total Visits', 'intentdeep-virtual-files'); ?></span>
                    </div>
                    <div class="vf-analytic-item">
                        <span class="vf-analytic-number"><?php echo esc_html($this->formatFileSize($content_size)); ?></span>
                        <span class="vf-analytic-label"><?php esc_html_e('File Size', 'intentdeep-virtual-files'); ?></span>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render Pro features sidebar widget
     */
    public function renderProFeaturesSidebar($post) {
        ?>
        <div class="vf-pro-sidebar">
            <span class="pro-icon">🚀</span>
            <h3><?php esc_html_e('Upgrade to Pro', 'intentdeep-virtual-files'); ?></h3>

            <?php
            // Show current usage
            $active_files = $this->getActiveFileCount();
            $max_files = 5;
            ?>

            <div class="limit-info">
                <?php
                if ($active_files > 5) {
                            printf(
                                /* translators: %1$d: number of active files */
                                esc_html__('You have %1$d active files. Only 5 most recently updated files will be accessible on frontend.', 'intentdeep-virtual-files'),
                                esc_html($active_files)
                            );
                } else {
                        printf(
                            /* translators: %1$d: number of active files, %2$d: maximum number of files */
                            esc_html__('You have %1$d of %2$d active files.', 'intentdeep-virtual-files'),
                            esc_html($active_files),
                            esc_html($max_files)
                        );
                }
                ?>
            </div>

            <ul>
                <li><?php esc_html_e('Unlimited virtual files', 'intentdeep-virtual-files'); ?></li>
                <li><?php esc_html_e('Advanced code editor', 'intentdeep-virtual-files'); ?></li>
                <li><?php esc_html_e('JSON-LD, XML, RSS, CSV, YAML support', 'intentdeep-virtual-files'); ?></li>
                <li><?php esc_html_e('File access analytics', 'intentdeep-virtual-files'); ?></li>
                <li><?php esc_html_e('Priority support', 'intentdeep-virtual-files'); ?></li>
            </ul>

            <a href="<?php echo esc_url(admin_url('edit.php?post_type=idep_virtual_file&page=intentdeep_vf_settings-pricing')); ?>" class="upgrade-btn">
                <?php esc_html_e('⭐ Upgrade', 'intentdeep-virtual-files'); ?>
            </a>
        </div>
        <?php
    }

    /**
     * Render file URL metabox
     */
    public function renderFileLinkMetaBox($post) {
        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $status = get_post_meta($post->ID, '_vf_status', true);

        if ($status !== 'active' || empty($filename)) {
            echo '<p>' . esc_html__('File URL will be available once the file is published and active.', 'intentdeep-virtual-files') . '</p>';
            return;
        }

        $file_url = $this->getFileUrl($post, $filename);
        ?>

        <div class="vf-file-link-display">
            <input type="text" class="vf-file-link-input" value="<?php echo esc_url($file_url); ?>" readonly onclick="this.select();" />
        </div>

        <div class="vf-file-link-actions">
            <a href="<?php echo esc_url($file_url); ?>" class="vf-link-primary" target="_blank">
                <?php esc_html_e('Visit File', 'intentdeep-virtual-files'); ?> ↗
            </a>
            <a href="#" class="vf-link-secondary vf-copy-url-btn" data-url="<?php echo esc_attr($file_url); ?>">
                <?php esc_html_e('Copy URL', 'intentdeep-virtual-files'); ?> 📋
            </a>
        </div>
        <?php
    }

    /**
     * Render file content meta box
     */
    public function renderFileContentMetaBox($post) {
        $content = get_post_meta($post->ID, '_vf_content', true);
        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $status = get_post_meta($post->ID, '_vf_status', true);
        $extension = $this->getFileExtension($filename);

        // Nonce is already added in renderFileDetailsMetaBox, no need to duplicate it
        ?>

        <div class="virtual-files-content-wrapper">
        <?php
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            // Premium editor with syntax highlighting
            ?>
            <div class="virtual-files-editor-wrapper">
                <div style="display: flex; align-items: center; margin-bottom: 12px; gap: 12px;">
                    <h4 style="margin: 0; font-size: 14px; color: #1e293b;">
                        <?php esc_html_e('Advanced Code Editor', 'intentdeep-virtual-files'); ?>
                    </h4>
                    <span class="vf-pro-badge">PRO</span>
                    <div style="margin-left: auto; display: flex; gap: 8px;">
                        <button type="button" id="vf-ai-generate-btn" class="button vf-ai-btn">
                            <span class="dashicons dashicons-superhero"></span> <?php esc_html_e('Generate with AI', 'intentdeep-virtual-files'); ?>
                        </button>
                        <button type="button" id="vf-ai-improve-btn" class="button vf-ai-btn vf-ai-improve-btn">
                            <span class="dashicons dashicons-star-filled"></span> <?php esc_html_e('Improve with AI', 'intentdeep-virtual-files'); ?>
                        </button>
                        <button type="button" id="vf-ai-analyze-btn" class="button vf-ai-btn vf-ai-analyze-btn">
                            <span class="dashicons dashicons-search"></span> <?php esc_html_e('Analyze', 'intentdeep-virtual-files'); ?>
                        </button>
                    </div>
                </div>
                <div class="vf-editor-loading" style="display: none;">
                    <?php esc_html_e('Loading advanced editor...', 'intentdeep-virtual-files'); ?>
                </div>
                <textarea id="vf_content" name="vf_content" class="vf-code-editor" rows="20"><?php echo esc_textarea($content); ?></textarea>
            </div>
            <?php
            // Add File Status Check for Pro users
            $this->renderFileStatusCheck($post);
        } else {
            // Basic textarea for free users
            ?>
            <div class="virtual-files-editor-wrapper">
                <div style="display: flex; align-items: center; margin-bottom: 12px; gap: 12px;">
                    <h4 style="margin: 0; font-size: 14px; color: #1e293b;">
                        <?php esc_html_e('Simple Editor', 'intentdeep-virtual-files'); ?>
                    </h4>
                    <a href="<?php echo esc_url(admin_url('edit.php?post_type=idep_virtual_file&page=intentdeep_vf_settings-pricing')); ?>" style="text-decoration: none;">
                        <span class="vf-pro-badge" style="cursor: pointer; background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                            <?php esc_html_e('UPGRADE FOR CODE EDITOR', 'intentdeep-virtual-files'); ?>
                        </span>
                    </a>
                    <div style="margin-left: auto; display: flex; gap: 8px;">
                        <button type="button" id="vf-ai-generate-btn" class="button vf-ai-btn">
                            <span class="dashicons dashicons-superhero"></span> <?php esc_html_e('Generate with AI', 'intentdeep-virtual-files'); ?>
                        </button>
                        <button type="button" id="vf-ai-improve-btn" class="button vf-ai-btn vf-ai-improve-btn">
                            <span class="dashicons dashicons-star-filled"></span> <?php esc_html_e('Improve with AI', 'intentdeep-virtual-files'); ?>
                        </button>
                        <button type="button" id="vf-ai-analyze-btn" class="button vf-ai-btn vf-ai-analyze-btn">
                            <span class="dashicons dashicons-search"></span> <?php esc_html_e('Analyze', 'intentdeep-virtual-files'); ?>
                        </button>
                    </div>
                </div>
                <textarea id="vf_content" name="vf_content" rows="20" class="large-text" style="width: 100%; max-width: 100%; border: 2px solid #e2e8f0; border-radius: 12px; padding: 16px 20px; font-family: 'SF Mono', Monaco, 'Cascadia Code', 'Roboto Mono', Consolas, monospace; font-size: 14px; line-height: 1.6;"><?php echo esc_textarea($content); ?></textarea>
                <p class="description" style="margin-top: 12px; font-size: 13px; color: #64748b; line-height: 1.5;">
                    <?php esc_html_e('Free version supports plain text editing. ', 'intentdeep-virtual-files'); ?>
                    <a href="<?php echo esc_url(admin_url('edit.php?post_type=idep_virtual_file&page=intentdeep_vf_settings-pricing')); ?>" style="color: #3b82f6; text-decoration: none; font-weight: 500;">
                        <?php esc_html_e('Upgrade to Pro', 'intentdeep-virtual-files'); ?>
                    </a>
                    <?php esc_html_e('for syntax highlighting, dark themes, line numbers, and advanced editing features.', 'intentdeep-virtual-files'); ?>
                </p>

                <!-- File Status Check - Moved from sidebar to content metabox -->
                <?php $this->renderFileStatusCheck($post); ?>
            </div>
            <?php
        }
        ?>

        <!-- AI Generation Modal -->
        <div id="vf-ai-modal" class="vf-modal" style="display: none;">
            <div class="vf-modal-overlay"></div>
            <div class="vf-modal-content">
                <div class="vf-modal-header">
                    <h3>
                        <span class="dashicons dashicons-superhero" style="margin-right: 8px;"></span>
                        <?php esc_html_e('Generate Content with AI', 'intentdeep-virtual-files'); ?>
                    </h3>
                    <button type="button" class="vf-modal-close">&times;</button>
                </div>
                <div class="vf-modal-body">
                    <p class="description">
                        <?php esc_html_e('Describe what you want to generate. The AI will use your file extension to determine the format.', 'intentdeep-virtual-files'); ?>
                    </p>

                    <!-- Quick Prompts Dropdown -->
                    <div style="margin-bottom: 10px;">
                        <select id="vf-ai-quick-prompts" class="regular-text" style="width: 100%;">
                            <option value=""><?php esc_html_e('-- Quick Prompts (Optional) --', 'intentdeep-virtual-files'); ?></option>
                            <option value="Create a comprehensive, SEO-optimized robots.txt file for WordPress following current best practices. Include XML sitemap reference, block bad bots and scrapers, protect sensitive WordPress directories, allow all legitimate search engines, and add helpful comments explaining each section."><?php esc_html_e('Robots.txt - SEO Optimized (Recommended)', 'intentdeep-virtual-files'); ?></option>
                            <option value="Generate a professional security.txt file that fully complies with RFC 9116 standard. Include all recommended fields: Contact, Expires, Encryption, Preferred-Languages, Canonical, Policy, Hiring, and Acknowledgments. Use proper formatting and provide a security contact email."><?php esc_html_e('Security.txt - RFC 9116 Complete', 'intentdeep-virtual-files'); ?></option>
                            <option value="Create a comprehensive llms.txt file that guides AI systems (ChatGPT, Claude, Perplexity, SearchGPT, Google Gemini) on how to properly index, cite, and use content from this WordPress site. Include instructions for attribution, allowed use cases, content freshness indicators, and citation preferences."><?php esc_html_e('llms.txt - AI Crawler Guidelines', 'intentdeep-virtual-files'); ?></option>
                            <option value="Generate a complete ads.txt file for Google AdSense and other ad networks following IAB standards. Include proper formatting, publisher verification, authorized sellers, and comments explaining each entry. Ask me for my publisher ID if needed."><?php esc_html_e('Ads.txt - AdSense & Ad Networks', 'intentdeep-virtual-files'); ?></option>
                            <option value="Create an SEO-optimized robots.txt specifically for WooCommerce stores. Block checkout, cart, and account pages from indexing while ensuring product pages, categories, and shop pages are crawlable. Include XML sitemap reference, bad bot blocking, and WooCommerce-specific best practices."><?php esc_html_e('Robots.txt - WooCommerce Optimized', 'intentdeep-virtual-files'); ?></option>
                            <option value="Generate a comprehensive JSON-LD structured data schema for a local business. Include Organization schema with full NAP (Name, Address, Phone), business hours, social media profiles, logo, geographic coordinates, service areas, and any other relevant local SEO markup following schema.org standards."><?php esc_html_e('JSON-LD - Local Business Complete', 'intentdeep-virtual-files'); ?></option>
                            <option value="Create an app-ads.txt file for mobile app monetization following IAB standards. Include proper app store identifiers (Google Play, Apple App Store), authorized sellers, and publisher verification for mobile ad networks."><?php esc_html_e('App-ads.txt - Mobile App Monetization', 'intentdeep-virtual-files'); ?></option>
                        </select>
                    </div>

                    <textarea id="vf-ai-prompt" class="large-text" rows="3" placeholder="<?php esc_attr_e('E.g., Create a robots.txt that allows all bots but blocks the /private directory.', 'intentdeep-virtual-files'); ?>"></textarea>

                    <!-- Context Section -->
                    <div class="vf-ai-context-section" style="margin-top: 15px;">
                        <label style="font-weight: 600; font-size: 12px; color: #64748b; margin-bottom: 5px; display: block;"><?php esc_html_e('Add Context (Optional):', 'intentdeep-virtual-files'); ?></label>
                        <div class="vf-context-search-wrapper" style="position: relative;">
                            <input type="text" id="vf-context-search" class="regular-text" style="width: 100%;" placeholder="<?php esc_attr_e('Search posts, pages, or products to attach...', 'intentdeep-virtual-files'); ?>">
                            <div id="vf-context-results" class="vf-search-results" style="display: none;"></div>
                        </div>
                        <div id="vf-attached-items" class="vf-attached-items">
                            <!-- Chips will be added here -->
                        </div>
                        <p class="description" style="font-size: 11px; margin-top: 4px;"><?php esc_html_e('Attach up to 3 items to provide specific content context to the AI.', 'intentdeep-virtual-files'); ?></p>
                    </div>

                    <div id="vf-ai-stream-container" style="display: none; margin-top: 15px;">
                        <label style="font-weight: 600; font-size: 12px; color: #64748b; margin-bottom: 5px; display: block;"><?php esc_html_e('Generated Content:', 'intentdeep-virtual-files'); ?></label>
                        <div id="vf-ai-stream-output" class="vf-ai-stream-box"></div>
                    </div>

                    <div id="vf-ai-error" class="notice notice-error inline" style="display: none; margin-top: 10px;">
                        <p></p>
                    </div>
                </div>

                <div class="vf-modal-footer" id="vf-ai-initial-footer">
                    <div class="vf-ai-loading" style="display: none;">
                        <span class="spinner is-active" style="float: none; margin: 0 8px 0 0;"></span>
                        <?php esc_html_e('Generating...', 'intentdeep-virtual-files'); ?>
                    </div>
                    <div class="vf-modal-buttons">
                        <button type="button" class="button vf-modal-cancel"><?php esc_html_e('Cancel', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" id="vf-ai-submit" class="button button-primary"><?php esc_html_e('Generate', 'intentdeep-virtual-files'); ?></button>
                    </div>
                </div>

                <div class="vf-modal-footer" id="vf-ai-options-footer" style="display: none;">
                    <div style="font-weight: 600; color: #1e293b; margin-right: auto;"><?php esc_html_e('How to use this content?', 'intentdeep-virtual-files'); ?></div>
                    <div class="vf-modal-buttons">
                        <button type="button" class="button vf-ai-action-btn" data-action="replace"><?php esc_html_e('Replace All', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" class="button vf-ai-action-btn" data-action="insert"><?php esc_html_e('Insert at Cursor', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" class="button vf-ai-action-btn" data-action="append"><?php esc_html_e('Append', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" class="button vf-ai-action-btn" data-action="copy"><?php esc_html_e('Copy', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" class="button vf-modal-cancel" style="margin-left: 10px;"><?php esc_html_e('Cancel', 'intentdeep-virtual-files'); ?></button>
                    </div>
                </div>

                <div class="vf-modal-footer" id="vf-ai-analyze-footer" style="display: none;">
                    <div style="font-weight: 600; color: #1e293b; margin-right: auto;"><?php esc_html_e('Analysis Output', 'intentdeep-virtual-files'); ?></div>
                    <div class="vf-modal-buttons">
                        <button type="button" class="button vf-ai-action-btn" id="vf-download-analysis" data-action="download">
                            <span class="dashicons dashicons-download" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
                            <?php esc_html_e('Download Analysis', 'intentdeep-virtual-files'); ?>
                        </button>
                        <button type="button" class="button vf-ai-action-btn" data-action="copy"><?php esc_html_e('Copy', 'intentdeep-virtual-files'); ?></button>
                        <button type="button" class="button vf-modal-cancel" style="margin-left: 10px;"><?php esc_html_e('Close', 'intentdeep-virtual-files'); ?></button>
                    </div>
                </div>
            </div>
        </div>
        </div>
        <?php
    }

    /**
     * Render AI Generation History sidebar
     */
    public function renderAiHistorySidebar($post) {
        ?>
        <div class="vf-ai-history-sidebar">
            <p class="description" style="margin-top: 0; margin-bottom: 12px; font-size: 12px; color: #64748b;">
                <?php esc_html_e('Your recent AI generations are saved here. Click to reopen in modal.', 'intentdeep-virtual-files'); ?>
            </p>
            <div id="vf-ai-history-list" class="vf-history-list">
                <p style="text-align: center; color: #64748b; font-size: 12px; padding: 20px;">
                    <?php esc_html_e('Loading...', 'intentdeep-virtual-files'); ?>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Render File Status Check section (reusable for both free and pro versions)
     */
    private function renderFileStatusCheck($post) {
        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $status = get_post_meta($post->ID, '_vf_status', true);
        ?>
        <!-- File Status Check - Moved from sidebar to content metabox -->
        <div class="vf-status-section">
            <label style="font-size: 14px; font-weight: 600; color: #1e293b; margin-bottom: 10px; display: flex; align-items: center; gap: 6px;">
                ✅ <?php esc_html_e('File Status Check', 'intentdeep-virtual-files'); ?>
                <?php if (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only()): ?>
                <span class="vf-pro-status-badge"><?php esc_html_e('PRO ONLY', 'intentdeep-virtual-files'); ?></span>
                <?php endif; ?>
            </label>
            <div class="vf-validation-items">
                <div class="vf-validation-item <?php echo !empty($filename) ? 'valid' : 'invalid'; ?>">
                    <?php echo !empty($filename) ? '✓' : '⚠️'; ?> <?php esc_html_e('Filename Set', 'intentdeep-virtual-files'); ?>
                </div>
                <div class="vf-validation-item <?php echo !empty(get_post_meta($post->ID, '_vf_content', true)) ? 'valid' : 'invalid'; ?>">
                    <?php echo !empty(get_post_meta($post->ID, '_vf_content', true)) ? '✓' : '⚠️'; ?> <?php esc_html_e('Content Added', 'intentdeep-virtual-files'); ?>
                </div>
                <?php
                // Only show accessibility status if file has a filename
                if (!empty($filename)) {
                    // Comprehensive accessibility check for File Status Check
                    $user_has_pro = function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only();

                    // Initialize accessibility result
                    $accessibility_icon = '❓';
                    $accessibility_text = __('Unknown status', 'intentdeep-virtual-files');
                    $accessibility_class = 'warning';

                    // Check if file is published
                    $is_published = $post->post_status === 'publish';

                    // For unpublished posts, show pending status
                    if (!$is_published) {
                        $accessibility_icon = '⏳';
                        $accessibility_text = __('Not Published Yet', 'intentdeep-virtual-files');
                        $accessibility_class = 'warning';
                    }
                    // Check if file is active
                    elseif ($status !== 'active') {
                        $accessibility_icon = '🔴';
                        $accessibility_text = __('Disabled - Not Accessible', 'intentdeep-virtual-files');
                        $accessibility_class = 'warning';
                    }
                    // Apply user-specific restrictions
                    elseif (!$user_has_pro) {
                        // Free user - check 5 file limit
                        $is_accessible_for_free = $this->isFileAccessibleForFreeUserInAdmin($post->ID);
                        if ($is_accessible_for_free) {
                            $accessibility_icon = '🟢';
                            $accessibility_text = __('Live - Accessible via URL', 'intentdeep-virtual-files');
                            $accessibility_class = 'valid';
                        } else {
                            $accessibility_icon = '🔒';
                            $accessibility_text = __('Limit Reached - Not Accessible', 'intentdeep-virtual-files');
                            $accessibility_class = 'warning';
                        }
                    }
                    else {
                        // Pro user - file is accessible if it has filename and is active
                        $accessibility_icon = '🟢';
                        $accessibility_text = __('Live - Accessible via URL', 'intentdeep-virtual-files');
                        $accessibility_class = 'valid';
                    }
                    ?>
                    <div class="vf-validation-item <?php echo esc_attr($accessibility_class); ?>">
                        <?php echo esc_attr($accessibility_icon); ?> <?php echo esc_html($accessibility_text); ?>
                    </div>
                    <?php if (!$user_has_pro && !empty($filename) && $status === 'active' && $is_published && !$this->isFileAccessibleForFreeUserInAdmin($post->ID)): ?>
                    <div class="vf-lock-notification">
                        <strong><?php esc_html_e('Free User Limit:', 'intentdeep-virtual-files'); ?></strong> <?php esc_html_e('You have more than 5 active files. Only the 5 most recently updated files are accessible. Deactivate older files or upgrade to Pro for unlimited access.', 'intentdeep-virtual-files'); ?>
                    </div>
                    <?php endif; ?>
                <?php } ?>
            </div>
        </div>
        <?php
    }
}