<?php
namespace IntentDeep\VirtualFiles\Traits;

use IntentDeep\VirtualFiles\Core\Option;
use IntentDeep\VirtualFiles\Core\Request;

trait SettingHelper {
    
    /**
     * Sanitizes option values based on predefined rules
     *
     * @param array $options Array of options to be sanitized
     * @return array Sanitized array of options
     */
    public function sanitize_options(array $options): array {
        $sanitized = [];

        // Process boolean options
        $boolean_fields = [
            'routing_enabled',
            'remove_settings_on_uninstall',
            'advanced_editor',
            'caching_enabled',
            'analytics_enabled'
        ];
        foreach ($boolean_fields as $field) {
            if (isset($options[$field])) {
                $sanitized[$field] = filter_var($options[$field], FILTER_VALIDATE_BOOLEAN);
            } else {
                $sanitized[$field] = false;
            }
        }

        // Note: allowed_extensions is no longer configurable (set based on user type)
        // The field is not processed here as it's automatically determined

        // Process numeric options
        $numeric_fields = [
            'max_files_pro' => -1
        ];
        foreach ($numeric_fields as $field => $default) {
            if (isset($options[$field])) {
                $sanitized[$field] = intval($options[$field]);
            } else {
                $sanitized[$field] = $default;
            }
        }

        return $sanitized;
    }

    /**
     * Get plugin settings
     *
     * @return array Plugin settings array
     */
    public function getSettings(): array {
        return Option::all() ?: [];
    }

    /**
     * Get allowed file extensions for the current user
     * This is now fixed based on user type (not configurable)
     *
     * @return array Array of allowed extensions
     */
    public function getAllowedExtensions(): array {
        $is_pro = function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only();

        // Free users: txt, md, json
        // Pro users: all extensions (including both yml and yaml for YAML files)
        $extensions = $is_pro
            ? ['txt', 'md', 'json', 'jsonld', 'xml', 'rss', 'csv', 'yml', 'yaml', 'log']
            : ['txt', 'md', 'json'];

        return apply_filters('intentdeep_vf_allowed_extensions', $extensions);
    }

    /**
     * Get maximum number of files allowed for current user
     *
     * @return int Maximum number of files (-1 for unlimited)
     */
    public function getMaxFiles(): int {
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            $settings = $this->getSettings();
            return $settings['max_files_pro'] ?? -1;
        }

        // Free users always have a fixed limit of 5 files
        return 5;
    }

    /**
     * Get currently active virtual files count
     *
     * @return int Number of active files
     */
    public function getActiveFileCount(): int {
        $files = get_posts([
            'post_type' => 'idep_virtual_file',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'meta_query' => [
                [
                    'key' => '_vf_status',
                    'value' => 'active',
                    'compare' => '='
                ]
            ]
        ]);

        return count($files);
    }
}