<?php
namespace IntentDeep\VirtualFiles\Core;
use IntentDeep\VirtualFiles\Core\Plugin;

class Asset {
    private const ERROR_PREFIX = 'Virtual Files Asset';

    /**
     * Registers and enqueues a local stylesheet
     *
     * @param string $handle Handle name for the stylesheet
     * @param string $path Path to the stylesheet file
     * @param array $deps Dependencies
     * @param bool $inFooter Whether to load in footer
     */
    public static function style(string $handle, string $path, array $deps = [], bool $inFooter = false): void {
        try {
            $fullPath = self::getAssetPath($path);
            if (!file_exists($fullPath)) {
                // Silent fail for missing asset files - just log and continue
                intentdeep_vf_log(sprintf("%s Warning: Stylesheet file not found: %s", self::ERROR_PREFIX, $path));
                return;
            }

            $url = plugins_url($path, dirname(__DIR__));
            wp_enqueue_style($handle, $url, $deps, filemtime($fullPath));
        } catch (\Exception $e) {
            intentdeep_vf_log(sprintf("%s Error: %s", self::ERROR_PREFIX, $e->getMessage()));
        }
    }   

    /**
     * Registers and enqueues a remote stylesheet
     *
     * @param string $handle Handle name for the stylesheet
     * @param string $url Remote URL of the stylesheet
     */
    public static function style_remote(string $handle, string $url): void {
        try {
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                throw new \RuntimeException("Invalid remote stylesheet URL: {$url}");
            }

            wp_register_style($handle, esc_url($url), [], $version);
            wp_enqueue_style($handle);
        } catch (\Exception $e) {
            intentdeep_vf_log("Virtual Files Asset Error: {$e->getMessage()}");
        }
    }

    /**
     * Registers and enqueues a local script
     *
     * @param string $handle Handle name for the script
     * @param string $path Path to the script file
     * @param array $deps Dependencies
     * @param bool $inFooter Whether to load in footer
     */
    public static function script(string $handle, string $path, array $deps = [], bool $inFooter = false): void {
        try {
            $fullPath = self::getAssetPath($path);
            if (!file_exists($fullPath)) {
                // Silent fail for missing asset files - just log and continue
                intentdeep_vf_log(sprintf("%s Warning: Script file not found: %s", self::ERROR_PREFIX, $path));
                return;
            }

            $url = plugins_url($path, dirname(__DIR__));
            wp_enqueue_script($handle, $url, $deps, filemtime($fullPath), $inFooter);
        } catch (\Exception $e) {
            intentdeep_vf_log(sprintf("%s Error: %s", self::ERROR_PREFIX, $e->getMessage()));
        }
    }

    /**
     * Registers and enqueues a remote script
     *
     * @param string $handle Handle name for the script
     * @param string $url Remote URL of the script
     * @param array $dependencies Array of script dependencies
     * @param string|null $version Version number of the script
     * @param bool $footer Whether to enqueue in footer
     */
    public static function script_remote(
        string $handle,
        string $url,
        array $dependencies = [],
        $version = null,
        bool $footer = false
    ): void {
        try {
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                throw new \RuntimeException("Invalid remote script URL: {$url}");
            }

            wp_register_script(
                $handle,
                esc_url($url),
                $dependencies,
                $version,
                $footer
            );

            wp_enqueue_script($handle);
        } catch (\Exception $e) {
            intentdeep_vf_log("Virtual Files Asset Error: {$e->getMessage()}");
        }
    }

    /**
     * Get the full filesystem path for an asset
     *
     * @param string $path Relative path to the asset
     * @return string Full filesystem path
     */
    private static function getAssetPath(string $path): string {
        $pluginDir = plugin_dir_path(dirname(__DIR__)); // Get plugin root directory
        return $pluginDir . $path;
    }
}