<?php
namespace IntentDeep\VirtualFiles\Controllers;

use IntentDeep\VirtualFiles\Core\Plugin;
use IntentDeep\VirtualFiles\Core\Cache;
use IntentDeep\VirtualFiles\Traits\SettingHelper;
use IntentDeep\VirtualFiles\Core\RewriteRulesTrait;
use IntentDeep\VirtualFiles\Traits\VirtualFilesHelper;
use IntentDeep\VirtualFiles\Traits\VirtualFilesRenderer;
use IntentDeep\VirtualFiles\Traits\VirtualFilesValidator;
use IntentDeep\VirtualFiles\Traits\AnalyticsHelper;

class VirtualFilesController {
    use SettingHelper;
    use RewriteRulesTrait;
    use VirtualFilesHelper;
    use VirtualFilesRenderer;
    use VirtualFilesValidator;
    use AnalyticsHelper;

    /**
     * @var Cache
     */
    private $cache;

    public function __construct() {
        $this->cache = new Cache();

        add_action('init', [$this, 'registerPostType']);
        add_action('add_meta_boxes', [$this, 'addMetaBoxes']);
        add_action('save_post_idep_virtual_file', [$this, 'saveVirtualFileMeta'], 10, 1);
        add_action('save_post', [$this, 'saveVirtualFileMeta'], 10, 1);
        add_filter('manage_idep_virtual_file_posts_columns', [$this, 'setCustomColumns']);
        add_action('manage_idep_virtual_file_posts_custom_column', [$this, 'customColumnContent'], 10, 2);
        add_filter('post_row_actions', [$this, 'rowActions'], 10, 2);
        add_action('pre_get_posts', [$this, 'customizeAdminListOrdering']);
        add_action('restrict_manage_posts', [$this, 'addExtensionFilterDropdown']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdminScripts']);
        add_action('init', [$this, 'registerRewriteRules'], 1);
        add_action('template_redirect', [$this, 'handleVirtualFileRequest']);
        add_action('admin_notices', [$this, 'showStoredMessages']);
        add_action('init', [$this, 'checkRewriteFlush'], 999);
        add_action('wp_insert_post_data', [$this, 'limitVirtualFileCreation'], 10, 2);

        // Cache invalidation hooks
        add_action('delete_post', [$this, 'invalidateFileCache']);
        add_action('wp_trash_post', [$this, 'invalidateFileCache']);
        add_action('untrash_post', [$this, 'invalidateFileCache']);

        // AJAX handler for dismissing limit banner
        add_action('wp_ajax_intentdeep_vf_dismiss_limit_banner', [$this, 'dismissLimitBanner']);

        // AJAX handlers for AI generation history
        add_action('wp_ajax_intentdeep_vf_save_ai_generation', [$this, 'saveAiGeneration']);
        add_action('wp_ajax_intentdeep_vf_get_ai_history', [$this, 'getAiHistory']);
        add_action('wp_ajax_intentdeep_vf_delete_ai_generation', [$this, 'deleteAiGeneration']);
    }

    public function registerPostType() {
        register_post_type('idep_virtual_file', array(
            'labels' => array(
                'name' => __('Virtual Files', 'intentdeep-virtual-files'),
                'singular_name' => __('Virtual File', 'intentdeep-virtual-files'),
                'menu_name' => __('Virtual Files', 'intentdeep-virtual-files'),
                'add_new' => __('Add New File', 'intentdeep-virtual-files'),
                'add_new_item' => __('Add New Virtual File', 'intentdeep-virtual-files'),
                'edit_item' => __('Edit Virtual File', 'intentdeep-virtual-files'),
                'new_item' => __('New Virtual File', 'intentdeep-virtual-files'),
                'view_item' => __('View Virtual File', 'intentdeep-virtual-files'),
                'search_items' => __('Search Virtual Files', 'intentdeep-virtual-files'),
                'not_found' => __('No virtual files found', 'intentdeep-virtual-files'),
                'not_found_in_trash' => __('No virtual files found in trash', 'intentdeep-virtual-files'),
            ),
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_icon' => 'dashicons-text-page',
            'capability_type' => 'post',
            'hierarchical' => false,
            'supports' => array('title'),
            'has_archive' => false,
            'rewrite' => false,
            'publicly_queryable' => false,
        ));
    }

    public function addMetaBoxes() {
        add_meta_box(
            'idep_virtual_file_details',
            __('File Details', 'intentdeep-virtual-files'),
            [$this, 'renderFileDetailsMetaBox'],
            'idep_virtual_file',
            'normal',
            'high'
        );

        add_meta_box(
            'idep_virtual_file_content',
            __('File Content', 'intentdeep-virtual-files'),
            [$this, 'renderFileContentMetaBox'],
            'idep_virtual_file',
            'normal',
            'high'
        );

        // Add file link metabox for published files
        // Add enhanced file info sidebar widget for all users (moved to top with high priority)
        add_meta_box(
            'idep_virtual_file_enhanced_info',
            __('File Information', 'intentdeep-virtual-files'),
            [$this, 'renderEnhancedInfoSidebar'],
            'idep_virtual_file',
            'side',
            'high'
        );

        // Disable File URL metabox since it's now integrated into the enhanced info sidebar
        // add_meta_box(
        //     'virtual_file_link',
        //     __('File URL', 'intentdeep-virtual-files'),
        //     [$this, 'renderFileLinkMetaBox'],
        //     'idep_virtual_file',
        //     'side',
        //     'high'
        // );

        // Add AI History sidebar
        add_meta_box(
            'idep_virtual_file_ai_history',
            __('🤖 AI Generation History', 'intentdeep-virtual-files'),
            [$this, 'renderAiHistorySidebar'],
            'idep_virtual_file',
            'side',
            'default'
        );

        // Add Pro features sidebar widget for free users
        $fs_check = function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only();
        if (!$fs_check) {
            add_meta_box(
                'virtual_file_pro_features',
                __('Upgrade to Pro', 'intentdeep-virtual-files'),
                [$this, 'renderProFeaturesSidebar'],
                'idep_virtual_file',
                'side',
                'default'
            );
        }
    }



    public function saveVirtualFileMeta($post_id) {
        // Check if this is an idep_virtual_file post type
        $post_type = get_post_type($post_id);
        if ($post_type !== 'idep_virtual_file') {
            return;
        }

        // Basic security checks
        if (!isset($_POST['idep_virtual_file_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['idep_virtual_file_nonce'])), 'idep_virtual_file_save_meta')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Check file limit for free users
        if (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            $current_status = get_post_meta($post_id, '_vf_status', true);
            $new_status = isset($_POST['vf_status']) ? sanitize_text_field($_POST['vf_status']) : 'inactive';

            // Only check limit if trying to create a new active file or activate an existing one
            if (($current_status !== 'active' && $new_status === 'active') || get_post_status($post_id) === 'auto-draft') {
                $active_files = $this->getActiveFileCount();
                if ($active_files >= 5) {
                    // Don't allow more than 5 active files for free users
                    if (get_post_status($post_id) === 'auto-draft') {
                        // Prevent new post from being published
                        wp_update_post([
                            'ID' => $post_id,
                            'post_status' => 'draft'
                        ]);
                    }

                    // Set error message
                    set_transient('intentdeep_vf_error_' . get_current_user_id(),
                        __('Free version limited to 5 active files. Upgrade to Pro for unlimited files.', 'intentdeep-virtual-files'), 60);
                    return;
                }
            }
        }

        // Handle both new format (separate name + extension) and legacy format (full filename)
        if (isset($_POST['vf_filename']) && isset($_POST['vf_extension'])) {
            // New format: separate filename/path and extension
            $input_path = sanitize_text_field($_POST['vf_filename']);
            $dropdown_extension = sanitize_text_field($_POST['vf_extension']);

            // WordPress validation
            if (!empty($input_path) && !empty($dropdown_extension)) {
                // Normalize the path (remove leading/trailing slashes, handle multiple slashes)
                $normalized_path = $this->normalizePath($input_path);

                // Validate path against restricted WordPress paths
                $path_validation = $this->validateFilePath($normalized_path);
                if ($path_validation !== true) {
                    // Path validation failed - set error and return
                    set_transient('intentdeep_vf_error_' . get_current_user_id(), $path_validation, 60);
                    return;
                }

                // Parse the input to handle cases where user includes extension
                // E.g., "robots.txt" or ".well-known/security.txt" or just "robots"
                $path_info = pathinfo($normalized_path);
                $input_extension = isset($path_info['extension']) ? strtolower($path_info['extension']) : '';

                // Allowed extensions list
                $allowed_extensions = ['txt', 'md', 'json', 'jsonld', 'xml', 'rss', 'csv', 'yml', 'yaml', 'log'];

                // Determine the filename without extension
                // If user included a valid extension in input, remove it
                if (!empty($input_extension) && in_array($input_extension, $allowed_extensions)) {
                    // User included extension in input (e.g., "robots.txt")
                    $filename_without_ext = isset($path_info['dirname']) && $path_info['dirname'] !== '.'
                        ? $path_info['dirname'] . '/' . $path_info['filename']
                        : $path_info['filename'];
                } else {
                    // No extension in input or invalid extension
                    $filename_without_ext = $normalized_path;
                }

                // Use dropdown extension (validated)
                $extension = strtolower($dropdown_extension);

                if (in_array($extension, $allowed_extensions)) {
                    // Combine path/filename with extension
                    $filename = $filename_without_ext . '.' . $extension;

                    // Sanitize path while preserving slashes and leading dots
                    $filename = $this->sanitizeFilePath($filename);

                    // Additional filename validation
                    if ($this->isValidFilename($filename, $post_id)) {
                        // Save extension separately for filtering
                        update_post_meta($post_id, '_vf_extension', $extension);

                        // Save the data - store with path
                        update_post_meta($post_id, '_vf_filename', $filename);
                        update_post_meta($post_id, '_vf_mime_type', $this->detectMimeType($filename));

                        // Save status with validation
                        if (isset($_POST['vf_status']) && in_array($_POST['vf_status'], ['active', 'inactive'])) {
                            update_post_meta($post_id, '_vf_status', sanitize_text_field($_POST['vf_status']));
                        }

                        // Save content with appropriate sanitization
                        if (isset($_POST['vf_content'])) {
                            $content = $this->sanitizeVirtualFileContent($_POST['vf_content'], $extension);
                            update_post_meta($post_id, '_vf_content', $content);
                        }

                        // Update modified timestamp
                        update_post_meta($post_id, '_vf_modified', current_time('mysql'));

                        // Invalidate cache for this file
                        $this->invalidateCacheOnSave($post_id);

                        // Schedule rewrite rules flush
                        $this->scheduleRewriteFlush();
                    }
                }
            }
        } elseif (isset($_POST['vf_filename'])) {
            // Legacy format: full filename
            $filename = sanitize_text_field($_POST['vf_filename']);

            if (!empty($filename)) {
                // WordPress filename validation
                $filename = sanitize_file_name($filename);

                // Additional validation
                if ($this->isValidFilename($filename, $post_id)) {
                    // Extract and save extension separately for filtering
                    $extension = $this->getFileExtension($filename);
                    update_post_meta($post_id, '_vf_extension', $extension);

                    // Save the data with proper sanitization
                    update_post_meta($post_id, '_vf_filename', sanitize_file_name($filename));
                    update_post_meta($post_id, '_vf_mime_type', $this->detectMimeType($filename));

                    // Save status with validation
                    if (isset($_POST['vf_status']) && in_array($_POST['vf_status'], ['active', 'inactive'])) {
                        update_post_meta($post_id, '_vf_status', sanitize_text_field($_POST['vf_status']));
                    }

                    // Save content with appropriate sanitization
                    if (isset($_POST['vf_content'])) {
                        $content = $this->sanitizeVirtualFileContent($_POST['vf_content'], $extension);
                        update_post_meta($post_id, '_vf_content', $content);
                    }

                    // Update modified timestamp
                    update_post_meta($post_id, '_vf_modified', current_time('mysql'));

                    // Invalidate cache for this file
                    $this->invalidateCacheOnSave($post_id);

                    // Schedule rewrite rules flush
                    $this->scheduleRewriteFlush();
                }
            }
        }
    }

    
    // Helper methods are now in VirtualFilesHelper and VirtualFilesValidator traits



    /**
     * Check if rewrite rules need to be flushed
     */
    public function checkRewriteFlush() {
        if (get_transient('intentdeep_vf_flush_rewrite_rules')) {
            delete_transient('intentdeep_vf_flush_rewrite_rules');

            // Register rewrite rules and flush
            $this->registerRewriteRules();
            flush_rewrite_rules(false);
        }
    }

    /**
     * Show stored error/success messages from transients
     */
    public function showStoredMessages() {
        $user_id = get_current_user_id();

        // Show error message
        $error_message = get_transient('intentdeep_vf_error_' . $user_id);
        if ($error_message) {
            ?>
            <div class="notice notice-error is-dismissible">
                <p><?php echo esc_html($error_message); ?></p>
            </div>
            <?php
            delete_transient('intentdeep_vf_error_' . $user_id);
        }

        // Show file limit warning for free users
        $this->showFileLimitNotice();
    }

    /**
     * Show file limit notice when approaching or at limit
     */
    private function showFileLimitNotice() {
        // Check if banner has been dismissed
        if (get_transient('intentdeep_vf_limit_banner_dismissed')) {
            return;
        }

        // Only show on Virtual Files pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'idep_virtual_file') === false) {
            return;
        }

        // Don't show for Pro users
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            return;
        }

        // Get count of published virtual files
        $file_count = wp_count_posts('idep_virtual_file');
        $published_count = isset($file_count->publish) ? intval($file_count->publish) : 0;
        $max_files = 5; // Free user limit

        // Only show if at 4 or 5 files
        if ($published_count < 4) {
            return;
        }

        $is_at_limit = $published_count >= $max_files;

        $upgrade_url = function_exists('intentdeep_vf_fs')
            ? intentdeep_vf_fs()->get_upgrade_url()
            : admin_url('edit.php?post_type=idep_virtual_file&page=intentdeep_vf_settings-pricing');
        ?>
        <div class="vf-limit-banner" style="margin: 35px 20px 15px 0; padding: 14px 16px; position: relative; z-index: 1; width: calc(100% - 40px); box-sizing: border-box; background: <?php echo $is_at_limit ? 'linear-gradient(135deg, #dc2626 0%, #b91c1c 100%)' : 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)'; ?>; border-radius: 6px; color: white; box-shadow: 0 3px 8px <?php echo $is_at_limit ? 'rgba(220, 38, 38, 0.25)' : 'rgba(245, 158, 11, 0.25)'; ?>;">
            <div style="display: flex; align-items: center; justify-content: space-between;">
                <div style="display: flex; align-items: center; gap: 14px; flex: 1;">
                    <div style="font-size: 28px; flex-shrink: 0;">
                        <?php echo $is_at_limit ? '🚫' : '⚠️'; ?>
                    </div>
                    <div style="min-width: 0;">
                        <strong style="font-size: 15px; display: block; margin-bottom: 4px; line-height: 1.3;">
                        <?php if ($is_at_limit): ?>
                            <?php esc_html_e('You\'ve Reached the Free Plan Limit', 'intentdeep-virtual-files'); ?>
                        <?php else: ?>
                            <?php esc_html_e('Almost at Your File Limit!', 'intentdeep-virtual-files'); ?>
                        <?php endif; ?>
                    </strong>
                    <p style="margin: 0; opacity: 0.95; font-size: 13px; line-height: 1.5;">
                        <?php
                        if ($is_at_limit):
                            echo sprintf(
                                /* translators: %d: number of files used */
                                esc_html__('You have %d active files. Free plan is limited to 5 files. Upgrade to Pro for unlimited files, advanced caching, analytics, and more.', 'intentdeep-virtual-files'),
                                $published_count
                            );
                        else:
                            echo sprintf(
                                /* translators: %1$d: number of files used, %2$d: maximum number of files allowed */
                                esc_html__('You\'re using %1$d of %2$d files. Upgrade now to unlock unlimited files and premium features.', 'intentdeep-virtual-files'),
                                $published_count,
                                $max_files
                            );
                        endif;
                        ?>
                    </p>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; align-items: center;">
                    <a href="<?php echo esc_url($upgrade_url); ?>"
                       class="button"
                       style="background: white; border: none; color: <?php echo $is_at_limit ? '#dc2626' : '#f59e0b'; ?>; font-weight: 600; padding: 8px 18px; font-size: 13px; box-shadow: 0 2px 4px rgba(0,0,0,0.12); text-decoration: none; border-radius: 4px; transition: all 0.2s ease; white-space: nowrap; text-transform: uppercase; flex-shrink: 0;">
                        <?php esc_html_e('UPGRADE TO PRO', 'intentdeep-virtual-files'); ?> →
                    </a>
                    <button type="button" id="vf-dismiss-banner" class="button" style="background: transparent; border: 1px solid rgba(255,255,255,0.4); color: white; font-weight: 600; padding: 8px 16px; font-size: 13px; text-decoration: none; border-radius: 4px; transition: all 0.2s ease; white-space: nowrap; flex-shrink: 0;">
                        <?php esc_html_e('Later', 'intentdeep-virtual-files'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Dismiss limit banner via AJAX - sets 30 day transient
     */
    public function dismissLimitBanner(): void {
        // Check nonce
        if (!check_ajax_referer('intentdeep_vf_dismiss_limit_banner', 'nonce', false)) {
            wp_send_json_error("Invalid nonce", 419);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error("Unauthorized user", 403);
        }

        // Set transient for 14 days (1209600 seconds)
        set_transient('intentdeep_vf_limit_banner_dismissed', true, DAY_IN_SECONDS * 14);

        wp_send_json_success([
            'message' => __("Banner dismissed for 30 days", 'intentdeep-virtual-files')
        ]);
    }

    public function setCustomColumns($columns) {
        // Add Extension column before title (insert at position 1, after checkbox)
        $columns = array_merge(
            array_slice($columns, 0, 1, true),
            ['vf_extension' => __('Type', 'intentdeep-virtual-files')],
            array_slice($columns, 1, NULL, true)
        );

        // Add custom columns
        $columns['vf_filename'] = __('Filename', 'intentdeep-virtual-files');
        $columns['vf_status'] = __('Status', 'intentdeep-virtual-files');

        // Add accessibility indicator for all users
        $columns['vf_accessibility'] = '<span class="dashicons dashicons-visibility" title="' . esc_attr__('Live Accessibility Check', 'intentdeep-virtual-files') . '"></span>';

        $columns['vf_mime_type'] = __('MIME Type', 'intentdeep-virtual-files');

        // Add Views column for Pro users
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            $columns['vf_access_count'] = __('Views', 'intentdeep-virtual-files');
        }

        $columns['vf_modified'] = __('Last Modified', 'intentdeep-virtual-files');

        // Remove date column and add our own
        unset($columns['date']);
        return $columns;
    }

    public function customColumnContent($column, $post_id) {
        switch ($column) {
            case 'vf_filename':
                $filename = get_post_meta($post_id, '_vf_filename', true);
                echo $filename ? esc_html($filename) : '-';
                break;

            case 'vf_extension':
                $filename = get_post_meta($post_id, '_vf_filename', true);
                if ($filename) {
                    // Extract extension from filename
                    $extension = $this->getFileExtension($filename);

                    // Define extension colors (same as settings page)
                    $extension_colors = [
                        'txt' => '#10b981',
                        'md' => '#8b5cf6',
                        'json' => '#f59e0b',
                        'jsonld' => '#14b8a6',
                        'xml' => '#ef4444',
                        'csv' => '#3b82f6',
                        'yml' => '#ec4899',
                        'yaml' => '#ec4899',
                        'log' => '#6b7280',
                        'rss' => '#f97316',
                    ];
                    $color = $extension_colors[$extension] ?? '#64748b';

                    // Display colored extension tag
                    printf(
                        '<span class="vf-extension-tag" style="--tag-color: %s;">%s</span>',
                        esc_attr($color),
                        esc_html(strtoupper($extension))
                    );
                } else {
                    echo '-';
                }
                break;

            case 'vf_status':
                $status = get_post_meta($post_id, '_vf_status', true);
                $status = $status ?: 'inactive';
                $class = $status === 'active' ? 'vf-status-active' : 'vf-status-inactive';
                echo '<span class="vf-status ' . esc_attr($class) . '">' . esc_html(ucfirst($status)) . '</span>';
                break;

            case 'vf_accessibility':
                // Compact accessibility check for all users
                $user_has_pro = function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only();
                $filename = get_post_meta($post_id, '_vf_filename', true);
                $status = get_post_meta($post_id, '_vf_status', true);

                // Initialize accessibility result
                $icon = '❓';
                $reason = __('Unknown status', 'intentdeep-virtual-files');
                $accessible = false;

                // Check 1: File must have a filename
                if (empty($filename)) {
                    $icon = '⚠️';
                    $reason = __('No filename set', 'intentdeep-virtual-files');
                    $accessible = false;
                }
                // Check 2: File must be active
                elseif ($status !== 'active') {
                    $icon = '🔴';
                    $reason = __('File status is disabled', 'intentdeep-virtual-files');
                    $accessible = false;
                }
                // Check 3: Apply user-specific restrictions
                elseif (!$user_has_pro) {
                    // Free user - check 5 file limit
                    $is_accessible_for_free = $this->isFileAccessibleForFreeUserInAdmin($post_id);
                    if ($is_accessible_for_free) {
                        $icon = '🟢';
                        $reason = __('Live - Accessible (Free user - within 5 file limit)', 'intentdeep-virtual-files');
                        $accessible = true;
                    } else {
                        $icon = '🔒';
                        $reason = __('Disabled - Free user limit exceeded (more than 5 files)', 'intentdeep-virtual-files');
                        $accessible = false;
                    }
                }
                else {
                    // Pro user - file is accessible if it has filename and is active
                    $icon = '🟢';
                    $reason = __('Live - Accessible (Pro user)', 'intentdeep-virtual-files');
                    $accessible = true;
                }

                // Display compact accessibility indicator with tooltip
                $class_name = $accessible ? 'vf-accessible' : 'vf-inaccessible';
                echo '<span class="vf-accessibility-indicator ' . esc_attr($class_name) . '" title="' . esc_attr($reason) . '">' . esc_html($icon) . '</span>';
                break;

            case 'vf_mime_type':
                $mime_type = get_post_meta($post_id, '_vf_mime_type', true);
                echo $mime_type ? esc_html($mime_type) : '-';
                break;

            case 'vf_access_count':
                if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
                    $count = get_post_meta($post_id, '_vf_access_count', true);
                    echo esc_html($count ? number_format_i18n($count) : '0');
                }
                break;

            case 'vf_modified':
                $modified = get_post_meta($post_id, '_vf_modified', true);
                echo esc_html($modified ? date_i18n(get_option('date_format'), strtotime($modified)) : '-');
                break;
        }
    }

    /**
     * Add row actions for virtual files
     */
    public function rowActions($actions, $post) {
        if ($post->post_type === 'idep_virtual_file') {
            $filename = get_post_meta($post->ID, '_vf_filename', true);
            $status = get_post_meta($post->ID, '_vf_status', true);

            if ($status === 'active' && !empty($filename)) {
                $file_url = $this->getFileUrl($post, $filename);
                $actions['visit_file'] = sprintf(
                    '<a href="%s" target="_blank" style="color: #0073aa; text-decoration: none;">%s</a>',
                    esc_url($file_url),
                    __('Visit File', 'intentdeep-virtual-files')
                );
            }
        }

        return $actions;
    }

    public function enqueueAdminScripts($hook) {
        global $post_type;

        if ($post_type === 'idep_virtual_file' || $hook === 'post-new.php') {
            // Use file modification time for cache busting
            $css_file = plugin_dir_path(dirname(__FILE__)) . 'assets/css/admin.css';
            $js_file = plugin_dir_path(dirname(__FILE__)) . 'assets/js/admin.js';
            $virtual_files_css_file = plugin_dir_path(dirname(__FILE__)) . 'assets/css/virtual-files-admin.css';
            $css_version = file_exists($css_file) ? filemtime($css_file) : time();
            $js_version = file_exists($js_file) ? filemtime($js_file) : time();
            $virtual_files_css_version = file_exists($virtual_files_css_file) ? filemtime($virtual_files_css_file) : time();

            // Enqueue our admin styles
            wp_enqueue_style(
                'intentdeep-vf-admin',
                plugin_dir_url(dirname(__FILE__)) . 'assets/css/admin.css',
                [],
                $css_version
            );

            // Enqueue virtual files specific styles
            wp_enqueue_style(
                'intentdeep-vf-admin-styles',
                plugin_dir_url(dirname(__FILE__)) . 'assets/css/virtual-files-admin.css',
                ['intentdeep-vf-admin'],
                $virtual_files_css_version
            );

            // Enqueue our admin JavaScript first (required for localization)
            wp_enqueue_script(
                'intentdeep-vf-admin',
                plugin_dir_url(dirname(__FILE__)) . 'assets/js/admin.js',
                ['jquery', 'wp-codemirror'],
                $js_version,
                true
            );

            // Prepare code editor settings for pro users
            $code_editor_settings = false;
            if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
                // Get code editor settings from WordPress
                $code_editor_settings = wp_enqueue_code_editor([
                    'type' => 'text/plain',
                    'codemirror' => [
                        'indentUnit' => 2,
                        'tabSize' => 2,
                        'lineNumbers' => true,
                        'lineWrapping' => true,
                    ]
                ]);
            }

            // Localize script with Pro status and code editor settings
            wp_localize_script('intentdeep-vf-admin', 'intentdeepVfData', [
                'isPro' => function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only(),
                'nonce' => wp_create_nonce('intentdeep_vf_settings'),
                'aiNonce' => wp_create_nonce('intentdeep_vf_ai_generation'),
                'dismissBannerNonce' => wp_create_nonce('intentdeep_vf_dismiss_limit_banner'),
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'debug' => defined('WP_DEBUG') && WP_DEBUG,
                'codeEditorSettings' => $code_editor_settings
            ]);
        }
    }

    public function registerRewriteRules() {
        $settings = $this->getSettings();

        if (empty($settings['routing_enabled'])) {
            return;
        }

        $virtual_files = $this->getActiveVirtualFiles();
        $registered_filenames = [];

        foreach ($virtual_files as $file) {
            $filename = get_post_meta($file->ID, '_vf_filename', true);

            if (!empty($filename) && !in_array($filename, $registered_filenames)) {
                // Register rewrite rule for exact filename without trailing slash
                add_rewrite_rule(
                    '^' . preg_quote($filename, '/') . '$',
                    'index.php?idep_virtual_file=1&file_id=' . intval($file->ID),
                    'top'
                );

                // Create separate rule for with trailing slash
                add_rewrite_rule(
                    '^' . preg_quote($filename, '/') . '/$',
                    'index.php?idep_virtual_file=1&file_id=' . intval($file->ID),
                    'top'
                );

                $registered_filenames[] = $filename;
            }
        }
    }

    public function handleVirtualFileRequest() {
        $file_id = get_query_var('file_id');

        if (get_query_var('idep_virtual_file') && !empty($file_id)) {
            $this->serveVirtualFile($file_id);
            exit;
        }
    }

    private function serveVirtualFile($file_id) {
        $post = get_post($file_id);

        if (!$post || $post->post_type !== 'idep_virtual_file') {
            wp_die(esc_html__('Virtual file not found', 'intentdeep-virtual-files'), 404);
        }

        $status = get_post_meta($post->ID, '_vf_status', true);
        if ($status !== 'active') {
            wp_die(esc_html__('Virtual file is inactive', 'intentdeep-virtual-files'), 404);
        }

        $filename = get_post_meta($post->ID, '_vf_filename', true);
        $content = get_post_meta($post->ID, '_vf_content', true);
        $mime_type = get_post_meta($post->ID, '_vf_mime_type', true);

        if (empty($filename) || empty($content)) {
            wp_die(esc_html__('Virtual file content not found', 'intentdeep-virtual-files'), 404);
        }

        // Set proper headers
        header('Content-Type: ' . $mime_type . '; charset=utf-8');
        header('Content-Length: ' . strlen($content));
        header('Cache-Control: public, max-age=3600'); // Cache for 1 hour

        // Output the content
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaping is not applicable for virtual file content that must be served as raw data (robots.txt, XML sitemaps, JSON-LD, etc.). Security measures: (1) Only users with edit_post capability can create/modify files, (2) Content is sanitized on input via ContentSanitizationHelper::sanitizeVirtualFileContent() which removes PHP tags, script tags, iframes, and dangerous functions, (3) Files are served with proper MIME type headers that prevent script execution in browsers.
        echo $content;

        // Update access count for pro users (with enhanced analytics)
        if (function_exists('intentdeep_vf_fs') && intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            $this->trackFileViewAnalytics($post->ID);
        }
    }

    // Helper methods are now in VirtualFilesHelper trait

    // These methods are now in VirtualFilesHelper trait

    /**
     * Customize admin list ordering to match accessibility check logic
     */
    public function customizeAdminListOrdering($query) {
        // Only modify the main query on the virtual file admin page
        if (!is_admin() || !$query->is_main_query() || $query->get('post_type') !== 'idep_virtual_file') {
            return;
        }

        // Handle extension filter
        if (isset($_GET['vf_extension_filter']) && !empty($_GET['vf_extension_filter'])) {
            $extension = sanitize_text_field($_GET['vf_extension_filter']);
            $meta_query = $query->get('meta_query');
            if (!is_array($meta_query)) {
                $meta_query = [];
            }

            // Simple filter: check if _vf_filename ends with the extension using REGEXP
            // Pattern matches: filename ending with .extension (case-insensitive)
            $meta_query[] = [
                'key' => '_vf_filename',
                'value' => '\.' . preg_quote($extension, '/') . '$',
                'compare' => 'REGEXP'
            ];
            $query->set('meta_query', $meta_query);
        }

        // Order by post modified date instead of meta to show all posts
        $query->set('orderby', 'modified');
        $query->set('order', 'DESC');
    }

    /**
     * Invalidate cache for a specific file when it's saved/updated
     *
     * @param int $post_id The virtual file post ID
     * @return bool True if cache was invalidated successfully
     */
    private function invalidateCacheOnSave(int $post_id): bool {
        // Only invalidate cache if caching is enabled (Pro feature)
        if (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            return false;
        }

        // Check if caching is enabled in settings
        $settings = $this->getSettings();
        if (empty($settings['cache_enabled'])) {
            return false;
        }

        // Clear the cache for this specific file
        $cache = new Cache();
        return $cache->delete_file_cache($post_id);
    }

    /**
     * Invalidate cache when a file is deleted, trashed, or untrashed
     * Called by WordPress hooks: delete_post, wp_trash_post, untrash_post
     *
     * @param int $post_id The post ID being deleted/trashed/untrashed
     * @return void
     */
    public function invalidateFileCache(int $post_id): void {
        // Only process virtual files
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'idep_virtual_file') {
            return;
        }

        // Only invalidate cache if caching is enabled (Pro feature)
        if (!function_exists('intentdeep_vf_fs') || !intentdeep_vf_fs()->can_use_premium_code__premium_only()) {
            return;
        }

        // Check if caching is enabled in settings
        $settings = $this->getSettings();
        if (empty($settings['cache_enabled'])) {
            return;
        }

        // Clear the cache for this specific file
        $cache = new Cache();
        $cache->delete_file_cache($post_id);
    }

    /**
     * Add extension filter dropdown to admin list
     */
    public function addExtensionFilterDropdown(): void {
        $screen = get_current_screen();
        if (!$screen || $screen->post_type !== 'idep_virtual_file') {
            return;
        }

        $current_filter = isset($_GET['vf_extension_filter']) ? sanitize_text_field($_GET['vf_extension_filter']) : '';
        $file_types = $this->getFileTypeOptions();

        ?>
        <select name="vf_extension_filter" id="vf-extension-filter">
            <option value=""><?php esc_html_e('All File Types', 'intentdeep-virtual-files'); ?></option>
            <?php foreach ($file_types as $ext => $type): ?>
                <option value="<?php echo esc_attr($ext); ?>" <?php selected($current_filter, $ext); ?>>
                    <?php echo esc_html($type['label']); ?>
                    <?php if ($type['pro']): ?>
                        <span class="dashicons dashicons-lock" style="font-size: 12px; vertical-align: middle;"></span>
                    <?php endif; ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * AJAX handler to save AI generation to post meta
     */
    public function saveAiGeneration() {
        check_ajax_referer('intentdeep_vf_ai_generation', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intentdeep-virtual-files')]);
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : '';
        $prompt = isset($_POST['prompt']) ? sanitize_textarea_field($_POST['prompt']) : '';
        $content = isset($_POST['content']) ? wp_unslash($_POST['content']) : '';

        if (!$post_id || !$action_type || !$content) {
            wp_send_json_error(['message' => __('Invalid data.', 'intentdeep-virtual-files')]);
        }

        // Get existing history
        $history = get_post_meta($post_id, '_vf_ai_history', true);
        if (!is_array($history)) {
            $history = [];
        }

        // Add new generation
        $history[] = [
            'id' => uniqid('ai_', true),
            'action_type' => $action_type,
            'prompt' => $prompt,
            'content' => $content,
            'timestamp' => current_time('timestamp')
        ];

        // Limit to last 20 generations
        if (count($history) > 20) {
            $history = array_slice($history, -20);
        }

        update_post_meta($post_id, '_vf_ai_history', $history);

        wp_send_json_success(['message' => __('Generation saved.', 'intentdeep-virtual-files')]);
    }

    /**
     * AJAX handler to get AI generation history
     */
    public function getAiHistory() {
        check_ajax_referer('intentdeep_vf_ai_generation', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intentdeep-virtual-files')]);
        }

        $post_id = isset($_GET['post_id']) ? intval($_GET['post_id']) : 0;

        if (!$post_id) {
            wp_send_json_error(['message' => __('Invalid post ID.', 'intentdeep-virtual-files')]);
        }

        $history = get_post_meta($post_id, '_vf_ai_history', true);
        if (!is_array($history)) {
            $history = [];
        }

        // Reverse to show newest first
        $history = array_reverse($history);

        // Add time_ago to each item
        foreach ($history as &$item) {
            $item['time_ago'] = human_time_diff($item['timestamp'], current_time('timestamp')) . ' ago';
        }

        wp_send_json_success($history);
    }

    /**
     * AJAX handler to delete AI generation from history
     */
    public function deleteAiGeneration() {
        check_ajax_referer('intentdeep_vf_ai_generation', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intentdeep-virtual-files')]);
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $item_id = isset($_POST['item_id']) ? sanitize_text_field($_POST['item_id']) : '';

        if (!$post_id || !$item_id) {
            wp_send_json_error(['message' => __('Invalid data.', 'intentdeep-virtual-files')]);
        }

        $history = get_post_meta($post_id, '_vf_ai_history', true);
        if (!is_array($history)) {
            wp_send_json_error(['message' => __('No history found.', 'intentdeep-virtual-files')]);
        }

        // Remove the item with matching ID
        $history = array_filter($history, function($item) use ($item_id) {
            return $item['id'] !== $item_id;
        });

        // Reindex array
        $history = array_values($history);

        update_post_meta($post_id, '_vf_ai_history', $history);

        wp_send_json_success(['message' => __('Generation deleted.', 'intentdeep-virtual-files')]);
    }

}