var analyticsBingUET = (function () {
    'use strict';

    var config = {
        debug: false,
        bingUETTagId: null, // Your Bing UET Tag ID
    };

    /**
     * Bing UET Plugin
     * @param {object} pluginConfig - Plugin configuration
     * @param {string} pluginConfig.bingUETTagId - Your Bing UET Tag ID
     * @param {boolean} [pluginConfig.debug=false] - Enable debug logs
     * @return {object} Analytics plugin
     */
    function bingUET() {
        var pluginConfig = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

        return {
            name: 'bing-uet',
            config: Object.assign({}, config, pluginConfig),

            initialize: function initialize(_ref) {
                var config = _ref.config;
                var bingUETTagId = config.bingUETTagId;

                if (!bingUETTagId) {
                    throw new Error('No Bing UET Tag ID defined');
                }

                if (!scriptLoaded()) {
                    /* Load Bing UET script */
                    !(function (w, d, t) {
                        var uet = d.createElement(t);
                        uet.async = true;
                        uet.src = 'https://bat.bing.com/bat.js';
                        uet.onload = function () {
                            if (window.uetq) {
                                window.uetq.push('event', 'pageView', { pixelId: bingUETTagId });
                                if (config.debug) {
                                    console.log('Bing UET Tag initialized with ID:', bingUETTagId);
                                }
                            }
                        };
                        var s = d.getElementsByTagName(t)[0];
                        s.parentNode.insertBefore(uet, s);
                    })(window, document, 'script');
                }
            },

            page: function page(_ref2) {
                var payload = _ref2.payload;
                if (window.uetq) {
                    window.uetq.push('event', 'pageView', { pixelId: config.bingUETTagId });
                    if (config.debug) {
                        console.log('Bing PageView tracked:', payload);
                    }
                }
            },

            track: function track(_ref3) {
                var payload = _ref3.payload;
                var event = payload.event,
                    properties = payload.properties;

                if (window.uetq) {
                    window.uetq.push('event', event, properties);
                    if (config.debug) {
                        console.log('Bing Event tracked:', event, properties);
                    }
                }
            },

            identify: function identify(_ref4) {
                var payload = _ref4.payload;
                var userId = payload.userId,
                    traits = payload.traits;

                if (window.uetq) {
                    window.uetq.push('event', 'identify', { userId: userId, traits: traits });
                    if (config.debug) {
                        console.log('Bing user identified:', { userId: userId, traits: traits });
                    }
                }
            },

            loaded: function loaded() {
                return typeof window.uetq !== 'undefined';
            },
        };
    }

    /* Check if Bing UET script is already loaded */
    function scriptLoaded() {
        var scripts = document.querySelectorAll('script[src]');
        return Array.from(scripts).some(function (script) {
            return script.src.indexOf('https://bat.bing.com/bat.js') !== -1;
        });
    }

    var index = bingUET;

    return index;
})();
