<?php
/*
Plugin Name: Integrio
Description: Adds the integrio integration with Inn Styl'es booking platform for your WordPress site.
Version: 1.0.1
Author: Integrio
License: GPLv2
*/

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

function integrio_innstyle_scripts() {
  // Enqueue the InnStyle calendar JavaScript file
  wp_enqueue_script(
      'innstyle-integration', // Handle for the script
      'https://developer.innstyle.co.uk/calendar.js', // Script URL
      array(), // Dependencies, if any
      '1.0.0', // Version number (null if you don’t need versioning)
      array( 
        'strategy'  => 'defer',
        'in_footer' => false, // Note: This is the default value.
    ) // Load in footer (true for footer, false for header)
  );
}

// Hook to enqueue the script on the front end
add_action('wp_enqueue_scripts', 'integrio_innstyle_scripts');

// Create the Admin Menu Item and Settings Page
function integrio_plugin_menu() {
  add_menu_page(
      'Integrio Plugin Settings', // Page title
      'Integrio', // Menu title
      'manage_options', // Capability
      'integrio-settings', // Menu slug
      'integrio_settings_page', // Callback function
      'dashicons-admin-generic', // Icon (uses a WordPress Dashicon)
      26 // Position in the menu
  );
}
add_action('admin_menu', 'integrio_plugin_menu');

// Custom sanitization for subdomain
function integrio_sanitize_subdomain($subdomain) {
    // Remove any non-alphanumeric characters
    $clean_subdomain = preg_replace('/[^a-zA-Z0-9-]/', '', $subdomain);
    // Convert to lowercase
    $clean_subdomain = strtolower($clean_subdomain);
    return $clean_subdomain;
}

// Register Settings
function integrio_register_settings() {
    register_setting(
        'integrio-settings-group',
        'integrio_subdomain',
        array(
            'type' => 'string',
            'sanitize_callback' => 'integrio_sanitize_subdomain'
        )
    );
    
    register_setting(
        'integrio-settings-group',
        'integrio_license_key',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        )
    );
    
    register_setting(
        'integrio-settings-group',
        'integrio_email',
        array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email'
        )
    );
}

add_action('admin_init', 'integrio_register_settings');
add_action('admin_init', 'integrio_validate_license');

// Settings Page Content
function integrio_settings_page() {
  ?>
  <div class="wrap">
      <h1>Integrio Plugin Settings</h1>
      <form method="post" action="options.php">
            <?php
            settings_fields('integrio-settings-group'); // Security fields
            do_settings_sections('integrio-settings-group'); // Settings section
            ?>
            <table class="form-table">
                <!-- Subdomain Field -->
                <tr valign="top">
                    <th scope="row">Subdomain</th>
                    <td>
                        <input type="text" name="integrio_subdomain" value="<?php echo esc_attr(get_option('integrio_subdomain', 'urbanhaven')); ?>" placeholder="urbanhaven" />
                        <p class="description">Enter the subdomain to use for Integrio URLs (e.g., 'urbanhaven').</p>
                    </td>
                </tr>
                
                <!-- License Key Field -->
                <tr valign="top">
                    <th scope="row">License Key</th>
                    <td>
                        <input type="text" name="integrio_license_key" value="<?php echo esc_attr(get_option('integrio_license_key')); ?>" placeholder="Enter your license key" />
                        <p class="description">Enter your license key to activate the plugin.</p>
                    </td>
                </tr>
                
                <!-- Email Field -->
                <tr valign="top">
                    <th scope="row">Email</th>
                    <td>
                        <input type="email" name="integrio_email" value="<?php echo esc_attr(get_option('integrio_email')); ?>" placeholder="Enter your email" />
                        <p class="description">Provide your email for activation and updates.</p>
                    </td>
                </tr>

                <!-- Shortcodes Section -->
                <tr>
                    <th>Shortcodes</th>
                    <td>
                        <p>[integrio_calendar]</p>
                        <p>[integrio_custom_search_enquiry]</p>
                        <p>[integrio_custom_search_results]</p>
                        <p>[integrio_gift_voucher]</p>
                        <p>[integrio_single_bookable bookable_id="123"]</p>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
  </div>
  <?php
}

function integrio_validate_license() {
    $license_key = get_option('integrio_license_key');
    $email = get_option('integrio_email');

    // Skip if no license key or email is provided
    if (empty($license_key) || empty($email)) {
        return;
    }

    // Send data to the backend to validate the license (firebase function - GCP Integrio project)
    $response = wp_remote_post('https://license.integrio.co.za', [
        'method' => 'POST',
        'body' => [
            'license_key' => $license_key,
            'email' => $email,
        ]
    ]);

    // Mocked response (remove this when your backend is ready)
    // $mock_response = json_encode([
    //     'status' => 'success', // or 'error'
    //     'message' => 'Mocked license validation successful!', // Example message
    // ]);

    // Simulate a response from your backend
    // $response = $mock_response;

    if (is_wp_error($response)) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>Could not validate the license. Please check your internet connection or contact support.</p></div>';
        });
        return;
    }

    // TODO: Need to uncomment this once backend is added
    $body = json_decode(wp_remote_retrieve_body($response), true);
    // Mocked response (remove this when your backend is ready)
    // $body = json_decode($response, true);

    if ($body['status'] !== 'success') {
        add_action('admin_notices', function () use ($body) {
            echo '<div class="notice notice-error"><p>' . esc_html($body['message']) . '</p></div>';
        });
    } else {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-success"><p>License successfully validated!</p></div>';
        });
    }
}

function integrio_scripts() {
    $subdomain = esc_attr(get_option('integrio_subdomain', 'urbanhaven')); // Get the subdomain from settings

    wp_enqueue_script('innstyle-configs',
    plugins_url('js/integrio.js', __FILE__),
    array('innstyle-integration'),
    '1.0.0',
    array( 
        'strategy'  => 'defer',
        'in_footer' => false, // Note: This is the default value.
    ));

    
    wp_localize_script('innstyle-configs', 'integrioData', array(
        'subdomain' => $subdomain
    ));
}
add_action('wp_enqueue_scripts', 'integrio_scripts');
 
// ============================================================
// Shortcodes
// ============================================================

// Shortcode for Calendar Initialization
function integrio_calendar_shortcode() {
    wp_enqueue_script('innstyle-configs');
    return '<div id="integrio-calendar"></div>';
}

add_shortcode('integrio_calendar', 'integrio_calendar_shortcode');

// Shortcode Custom search enquiry bar
function integrio_custom_search_enquiry_shortcode() {
    wp_enqueue_script('innstyle-configs');
    return '<div id="integrio-custom-search-enquiry"></div>';
}
add_shortcode('integrio_custom_search_enquiry', 'integrio_custom_search_enquiry_shortcode');

// Shortcode for Custom Search Results
function integrio_custom_search_results_shortcode() {
    wp_enqueue_script('innstyle-configs');
    return '<div id="integrio-custom-search-results"></div>';
}
add_shortcode('integrio_custom_search_results', 'integrio_custom_search_results_shortcode');

// Shortcode for a single bookable
function integrio_single_bookable_shortcode($atts) {
    // Set default attribute values and extract user-provided values
    $atts = shortcode_atts(array('bookable_id' => 1 ), $atts);// Default bookable_id as a numeric value
    $bookable_id = intval($atts['bookable_id']); // Ensure bookable_id is an integer

    wp_enqueue_script('innstyle-configs');
    return sprintf('<div id="integrio-single-bookable" data-bookable-id="%s"></div>', esc_attr($bookable_id));

}
add_shortcode('integrio_single_bookable', 'integrio_single_bookable_shortcode');


// TODO: THIS STILL WORKS THE OLD WAY, NEED TO THINK IF WE WANT TO PUSH THIS CODE INTO JAVASCRIPT RATHER
// Shortcode for Gift Voucher Embed
function integrio_gift_voucher_shortcode() {
    $subdomain = esc_attr(get_option('integrio_subdomain', 'stayunique')); // Get the subdomain from settings

    return "
    <iframe frameborder='0' allowtransparency='true' scrolling='no' role='presentation' src='https://{$subdomain}.innstyle.co.uk/gift-vouchers' tabindex='-1' style='border: none !important; margin: 0px !important; padding: 0px !important; min-width: 100% !important; overflow: auto !important; display: block !important; min-height: 860px;'></iframe>
    <style>
        @media (max-width: 1180px) {
            iframe[src='https://{$subdomain}.innstyle.co.uk/gift-vouchers'] {
                min-height: 1220px !important;
            }
        }
        @media (max-width: 600px) {
            iframe[src='https://{$subdomain}.innstyle.co.uk/gift-vouchers'] {
                min-height: 1240px !important;
            }
        }
    </style>";
}
add_shortcode('integrio_gift_voucher', 'integrio_gift_voucher_shortcode');
