<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit; // Prevent direct access to the file.
}

require_once INTEGRAZO_WZ_PLUGIN_DIR . 'src/products/zoho_bigin/util.php'; // <- adjust path to your Bigin util if needed

// Check if the class already exists to avoid redeclaration.
if (!class_exists('INTEGRAZO_WZ_Zoho_Bigin')) {
    class INTEGRAZO_WZ_Zoho_Bigin
    {
        public $refresh_token;
        public $access_token;
        public $data_center; // 'zoho.com','zoho.in','zoho.eu','zoho.uk','zoho.jp','zoho.sa','zoho.com.cn','zoho.com.au','zohocloud.ca'
        public $account_id;

        public function __construct($account_details = [])
        {
            $this->refresh_token = $account_details['account_data']['refresh_token'] ?? null;
            // prefer 'dc' if that’s how you store it; fallback to 'data_center'
            $this->data_center   = $account_details['dc'] ?? ($account_details['data_center'] ?? 'zoho.com');
            $this->account_id    = $account_details['id'] ?? null;
            $this->access_token  = INTEGRAZO_WZ_Util::getAccessToken($this->account_id);
        }

        /**
         * Resolve endpoints (Accounts + API) for the current data center.
         * @return array{accounts_base:string, api_base:string}
         */
        private function resolve_endpoints()
        {
            $dc = strtolower(trim((string) $this->data_center));

            switch ($dc) {
                case 'zoho.in':
                    return ['accounts_base' => 'https://accounts.zoho.in',     'api_base' => 'https://www.zohoapis.in'];
                case 'zoho.eu':
                    return ['accounts_base' => 'https://accounts.zoho.eu',     'api_base' => 'https://www.zohoapis.eu'];
                case 'zoho.uk':
                    return ['accounts_base' => 'https://accounts.zoho.uk',     'api_base' => 'https://www.zohoapis.uk'];
                case 'zoho.jp':
                    return ['accounts_base' => 'https://accounts.zoho.jp',     'api_base' => 'https://www.zohoapis.jp'];
                case 'zoho.sa':
                    return ['accounts_base' => 'https://accounts.zoho.sa',     'api_base' => 'https://www.zohoapis.sa'];
                case 'zoho.com.au':
                    return ['accounts_base' => 'https://accounts.zoho.com.au', 'api_base' => 'https://www.zohoapis.com.au'];
                case 'zoho.com.cn':
                    return ['accounts_base' => 'https://accounts.zoho.com.cn', 'api_base' => 'https://www.zohoapis.com.cn'];
                case 'zohocloud.ca':
                    return ['accounts_base' => 'https://accounts.zohocloud.ca', 'api_base' => 'https://www.zohoapis.ca'];
                case 'zoho.com':
                default:
                    return ['accounts_base' => 'https://accounts.zoho.com',    'api_base' => 'https://www.zohoapis.com'];
            }
        }
        /**
         * Record action options for radio buttons.
         * Returns an ordered list of [ 'value' => string, 'label' => string, 'disabled' => bool, 'default' => bool ].
         */
        function record_action_options()
        {
            return array(
                array(
                    'value'    => 'add',
                    'label'    => 'Add New Record',
                    'disabled' => false,
                    'default'  => true   // This one will be selected by default
                ),
                array(
                    'value'    => 'add_or_update',
                    'label'    => 'Add or Update Existing Record',
                    'disabled' => true,
                    'default'  => false
                )
            );
        }
        /** Build a full Accounts URL. */
        private function build_accounts_url(string $path): string
        {
            $endpoints = $this->resolve_endpoints();
            return rtrim($endpoints['accounts_base'], '/') . '/' . ltrim($path, '/');
        }

        /** Build a full API URL (Bigin). */
        public function build_api_url(string $path): string
        {
            $endpoints = $this->resolve_endpoints();
            return rtrim($endpoints['api_base'], '/') . '/' . ltrim($path, '/');
        }

        // ---------------- ACCESS TOKEN ----------------

        /** Refresh access token using refresh token */
        public function get_access_token()
        {
            $url = $this->build_accounts_url('/oauth/v2/token');

            $body = [
                'refresh_token' => $this->refresh_token,
                'client_id'     => '1000.R6NADONBVTIS0VZSZ0DE5MNO3DEV7T',
                'client_secret' => '52f6c37069fd0db03ed6025766e968214d35bffbe5',
                'grant_type'    => 'refresh_token',
            ];

            $response = wp_remote_post($url, [
                'body'    => $body,
                'headers' => ['Content-Type' => 'application/x-www-form-urlencoded'],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return false;
            }

            $data = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($data['access_token'])) {
                INTEGRAZO_WZ_Util::updateAccessToken($this->account_id, $data['access_token']);
                $this->access_token = $data['access_token'];
                return true;
            }

            return false;
        }

        // ---------------- BIGIN: MODULES ----------------

        /** Get all modules from Bigin */
        public function get_modules($retry = true)
        {
            // v2 modules endpoint
            $url = $this->build_api_url('bigin/v2/settings/modules'); // docs: /bigin/v2/settings/modules
            $response = wp_remote_get($url, [
                'headers' => ['Authorization' => 'Zoho-oauthtoken ' . $this->access_token],
                'timeout' => 60
            ]);

            if (is_wp_error($response)) {
                return ['error' => "Connection failed: Please verify your internet connection, or the Zoho server may not be responding right now. Try again, and if the issue continues, contact support"];
            }

            $status = wp_remote_retrieve_response_code($response);
            $body   = json_decode(wp_remote_retrieve_body($response), true);

            if ($status === 401 && isset($body['code']) && in_array($body['code'], ['AUTHENTICATION_FAILURE', 'INVALID_TOKEN'], true)) {
                if ($retry && $this->get_access_token()) {
                    return $this->get_modules(false);
                }
                return ['error' => 'Authentication failed. Your session may have expired or your token may be invalid. Please reauthenticate and try again.'];
            }

            if ($status !== 200) {
                return ['error' => "An unexpected error occurred while fetching modules. Please try again later or contact support if the issue persists."];
            }

            if (empty($body['modules'])) {
                return ['error' => "No modules found. This may be due to access restrictions or an empty configuration. Please check your account settings or contact support."];
            }

            return $body['modules'];
        }

        // ---------------- BIGIN: FIELDS ----------------

        /** Get fields of a specific Bigin module */
        public function get_module_fields($module_api_name, $retry = true)
        {
            $url = $this->build_api_url('bigin/v2/settings/fields?module=' . urlencode($module_api_name)); // docs: /bigin/v2/settings/fields

            $response = wp_remote_get($url, [
                'headers' => ['Authorization' => 'Zoho-oauthtoken ' . $this->access_token],
                'timeout' => 60
            ]);

            if (is_wp_error($response)) {
                return ['error' => "Connection failed: Please verify your internet connection, or the Zoho server may not be responding right now. Try again, and if the issue continues, contact support"];
            }

            $status = wp_remote_retrieve_response_code($response);
            $body   = json_decode(wp_remote_retrieve_body($response), true);

            if ($status === 401 && isset($body['code']) && in_array($body['code'], ['AUTHENTICATION_FAILURE', 'INVALID_TOKEN'], true)) {
                if ($retry && $this->get_access_token()) {
                    return $this->get_module_fields($module_api_name, false);
                }
                return ['error' => 'Authentication failed. Your session may have expired or your token may be invalid. Please reauthenticate and try again.'];
            }

            if ($status !== 200) {
                return ['error' => "An unexpected error occurred while fetching fields. Please try again later or contact support if the issue persists."];
            }

            if (empty($body)) {
                return ['error' => "No fields were found for the specified module. This module may not contain fields or there may be a configuration issue. Please check your module setup or contact support."];
            }

            return $body;
        }

        // ---------------- BIGIN: ADD RECORD ----------------

        /**
         * Adds a record to a specific module in Bigin.
         * Example module names: Contacts, Deals, Accounts, Tasks, Events, Calls, Products, Pipelines, etc.
         */
        public function add_module_record($module_api_name, $record_data, $retry = true)
        {
            try {
                $url = $this->build_api_url('bigin/v2/' . urlencode($module_api_name)); // v2 base

                $response = wp_remote_post($url, [
                    'headers' => [
                        'Authorization' => 'Zoho-oauthtoken ' . $this->access_token,
                        'Content-Type'  => 'application/json',
                    ],
                    'body'    => wp_json_encode(['data' => [$record_data]]),
                    'timeout' => 60
                ]);

                if (is_wp_error($response)) {
                    return INTEGRAZO_WZ_Product_Util::create_error_response('NETWORK_ISSUE', $response->get_error_message(), null);
                }

                $status = wp_remote_retrieve_response_code($response);
                $body   = json_decode(wp_remote_retrieve_body($response), true);

                if (!is_array($body)) {
                    return INTEGRAZO_WZ_Product_Util::create_error_response('INVALID_RESPONSE', 'Invalid response from API.', $body);
                }

                if ($status === 401 && isset($body['code']) && $this->is_authentication_error($body['code'])) {
                    if ($retry && $this->get_access_token()) {
                        return $this->add_module_record($module_api_name, $record_data, false);
                    }
                    return INTEGRAZO_WZ_Product_Util::create_error_response('AUTHENTICATION_FAILURE', 'Authentication failed.', $body);
                }

                return $body;
            } catch (Throwable $e) {
                return INTEGRAZO_WZ_Product_Util::create_error_response('EXCEPTION_THROWN', $e->getMessage(), null);
            }
        }

        /** Helper: is this an authentication error code? */
        public function is_authentication_error($code)
        {
            $auth_errors = ['AUTHENTICATION_FAILURE', 'INVALID_TOKEN'];
            return in_array($code, $auth_errors, true);
        }
    }
}
