<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Reusable Organizations API for Inventory/Books/Invoice.
 * Host class must extend INTEGRAZO_WZ_ZOHO_Base (provides ->request()).
 */
if (!trait_exists('INTEGRAZO_WZ_ZOHO_Organizations_Trait')) {

    trait INTEGRAZO_WZ_ZOHO_Organizations_Trait
    {
        /**
         * Get list of organizations.
         * Returns ['organizations' => [...]] on success or ['error' => '...'] on failure.
         */
        public function get_organizations_list(bool $retry = true): array
        {
            $res = $this->request('GET', 'organizations', [
                'json' => false,
            ], $retry);

            if (!empty($res['error'])) {
                return ['error' => 'Connection failed. Please check connectivity or try again later.', 'data' => $res['data'] ?? null];
            }

            if ($res['status'] < 200 || $res['status'] >= 300 || !is_array($res['body'])) {
                return ['error' => 'Unable to fetch organizations list.', 'data' => $res['body'] ?? null];
            }

            if (isset($res['body']['organizations']) && is_array($res['body']['organizations'])) {
                return ['organizations' => $res['body']['organizations']];
            }

            return ['error' => 'Unexpected response while fetching organizations.', 'data' => $res['body'] ?? null];
        }
        /**
         * Get adjustment accounts (Chart of Accounts) from Zoho Inventory.
         * Returns ['accounts' => [...]] or ['error' => '...']
         */
        public function get_adjustment_accounts(string $organization_id, bool $retry = true): array
        {
            $res = $this->request('GET', 'chartofaccounts', [
                'query' => [
                    'organization_id' => $organization_id,
                ],
                'json' => false,
            ], $retry);

            if (!empty($res['error'])) {
                return ['error' => 'Connection failed while loading accounts.', 'data' => $res['data'] ?? null];
            }

            if ($res['status'] < 200 || $res['status'] >= 300 || !is_array($res['body'])) {
                return ['error' => 'Unable to fetch chart of accounts.', 'data' => $res['body'] ?? null];
            }

            $accounts = $res['body']['chartofaccounts'] ?? null;

            if (!is_array($accounts)) {
                return ['error' => 'Invalid account response.', 'data' => $res['body'] ?? null];
            }

            // Filter only active accounts (optional but recommended)
            $accounts = array_filter($accounts, function ($acc) {
                return !empty($acc['is_active']);
            });

            // Convert to simple mapping array for UI dropdown
            $list = array_map(function ($acc) {
                return [
                    'id'    => $acc['account_id'],
                    'name'  => $acc['account_name'],
                    'type'  => $acc['account_type'] ?? '',
                ];
            }, $accounts);

            return ['accounts' => array_values($list)];
        }
    }
}
