<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
if (!class_exists('INTEGRAZO_WZ_CF7_Helper')) {
    class INTEGRAZO_WZ_CF7_Helper
    {
        /**
         * Validate if Contact Form 7 plugin is active.
         *
         * @return true|string
         */
        private static function validate_plugin()
        {
            if (!function_exists('wpcf7_contact_form')) {
                return 'Contact Form 7 plugin is not installed or activated. Please install and activate Contact Form 7 to continue.';
            }

            return true;
        }

        /**
         * Get all Contact Form 7 forms with ID and title.
         *
         * @return array|string Returns an array of forms or an error message.
         */
        public static function get_all_forms()
        {
            $validation = self::validate_plugin();
            if (is_string($validation)) {
                return $validation;
            }

            $args = array(
                'post_type'      => 'wpcf7_contact_form',
                'posts_per_page' => -1,
                'post_status'    => 'publish',
            );

            $forms = get_posts($args);

            if (empty($forms)) {
                return 'No Contact Form 7 forms found. Please create at least one form to continue.';
            }

            $data = [];

            foreach ($forms as $form) {
                $data[] = [
                    'id'    => (int) $form->ID,
                    'title' => $form->post_title,
                ];
            }

            return $data;
        }

        /**
         * Get all supported fields for a specific Contact Form 7 form.
         *
         * @param int $form_id
         * @return array|string Returns an array of fields or an error message.
         */
        public static function get_form_fields($form_id)
        {

            $contact_form = WPCF7_ContactForm::get_instance($form_id);
            if (!$contact_form) {
                return 'The mapped Contact Form 7 form was not found. It may have been deleted or is currently unavailable.';
            }

            $form_tags = $contact_form->scan_form_tags();

            if (empty($form_tags)) {
                return 'This Contact Form 7 form does not contain any fields. Please add at least one field to continue.';
            }

            $fields = [];

            foreach ($form_tags as $tag) {
                if ($tag->basetype !== 'submit') {
                    $fields[] = [
                        'name'       => $tag->name,
                        'type'       => $tag->basetype,
                        'label'      => $tag->name,
                        'identifier' => $tag->name,
                    ];
                }
            }

            return $fields;
        }
        /**
         * Hook to capture form submission data.
         */
        public static function cf7_capture_submission($contact_form)
        {
            try {
                $form_id = $contact_form->id();
                $submission = WPCF7_Submission::get_instance();
                if ($submission) {
                    $form_data = $submission->get_posted_data();
                    $uploaded_files = $submission->uploaded_files();  // Uploaded file paths
                    // Replace file keys in form_data with full file paths
                    foreach ($uploaded_files as $key => $file_path) {
                        if (isset($form_data[$key])) {
                            $form_data[$key] = $file_path;
                        }
                    }
                    INTEGRAZO_WZ_Form_Hooks::process_submission($form_id, $form_data, "contact_form_7");
                }
            } catch (Throwable $e) {
                return;
            }
        }
    }
}
// ✅ Safe hook, only if class exists
if (class_exists('INTEGRAZO_WZ_CF7_Helper')) {
    add_action('wpcf7_before_send_mail', ['INTEGRAZO_WZ_CF7_Helper', 'cf7_capture_submission']);
}
